/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Sat Feb 18 17:14:51 2006 by Jeff Dalton
 * Copyright: (c) 2005, AIAI, University of Edinburgh
 */

package ix.util.lisp;

import java.io.IOException;
import java.util.*;

import junit.framework.*;

import ix.util.*;
import ix.util.match.*;

public class LispInterpreterTest extends TestCase {

    final Symbol ERROR = Symbol.intern(":error");

    int groupCount = 0;
    int testCount = 0;

    public LispInterpreterTest(String name) {
	super(name);
    }

    static final Object testGroupSyntax =
	Lisp.readFromString("(define-test-group ?name &rest ?clauses)");

    static final Object testSyntax =
	Lisp.readFromString("(?test-form ==> ?expected-result-form)");

    public void testLispInterpreterTests() throws IOException {
	LispReader testReader = Lisp.openForInput
	                          ("ix/util/lisp/lisp-interpreter-tests.lsp");
	testReader.setSingleQuoteIsMacro(true);
	groupCount = 0;
	testCount = 0;
	while (true) {
	    Object in = testReader.readObject();
	    if (in == Lisp.EOF)
		break;
	    runTestGroup((LList)in);
	}
	Debug.noteln("");
	Debug.noteln("[" + testCount + " Lisp tests in " +
		     groupCount + " groups]");
    }

    void runTestGroup(LList testGroup) {
	// Each test group gets a new LispInterpreter, and all
	// the tests in the group use that interpreter.
	groupCount++;
	MatchEnv e = SimpleMatcher.mustMatch(testGroupSyntax, testGroup);
	Symbol name = (Symbol)valueOf("?name", e);
	LList clauses = (LList)valueOf("?clauses", e);
	LispInterpreter lin = makeLispInterpreter();
	for (Iterator i = clauses.iterator(); i.hasNext();) {
	    LList test = (LList)i.next();
	    runTest(lin, name, test);
	}
    }

    protected LispInterpreter makeLispInterpreter() {
	return new LispInterpreter();
    }

    void runTest(LispInterpreter lin, Symbol testGroupName, LList test) {
	testCount++;
	MatchEnv e = SimpleMatcher.mustMatch(testSyntax, test);
	Object testForm = valueOf("?test-form", e);
	Object expectedResultForm = valueOf("?expected-result-form", e);
	if (expectedResultForm == ERROR) {
	    try {
		lin.topLevelEval(testForm);
		fail("Expected an error from " + testForm);
	    }
	    catch (Exception ex) {
		return;
	    }
	}
	Object testResult = lin.topLevelEval(testForm);
	Object expectedResult = lin.topLevelEval(expectedResultForm);
	Debug.expectEquals(expectedResult, testResult,
			   "failure in test group " + testGroupName +
			   " test " + test);
    }

    static Object valueOf(String name, MatchEnv e) {
	Object value = e.get(Symbol.intern(name));
	if (value != null)
	    return value;
	else
	    throw new IllegalArgumentException("No value for " + name);
    }

}
