/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Mon Feb 23 19:48:22 2004 by Jeff Dalton
 * Copyright: (c) 2004, AIAI, University of Edinburgh
 */

package ix.util.context;

import junit.framework.*;

import ix.util.*;
import ix.util.lisp.*;

/**
 * Simple context-related test cases for LLQueues.
 */
public class LLQueueContextTest extends TestCase {

    public LLQueueContextTest(String name) {
	super(name);
    }

    protected void setUp() {
	Context.clearContexts();
    }

    protected void tearDown() {
	Context.clearContexts();
    }

    public void testInOneContext() {
	LLQueue q = new LLQueue();
	assertEquals(Lisp.NIL, q);
	q.add("a");
	q.add("b");
	assertEquals(Lisp.list("a", "b"), q);
	assertEquals(Lisp.list("a", "b"), q);
	q.add("c");
	assertEquals(Lisp.list("a", "b", "c"), q);
    }

    public void testInThreeContexts() {
	// Creates two children of the root context and gives a queue
	// different contents in each.  Then gives it some contents
	// in the root context as well, creates another child of the
	// root, and checks that the queue has the right contents in
	// all three children.
	LLQueue q = new LLQueue();
	assertSame(Context.rootContext, Context.getContext());
	// First context
	Context c1 = Context.pushContext();
	q.add("a1");
	q.add("b1");
	assertEquals(Lisp.list("a1", "b1"), q);
	// Back to root
	Context.popContext();
	assertEquals(Lisp.NIL, q);
	// Second context
	Context c2 = Context.pushContext();
	q.add("a2");
	assertEquals(Lisp.list("a2"), q);
	// Back to root
	Context.popContext();
	assertEquals(Lisp.NIL, q);
	q.add("root-a");
	assertEquals(Lisp.list("root-a"), q);
	// Check first context
	Context.setContext(c1);
	assertEquals(Lisp.list("a1", "b1"), q);
	// Check second context
	Context.setContext(c2);
	assertEquals(Lisp.list("a2"), q);
	// Back to root
	Context.popContext();
	assertSame(Context.rootContext, Context.getContext());
	// Create a third root-child and check the q again
	Context c3 = Context.pushContext();
	assertTrue(c1 != c2 && c1 != c3 && c2 != c3);
	assertEquals(Lisp.list("root-a"), q); // should inherit from root
    }

    public void testWhenCreatedInSibling() {
	// A context, c1
	Context c1 = Context.pushContext();
	LLQueue q = new LLQueue(); // create in c1
	q.add("a");
	assertEquals(Lisp.list("a"), q);
	// c2, a sibling of c1
	Context.popContext();
	Context c2 = Context.pushContext();
	assertSame(c1.getParent(), c2.getParent());
	// q should be empty here.
	assertEquals(Lisp.list(), q);
    }

    public void testInContextChain() {
	LLQueue q = new LLQueue();
	assertSame(Context.rootContext, Context.getContext());
	q.add("root");
	Context.pushContext();
	q.add("one");
	Context.pushContext();
	q.add("two-a");
	q.add("two-b");
	Context.pushContext();
	q.add("three");
	Context.pushContext();
	q.add("four");
	assertEquals(Lisp.list("root", "one", "two-a", "two-b",
			       "three", "four"),
		     q);
	Context.popContext();
	assertEquals(Lisp.list("root", "one", "two-a", "two-b", "three"), q);
	Context.popContext();
	assertEquals(Lisp.list("root", "one", "two-a", "two-b"), q);
	Context.popContext();
	assertEquals(Lisp.list("root", "one"), q);
	Context.popContext();
	assertEquals(Lisp.list("root"), q);
	assertSame(Context.rootContext, Context.getContext());
    }

}
