/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Thu Apr 15 02:35:25 2004 by Jeff Dalton
 * Copyright: (c) 1998, 2001, AIAI, University of Edinburgh
 */

package ix.util;

import java.util.*;

import junit.framework.*;

import ix.util.lisp.*;

/** ix.util.Collect test cases */

public class CollectTest extends TestCase {

    private static List empty_LList = Lisp.NIL;
    private static List abcd_LList = Lisp.list("a", "b", "c", "d");
    private static List abc_LList = Lisp.list("a", "b", "c");
    private static List ac_LList = Lisp.list("a", "c");
    private static List bd_LList = Lisp.list("b", "d");
    private static List b_LList = Lisp.list("b");

    public CollectTest(String name) {
	super(name);
    }

    // Some minimal LList map and filter tests.

    public void testEmptyLListMap() {
	Collection result = Collect.map(Lisp.NIL, Fn.identity);
	assertSame(Lisp.NIL, result);
    }

    public void testEmptyLListFilter() {
	Collection result = Collect.filter(Lisp.NIL, Fn.alwaysTrue);
	assertSame(Lisp.NIL, result);
    }

    // Some non-LList tests.

    public void testIntersection() {
	assertEquals(new LinkedList(b_LList),
		     Collect.intersection(new LinkedList(abc_LList),
					  new LinkedList(bd_LList)));
	assertEquals(new TreeSet(b_LList),
		     Collect.intersection(new TreeSet(abc_LList),
					  new TreeSet(bd_LList)));
    }

    public void testUnion() {
	assertEquals(new LinkedList(abcd_LList),
		     Collect.union(new LinkedList(abc_LList),
				   new LinkedList(bd_LList)));
	assertEquals(new TreeSet(abcd_LList),
		     Collect.union(new TreeSet(abc_LList),
				   new TreeSet(bd_LList)));
    }

    public void testDifference() {
	assertEquals(new LinkedList(ac_LList),
		     Collect.difference(new LinkedList(abc_LList),
					new LinkedList(bd_LList)));
	assertEquals(new TreeSet(ac_LList),
		     Collect.difference(new TreeSet(abc_LList),
					new TreeSet(bd_LList)));
    }

    public void testAppend() {
	assertEquals(new LinkedList(Strings.breakAt(" ", "a b c b d")),
		     Collect.append(new LinkedList(abc_LList),
				    new LinkedList(bd_LList)));
    }

    // LList tests similar to those in the previous block.

    public void testLListIntersection() {
	assertEquals(b_LList,
		     Collect.intersection(abc_LList, bd_LList));
	assertEquals(ac_LList,
		     Collect.intersection(abc_LList, ac_LList));
	assertEquals(ac_LList,
		     Collect.intersection(ac_LList, abc_LList));
	assertSame(empty_LList,
		   Collect.intersection(ac_LList, bd_LList));
	assertSame(empty_LList,
		   Collect.intersection(empty_LList, abc_LList));
	assertSame(empty_LList,
		   Collect.intersection(abc_LList, empty_LList));
    }

    public void testLListUnion() {
	assertEquals(abcd_LList,
		     Collect.union(abc_LList, bd_LList));
    }

    public void testLListDifference() {
	assertEquals(ac_LList,
		     Collect.difference(abc_LList, bd_LList));
    }

    public void testLListAppend() {
	assertEquals(Strings.breakAt(" ", "a b c b d"),
		     Collect.append(abc_LList, bd_LList));
    }

    // Map consistency

    public void testMapConsistency() {
	LList a1 = (LList)Lisp.readFromString("((a 1) (b 2) (c 3))");
	LList a2 = (LList)Lisp.readFromString("((a 1) (c 3))");
	LList a3 = (LList)Lisp.readFromString("((a 1) (d 4))");
	LList a4 = (LList)Lisp.readFromString("((a 1) (c 4))");
	Map m1 = a1.alistToMap(new HashMap());
	Map m2 = a2.alistToMap(new HashMap());
	Map m3 = a3.alistToMap(new HashMap());
	Map m4 = a4.alistToMap(new HashMap());
	assertTrue(Collect.areConsistent(m1, m2));
	assertTrue(Collect.areConsistent(m1, m3));
	assertFalse(Collect.areConsistent(m1, m4));
	assertTrue(Collect.areConsistent(m3, m4));
    }

    // insertBeforeFirst

    public void testInsertBeforeFirst() {
	List ade = Strings.breakAt(" ", "a d e");
	List bc = Strings.breakAt(" ", "b c");
	assertEquals(Strings.breakAt(" ", "a b c d e"),
		     Collect.insertBeforeFirst("d", ade, bc));
    }

}
