;;;; Co-OPR Personnel Recovery Domain Model
;;;; Jeff Dalton and Austin Tate, AIAI, University of Edinburgh
;;;; Updated: Mon Apr  4 20:41:25 2005 by Jeff Dalton

;;;; Modelling conventions

;;; Object types -- see the initial state description.

;;; Locations may be known or unknown:
;;;   (location-status ?object) = known | unknown

;;; That usually matters only for the IP.

;;; Some locations are named
;;;   (location ?object) = ?place-name

;;; The map viewer requires:
;;;   (latitude ?object) = ?double
;;;   (longitude ?object) = ?double
;;; Optionally, for 3-D Views, icon association and Map Tool handling:
;;;   (altitude ?object) = ?double
;;;   (type ?object) = ?type-name
;;;   (geo-fixed ?object) = true

;;; It may or may not be possible to communicate with the IP:
;;;   (communication-status ?ip) = two-way | none | ... ? ...

;;; Distances are qualitative:
;;;   (transport-distance ?from ?to) = short | long
;;;   short -- walkable
;;;   long  -- air or ground transport

;;; A medium distance might also be used, and perhaps long
;;; should be air-only.  However, it's a lot easier for us
;;; if there are only two possible values, because we don't
;;; have a good way to say "or".

(annotations
  ;; The list will be extended as needed.
  (achievable-world-state-conditions =
    (communication-status)))

;;;; Top level

;;; /\/: Locate "continues until isolated personnel are recovered".
;;; /\/: Support "continues through the debrief and return process".

;;; Instead of having "locate" continue, we have it finish before
;;; support and recover can begin.  However, additional "locate"
;;; activities (e.g. to verifyig the location) can be added-in if needed.

;;; /\/: Node-ids that aren't sequential numbers aren't handled
;;; correctly in I-DE as at 4-Nov-2004. A fix in the .lsp format
;;; reader compensates.

(refinement rescue-generic (rescue ?ip)
  (variables ?ip)
  (nodes
    (1 (report ?ip))
    (2 (locate ?ip))
    (3 (support ?ip))
    (4 (recover ?ip))
    (5 (return ?ip)))
  (orderings
    (1 2) (2 (3 4)) (4 5)))

(refinement rescue-1-team-sof-support-sof-recovery (rescue ?ip)
  (variables ?ip ?sof)
  (nodes
    (1 (report ?ip))
    (2 (locate ?ip))
    (a1 (allocate-one sof-oda ?sof (support-and-recover ?ip)))
    (m1 (move-from-base ?sof ?ip))
    (3 (support ?sof ?ip))
    (4 (recover ?sof ?ip))
    (5 (return ?ip)))
  (orderings
    (1 2)
    (2 (3 4))
    (4 5)
    (a1 m1 (3 4)))
  (annotations
    (note = "1 SOF ODA used, for both support and recovery")))

(refinement rescue-2-teams-sof-support-sof-recovery (rescue ?ip)
  (variables ?ip ?support-sof ?recover-sof)
  (nodes
    (rep (report ?ip))
    (loc (locate ?ip))
    (a1 (allocate-one sof-oda ?support-sof (support ?ip)))
    (move1 (move-from-base ?support-sof ?ip))
    (sup (support ?support-sof ?ip))
    (a2 (allocate-one sof-oda ?recover-sof (recover ?ip)))
    (move2 (move-from-base ?recover-sof ?ip))
    (rec (recover ?recover-sof ?ip))
    (ret (return ?ip)))
  (orderings
    (a1 a2)
    (a1 move1 sup)
    (a2 move2 rec)
    (rep loc) (loc (sup rec)) (rec ret))
  (annotations
    (note = "2 SOF ODAs used, one for support, one for recovery")))

(refinement rescue-sof-support-covert-recovery (rescue ?ip)
  (variables ?ip ?support-sof)
  (nodes
    (1 (report ?ip))
    (2 (locate ?ip))
    (a1 (allocate-one sof-oda ?support-sof (support ?ip)))
    (3 (support ?support-sof ?ip))
    (4 (recover-by-covert-means ?ip))
    (5 (return ?ip)))
  (orderings
    (a1 (3 4))
    (1 2) (2 (3 4)) (4 5)))

(refinement rescue-sof-support-civilian-recovery (rescue ?ip)
  (variables ?ip ?support-sof)
  (nodes
    (1 (report ?ip))
    (2 (locate ?ip))
    (a1 (allocate-one sof-oda ?support-sof (support ?ip)))
    (3 (support ?support-sof ?ip))
    (4 (recover-by-civilian-means ?ip))
    (5 (return ?ip)))
  (orderings
    (a1 (3 4))
    (1 2) (2 (3 4)) (4 5)))

(refinement rescue-unassisted-recovery (rescue ?ip)
  (variables ?ip)
  (issues
    (issue (can ?ip be recovered unassisted)))
  (nodes
    (1 (report ?ip))
    (2 (locate ?ip))
    (3 (support ?ip))
    (4 (recover-unassisted ?ip))
    (5 (return ?ip)))
  (orderings
    (1 2) (2 (3 4)) (4 5)))

;;;; Mission and Task Setup refinements

;;; Test-menu entries might be used instead.

;;;; Report phase

;;; Report is the notification that personnel have or may have
;;; become isolated.  /\/: Is that all, or does it include any
;;; reporting that happens during the operation?

;;; This is when the report of the isolated personnel (IP) comes in --
;;; either direct form them, or via some reporting method (civilians,
;;; command structure or reporting system).

;;; It's not clear that this is really part of the recovery operation
;;; rather than something that happens to start things happening --
;;; things that will, for instance, include planning.  Yet everything
;;; else (except perhaps "locate") is part of executing the plan that's
;;; produced.

;;; No refinements for now.  Assume activities are manually "done".

;;;; Locate phase

;;; Since we provide refinements for the "locate" pattern,
;;; the automatic planner will insist on using one of them;
;;; we therefore require one for the case where nothing needs
;;; to be done.

;;; /\/: There should be variations that use local people
;;; or the local government, even if they are possibly hostile.
;;; The locate-using-human-intelligence refinement is currently
;;; all we have in this area.

;;; To make it easier to reuse locate methods to verify location,
;;; the ones that are not too expensive can be used even when
;;; the location-status is known.  /\/: This could lead to
;;; irritating extra locate activities if location-status
;;; were made achievable.  /\/: It also means that replans
;;; may just change the location method even though that's
;;; uninteresting.  So it might be better to restrict the
;;; locate methods to location-status = unknown and complicate
;;; verification.

;;; N.B. The locate refinements do not record a location in
;;; the world state; that must be done independently.

(refinement locate-already-known (locate ?ip)
  (variables ?ip)
  (issues
    (issue (consider adding verify-location task afterwards)))
  (constraints
    (world-state condition (location-status ?ip) = known)))

(refinement locate-by-asking (locate ?ip)
  (variables ?ip)
  (issues
    (issue (can ?ip determine their location reliably)))
  (nodes
    (1 (ask ?ip location)))
  (constraints
    ;; (world-state condition (location-status ?ip) = unknown)
    (world-state condition (communication-status ?ip) = two-way)
    (world-state effect (location-status ?ip) = known)))

(refinement locate-air-search (locate ?ip)
  (variables ?ip)
  (issues
    (issue (can ?ip be seen from the air)))
  (nodes
    (1 (search air ?ip)))
  (constraints
    (world-state condition (location-status ?ip) = unknown)
    (world-state effect (location-status ?ip) = known)))

(refinement locate-ground-search (locate ?ip)
  (variables ?ip)
  (issues
    (issue (is there enough time for a ground search for ?ip)))
  (nodes
    (1 (search ground ?ip)))
  (constraints
    (world-state condition (location-status ?ip) = unknown)
    (world-state effect (location-status ?ip) = known)))

(refinement locate-air-and-ground-search (locate ?ip)
  (variables ?ip)
  (nodes
    (1 (search air-and-ground ?ip)))
  (constraints
    (world-state condition (location-status ?ip) = unknown)
    (world-state effect (location-status ?ip) = known)))

(refinement locate-using-satellite-intelligence (locate ?ip)
  (variables ?ip)
  (issues
    (issue (can ?ip be seen from space)))
  (constraints
    ;; (world-state condition (location-status ?ip) = unknown)
    (world-state effect (location-status ?ip) = known)))

(refinement locate-using-human-intelligence (locate ?ip)
  (variables ?ip)
  (issues
    (issue (are the human intelligence sources reliable)))
  (constraints
    ;; (world-state condition (location-status ?ip) = unknown)
    (world-state effect (location-status ?ip) = known)))

;;; Search methods:
;;;   Boundary method.
;;;   Grid method.
;;;   Track line method.
;;;   Feature trace method.
;;;   Automatic direction-finding.

;;;; Location add-ins

;;; These refinements are for activities that are included as needed
;;; by the user, though they may also be explicitly included as
;;; sub-activities.  The automatic planner is not allowed to use
;;; them to achieve conditions.

;;; verify-location-by-relocating lets us avoid repeating all
;;; the location methods.

(refinement verify-locate-by-re-locating (verify-location ?ip)
  (variables ?ip)
  (nodes
    (1 (locate ?ip))))

;;;; Support phase

;;; Support is a continuous activity; and although specific support
;;; actions may occur, they aren't determined in advance; and after
;;; each one, the support process should continue.  Rather
;;; than model this using recursion, which would cause problems
;;; for the automatic planner (what actions would it include
;;; and when would it stop?), we have [amost] no refinements that
;;; directly expand the "support" pattern and instead rely on the
;;; user to insert suitable activities as needed.

(refinement provide-other-support (support ?ip)
  (variables ?ip)
  (issues
    (issue (consider adding specific support afterwards: provide-things, protect-things))
  ))

(refinement provide-sof-support (support ?sof ?ip)
  (variables ?sof ?ip)
  (issues
    (issue (consider adding other support afterwards: provide-things, protect-things))
  ))

;;; These are just simple examples of support add-ins, without any detail
;;; filled-in.

(refinement provide-communication (provide-communication-with ?ip)
  (variables ?ip)
  (issues
    (issue (is communication with ?ip secure))
    (issue (can communication with ?ip be disrupted)))
  (constraints
    (world-state effect (communication-status ?ip) = two-way)))

(refinement provide-technical-advice (provide-technical-advice ?ip)
  (variables ?ip)
  (constraints
    (world-state condition (communication-status ?ip) = two-way)))

(refinement provide-food (provide-food ?ip)
  (variables ?ip))

(refinement provide-medical-advice (provide-medical-advice ?ip)
  (variables ?ip)
  (issues
    (issue (should a medical team be sent to ?ip)))
  (constraints
    (world-state condition (communication-status ?ip) = two-way)))

(refinement provide-medical-supplies (provide-medical-supplies ?ip)
  (variables ?ip)
  (issues
    (issue (should a medical team accompany the medical supplies to ?ip))))

(refinement provide-medical-team (provide-medical-team ?ip)
  (variables ?ip ?ip-loc ?med-team ?method)
  (issues
    (issue (are there injured IP needing evacuation)))
  (nodes
    (1 (allocate-one med-team ?med-team (treating ?ip)))
    (2 (move ?method ?med-team ?ip-loc)))
  (orderings
    (1 2))
  (constraints
    (world-state condition (location ?ip) = ?ip-loc)))

(refinement support-by-sof (support ?ip)
  ;; This is a bridge from generic (support ?ip) to
  ;; support by an SOF team.
  (variables ?support-sof ?ip)
  (nodes
    (1 (allocate-one sof-oda ?support-sof (support ?ip)))
    (2 (move-from-base ?support-sof ?ip))
    (3 (support ?support-sof ?ip)))
  (orderings
    (1 2 3)))

(refinement delay-enemy (delay-enemy ?unit ?enemy)
  (variables ?unit ?enemy))

(refinement protect-location (protect-location ?unit ?loc)
  (variables ?unit ?loc))

;;;; Recover phase

;;; Recover can be:
;;;   (recover ?ip)                    -- generic
;;;   (recover ?sof-unit ?ip)          -- by SOF team
;;;   (recover-by-covert-means ?ip)
;;;   (recover-by-civilian-means ?ip)
;;;   (recover-unassisted ?ip)

;;; First some simple refinements for non-SOF cases.

(refinement recover-by-negotiation-generic (recover ?ip)
  (variables ?ip)
  (nodes
    (1 (negotiate-recovery ?ip))))

(refinement recover-by-negotiation (recover-unassisted ?ip)
  (variables ?ip)
  (nodes
    (1 (negotiate-recovery ?ip))))

(refinement recover-by-grey-force (recover ?ip)
  (variables ?ip)
  (nodes
    (1 (obtain-cooperation-of-grey-forces))
    (2 (recover-using-grey-force ?ip)))
  (orderings
    (1 2)))

;;; Bridges from generic recovery

(refinement recover-by-sof (recover ?ip)
  ;; This is a bridge from generic (recover ?ip) to
  ;; recovery by an SOF team.  Because the SOF allocation
  ;; is normally done at the top level, before the
  ;; (recover ?recover-sof ?ip) action, but isn't done
  ;; there in a generic recovery, we have to do it here
  ;; before we introduce the (recover ?recover-sof ?ip) action.
  ;; Non-SOF bridge refinements are simpler.
  (variables ?recover-sof ?ip)
  (nodes
    (1 (allocate-one sof-oda ?recover-sof (recover ?ip)))
    (2 (move-from-base ?recover-sof ?ip))
    (3 (recover ?recover-sof ?ip)))
  (orderings
    (1 2 3)))

(refinement recover-by-covert-means (recover ?ip)
  (variables ?ip)
  (nodes
    (1 (recover-by-covert-means ?ip))))

(refinement recover-by-civilian-means (recover ?ip)
  (variables ?ip)
  (nodes
    (1 (recover-by-civilian-means ?ip))))

(refinement recover-unassisted (recover ?ip)
  (variables ?ip)
  (nodes
    (1 (recover-unassisted ?ip))))

;;; SOF recovery

;;; This usually takes place in two phases: some setup and preparation,
;;; then an "extract" activity in which the IP are met and moved.

;;; /\/: Perhaps ?safe should be found in (recover ?unit ?ip)
;;; activities rather than in the (extract ...) ones.

(refinement recover-by-sof-heli (recover ?unit ?ip)
  (variables ?unit ?ip)
  (nodes
    (1 (extract-by-helicopter ?unit ?ip))))

(refinement extract-by-heli (extract-by-helicopter ?unit ?ip)
  (variables ?unit ?ip ?ip-loc ?lz ?safe)
  (nodes
    (1 (protect ?unit ?ip))
    (2 (make-contact ?unit ?ip))
    (3 (authenticate ?unit ?ip))
    (4 (walk ?unit ?ip ?lz))
    (5 (fly-by-heli ?unit ?ip ?safe)))
  (orderings
    (2 3 4 5))
  (constraints
    (world-state condition (location ?ip) = ?ip-loc)
    (world-state condition (heli-landing-zone ?ip-loc) = ?lz)
    (world-state condition (is-safe-destination ?safe) = true)))

;;; "Civilian" recovery.

(refinement recover-by-civilian-trucks (recover-by-civilian-means ?ip)
  (variables ?team ?ip ?ip-loc)
  (nodes
    (1 (allocate-one civilian-driver-team ?team (recover ?ip)))
    (2 (move by-road ?team ?ip-loc))
    (3 (extract-by-road ?team ?ip)))
  (orderings
    (1 2 3))
  (constraints
    (world-state condition (location ?ip) = ?ip-loc)))

(refinement extract-by-road (extract-by-road ?team ?ip)
  (variables ?team ?ip ?safe)
  (issues
    (issue (is ?safe reachable by road)))
  (nodes
    (1 (protect ?team ?ip))
    (2 (make-contact ?team ?ip))
    (3 (authenticate ?team ?ip))
    (4 (drive ?team ?ip ?safe)))
  (orderings
    (2 3 4))
  (constraints
    ;; /\/: Need to make sure ?safe isn't at sea.
    (world-state condition (is-safe-destination ?safe) = true)))

;;; SOF recovery using civilian transport

(refinement recover-by-civilian-heli (recover ?unit ?ip)
  (variables ?unit ?ip ?ip-loc ?lz)
  (nodes
    (1 (obtain-civilian-helicopter ?unit))
    (2 (move by-heli ?unit ?lz))
    (3 (extract-by-helicopter ?unit ?ip)))
  (orderings
    (1 2 3))
  (constraints
    (world-state condition (location ?ip) = ?ip-loc)
    (world-state condition (heli-landing-zone ?ip-loc) = ?lz)))

(refinement recover-by-civilian-fixed-wing (recover ?unit ?ip)
  (variables ?unit ?ip ?ip-loc ?airport ?lz)
  (nodes
    (1 (obtain-civilian-fixed-wing ?unit))
    (2 (move by-fixed-wing ?unit ?airport))
    (3 (move by-foot ?unit ?ip-loc))
    (4 (extract-by-fixed-wing ?unit ?ip ?airport)))
  (orderings
    (1 2 3 4))
  (constraints
    (world-state condition (location ?ip) = ?ip-loc)
    (world-state condition (type ?airport) = airport)
    (world-state condition (transport-distance ?ip-loc ?airport) = short)
    (world-state condition (heli-landing-zone ?ip-loc) = ?lz)))

(refinement extract-by-fixed-wing
	    (extract-by-fixed-wing ?unit ?ip ?airport)
  (variables ?unit ?ip ?airport ?safe)
  (nodes
    (1 (protect ?unit ?ip))
    (2 (make-contact ?unit ?ip))
    (3 (authenticate ?unit ?ip))
    (4 (walk ?unit ?ip ?airport))
    (5 (fly-by-fixed-wing ?unit ?ip ?safe)))
  (orderings
    (2 3 4 5))
  (constraints
    (world-state condition (is-safe-destination ?safe) = true)))

;;; "Covert" recovery.

(refinement recover-covertly-with-civ-trucks (recover-by-covert-means ?ip)
  (variables ?team ?ip ?ip-loc)
  (nodes
    (1 (allocate-one para-military-team ?team (recover ?ip)))
    (2 (move by-road ?team ?ip-loc))
    (3 (extract-by-road ?team ?ip)))
  (orderings
    (1 2 3))
  (constraints
    (world-state condition (location ?ip) = ?ip-loc)))

;;; Diversion ad-ins

;;; Manually expand at least down to selecting the "recover" method
;;; and performing any allocation activities.

;;; *** Diversions should be inserted *after*, not before,
;;; because you don't want them to have to complete execution
;;; before whatever's after them can execute.

(refinement diversion-recover-generic (recover-as-diversion ?ip)
  (variables ?ip)
  (nodes
    (1 (recover ?ip))))

;;;; Return phase

;;; No refinements for now.

;;;; General add-ins

;;; Authentication of isolated personnel.
;;;   Procedures may be electronic or visual.

;;; Authentication procedures / techniques:

;;;   Personal identification [by someone who knows them]
;;;   Isolation preparation packages (ISOPREPs).
;;;     [Contain personal information that can turn into questions
;;;      to ask]
;;;   Code words.
;;;   Visual signals.

;;;; Transportation layer

;;; There are 3 types of transportation activity.

;;; 1. Moving a unit from its base to the area around the IP.
;;;    Refinement names are from-base-by-<mechanism>.
;;;    The pattern is always (move-from-base ?unit ?ip).

;;; 2. A unit taking an object such as the IP to a location.
;;;    Refinement names are take-by-<mechanism>.
;;;    The pattern varies: walk, drive, fly-by-heli, fly-by-fixed-wing.

;;; 3. Moving a single object, typically a unit, to a location.
;;;    Refinement names are move-by-<mechanism>.
;;;    The pattern is (move by-<mechanism> ?obj ?to-location).
;;;    ** It's important to think of the operation being "move"
;;;       with 3 parameters, the first of which gives the mechanism,
;;;       and *not* as "move by-<mechanism" with 2 parameters
;;;       and a space in an odd place.
;;;    This type of activity can be used "generically" by putting
;;;    a variable as the first parameter.  ** Be very careful if
;;;    you do this twice at the same level, or you'll end up with actvities
;;;    having to use the same mehcanism when you don't want them to.

;;; /\/: We treat transport-from-base as a special case because
;;; otherwise we'd have to handle details such as ensuring that
;;; the plan didn't say to transport by road from a ship.
;;; It also lets us specify the move at a higher level,
;;; without having to work out where the ?ip are located.

(refinement from-base-by-heli (move-from-base ?unit ?ip)
  (variables ?unit ?ip ?ip-loc ?lz)
  (nodes
    (1 (move by-heli ?unit ?lz)))
  (constraints
    (world-state condition (location ?ip) = ?ip-loc)
    (world-state condition (heli-landing-zone ?ip-loc) = ?lz)))

(refinement from-base-by-halo-drop (move-from-base ?unit ?ip)
  (variables ?unit ?ip ?ip-loc)
  (issues
    (issue (are all ?unit trained for halo missions)))
  (nodes
    (1 (move by-fixed-wing ?unit ?ip-loc))
    (2 (halo-drop ?unit)))
  (orderings
    (1 2))
  (constraints
    (world-state condition (halo-qualified ?unit) = true)
    (world-state condition (location ?ip) = ?ip-loc)))

;;; A Unit taking an object to a location

;;; Without these activities, we'd either have to use
;;; pairs of "move" activities or else have either
;;; the unit or the object be implicit.  We want the units
;;; to be explicit so that we can more easily extract what
;;; each unit does.

;;; /\/: We could use patterns such as (take by-heli ...) instead,
;;; similar to what we do for "move" activities.

(refinement take-by-foot (walk ?unit ?obj ?to)
  (variables ?unit ?obj ?to ?lat ?lon)
  (constraints
    ;; /\/: "Walk" activities are used only when we know
    ;; we want to walk, so let them work regardless of distance.
    ;; /\/: Don't have distance between Brittan-Ranch and LZ-N-Brittan-Ranch 
    ;; (world-state condition (location ?obj) = ?from)
    ;; (world-state condition (transport-distance ?from ?to) = short)
    (world-state condition (latitude ?to) = ?lat)
    (world-state condition (longitude ?to) = ?lon)
    (world-state effect (location ?unit) = ?to)
    (world-state effect (latitude ?unit) = ?lat)
    (world-state effect (longitude ?unit) = ?lon)
    (world-state effect (location ?obj) = ?to)
    (world-state effect (latitude ?obj) = ?lat)
    (world-state effect (longitude ?obj) = ?lon)))

(refinement take-by-road (drive ?unit ?obj ?to)
  (variables ?unit ?obj ?to ?lat ?lon)
  (constraints
    (world-state condition (latitude ?to) = ?lat)
    (world-state condition (longitude ?to) = ?lon)
    (world-state effect (location ?unit) = ?to)
    (world-state effect (latitude ?unit) = ?lat)
    (world-state effect (longitude ?unit) = ?lon)
    (world-state effect (location ?obj) = ?to)
    (world-state effect (latitude ?obj) = ?lat)
    (world-state effect (longitude ?obj) = ?lon)))

(refinement take-by-heli (fly-by-heli ?unit ?obj ?to)
  (variables ?unit ?obj ?to ?lat ?lon)
  (constraints
    (world-state condition (latitude ?to) = ?lat)
    (world-state condition (longitude ?to) = ?lon)
    (world-state effect (location ?unit) = ?to)
    (world-state effect (latitude ?unit) = ?lat)
    (world-state effect (longitude ?unit) = ?lon)
    (world-state effect (location ?obj) = ?to)
    (world-state effect (latitude ?obj) = ?lat)
    (world-state effect (longitude ?obj) = ?lon)))

(refinement take-by-fixed-wing (fly-by-fixed-wing ?unit ?obj ?to)
  (variables ?unit ?obj ?to ?lat ?lon)
  (constraints
    (world-state condition (latitude ?to) = ?lat)
    (world-state condition (longitude ?to) = ?lon)
    (world-state effect (location ?unit) = ?to)
    (world-state effect (latitude ?unit) = ?lat)
    (world-state effect (longitude ?unit) = ?lon)
    (world-state effect (location ?obj) = ?to)
    (world-state effect (latitude ?obj) = ?lat)
    (world-state effect (longitude ?obj) = ?lon)))

;;; Generic transportation

;;; "Move" activities can move an object without specifying
;;; the means, or the means can be specified.

;;; Each "move-" refinement picks a particular mechanism.

;;; To move using a specific mechanism, use
;;;   (move by-foot ...)
;;;   (move by-heli ...)
;;;   (move by-fixed-wing ...)
;;;   (move by-halo ...)
;;;   (move by-road ...)

;;; To move something without saying how, use an activity
;;; such as
;;;   (move ?how object destination).

;;; *** Be careful when using such a variable at the top level.
;;; If you use the same variable name each time, the same mechanism
;;; will have to be used.

;;; /\/: For now, we assume helicopters and halo drops can go directly
;;; to any location.  This might be changed so that they have to
;;; go to a landing zone near the desired location.

;;; /\/: We allow fixed-wing transport directly to any location
;;; on the grounds that the object might be dropped, rather
;;; than the aircraft having to land.

;;; /\/: In most cases, we don't look at the start location.
;;; Higher-level activities should ensure that objects are in
;;; a suitable place before the "move" activity takes place.

;;; /\/: We do not model details such as which individual vehicles
;;; are used.

(refinement move-by-foot (move by-foot ?obj ?to)
  (variables ?obj ?from ?to ?lat ?lon)
  (issues
    (issue (is ?obj suitable for walk or carry)))
  (constraints
    (world-state condition (location ?obj) = ?from)
    (world-state condition (transport-distance ?from ?to) = short)
    (world-state condition (latitude ?to) = ?lat)
    (world-state condition (longitude ?to) = ?lon)
    (world-state effect (location ?obj) = ?to)
    (world-state effect (latitude ?obj) = ?lat)
    (world-state effect (longitude ?obj) = ?lon)))

(refinement move-by-heli (move by-heli ?obj ?to)
  (variables ?obj ?to ?lat ?lon)
  (constraints
    (world-state condition (latitude ?to) = ?lat)
    (world-state condition (longitude ?to) = ?lon)
    (world-state effect (location ?obj) = ?to)
    (world-state effect (latitude ?obj) = ?lat)
    (world-state effect (longitude ?obj) = ?lon)))

(refinement move-by-fixed-wing (move by-fixed-wing ?obj ?to)
  (variables ?obj ?to ?lat ?lon)
  (constraints
    (world-state condition (latitude ?to) = ?lat)
    (world-state condition (longitude ?to) = ?lon)
    (world-state effect (location ?obj) = ?to)
    (world-state effect (latitude ?obj) = ?lat)
    (world-state effect (longitude ?obj) = ?lon)))

(refinement move-by-halo (move by-halo ?obj ?to)
  (variables ?obj ?to)
  (issues
    (issue (are all ?obj trained for halo missions)))
  (nodes
    (1 (move by-fixed-wing ?obj ?to))
    (2 (halo-drop ?obj)))
  (orderings
    (1 2))
  (constraints
    (world-state condition (halo-qualified ?obj) = true)))

(refinement move-by-road (move by-road ?obj ?to)
  (variables ?obj ?to ?lat ?lon)
  (constraints
    (world-state condition (latitude ?to) = ?lat)
    (world-state condition (longitude ?to) = ?lon)
    (world-state effect (location ?obj) = ?to)
    (world-state effect (latitude ?obj) = ?lat)
    (world-state effect (longitude ?obj) = ?lon)))

;;;; Utilities

;;; N.B. Make sure all allocation activities happen in some
;;; definite order.

;;; /\/: This is a problem when using the rescue-generic refinement,
;;; because (as always) "support" is not required to complete before
;;; "recover" begins, and the "bridge" refinements that allow
;;; SOF support and/or recovery to be brought into the generic
;;; case contain allocate-one activities.

;;; /\/: The same applies to rescue-unassisted-recovery,
;;; though in that case it might be argued that the problem
;;; is that it doesn't forbid SOF recovery for the "recover"
;;; activity.  To fix that, the activity was changed to
;;; "recover-unassisted".

;;; /\/: *** Be especially careful when inserting a diversion.

(refinement allocate-one (allocate-one ?type ?object ?new-status)
  ;; /\/ (allocate-one ?type ?object ?new-status)?
  (variables ?type ?object ?new-status)
  (constraints
    (world-state condition (type ?object) = ?type)
    (world-state condition (status ?object) = available)
    (world-state effect (status ?object) = ?new-status))
  (annotations
    (use-for-world-state-effects = :none)))

;;; End
