/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Tue Sep 28 17:41:50 2004 by Jeff Dalton
 * Copyright: (c) 2003, 2004, AIAI, University of Edinburgh
 */

package ix.util.xml;

import javax.swing.*;
import javax.swing.event.*;
import javax.swing.table.*;

import java.awt.Color;
import java.awt.Container;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.BorderLayout;
import java.awt.FlowLayout;
import java.awt.event.*;

import java.util.*;

import org.jdom.Namespace;

import ix.iface.util.*;

import ix.util.*;
import ix.util.lisp.*;
import ix.util.xml.event.*;

/**
 * Handles XML namespace definitions for an XML tree editor.
 *
 * @see XMLTreeEditFrame
 * @see XMLTreeEditPanel
 */

public class XMLTreeNamespaceManager {

    protected NamespaceFrame frame;
    protected JTable table;
    protected NamespaceTableModel model;
    protected NamespaceMap namespaces;
    protected List namespaceListeners = new LinkedList();

    public XMLTreeNamespaceManager(String frameTitle) {
	namespaces = new NamespaceMap();
	addInitialNamespaces(namespaces);
	model = new NamespaceTableModel();
	table = new NamespaceJTable(model);
	frame = new NamespaceFrame(frameTitle);
	frame.setupFrame();
    }

    protected void addInitialNamespaces(NamespaceMap map) {
	map.add(Namespace.NO_NAMESPACE, Color.black);
	map.add(XML.getHomeNamespace(), Color.blue);
    }

    public void setVisible(boolean v) {
	frame.setVisible(v);
    }

    public List getNamespaces() {
	return namespaces.getNamespaces();
    }

    public Namespace parseNamespaceSpec(String spec) {
	String[] parts = Strings.breakAtFirst("=", spec);
	String prefix = parts[0].trim();
	String uri = parts[1].trim();
	if (uri.equals("") && !prefix.equals("")) {
	    // No "=", so uri should have the non-empty value
	    uri = prefix;
	    prefix = "";
	}
	return Namespace.getNamespace(prefix, uri);
    }

    public Color getNamespaceColor(Namespace n) {
	Color c = namespaces.getColor(n);
	if (c == null)
	    throw new IllegalArgumentException("Unknown namespace " + n);
	return c;
    }

    public void noteNamespace(Namespace n) {
	Color c = namespaces.getColor(n);
	if (c == null) {
	    // New Namespace
	    namespaces.add(n, pickNewNamespaceColor(n));
	    int i = namespaces.indexOf(n);
	    model.fireTableRowsInserted(i, i);
	}
    }

    protected Color pickNewNamespaceColor(Namespace n) {
	if (niceColorsUsed >= niceColors.length) {
	    Debug.noteln("Out of nice colors");
	    return Color.red;
	}
	return new Color(niceColors[niceColorsUsed++]);
    }

    protected int niceColorsUsed = 0;
    protected int[] niceColors = {
	// The repeated order is roughly blue, green, red, yellow/brown.
	// There's an implicit blue for the home namespace at the start.
//  	0x009933,		// near forrest green
	0x228B22,		// X's ForestGreen
	0x8B0000,		// X's DarkRed
	0xB8860B,		// X's DarkGoldenRod
	0x8A2BE2,		// X's BlueViolet
	0x008B8B,		// X's DarkCyan
  	0xff6666,		// a pink
	0xA0522D,		// X's Sienna
	0x00BFFF,		// X's DeepSkyBlue
//	0x008000,		// X's Green -- too close to ForestGreen
	0x00FF7F,		// X's SpringGreen
	0xC71585,		// X's MediumVioletRed
	0x808000,		// X's Olive
//	0x4682B4,		// X's SteelBlue -- too close to DarkCyan
	0x000080,		// X's Navy
	0x00CED1,		// X's DarkTurquoise
//	0x800000,		// X's Maroon -- too close to DarkRed
	0xFF4500,		// X's OrangeRed
  	0xffcc33,		// a yellow
	0x9ACD32		// X's YellowGreen
    };

    public void addNamespaceListener(NamespaceListener listener) {
	namespaceListeners.add(listener);
    }

    public void fireNamespaceEvent(NamespaceEvent e) {
	for (Iterator i = namespaceListeners.iterator(); i.hasNext();) {
	    ((NamespaceListener)i.next()).namespaceEvent(e);
	}
    }

    protected String[] helpText = {
	"Right-press on a row to change the color."
    };

    protected class NamespaceMap {
	Comparator namespaceComparator = new NamespaceComparator();
	Map namespaceToColorMap = new TreeMap(namespaceComparator);
	List namespaceList = new ArrayList(10);

	NamespaceMap() { }

	void add(Namespace n, Color c) {
	    // Add to Map
	    namespaceToColorMap.put(n, c);
	    // Add to List (awkward /\/)
	    namespaceList.add(n);
	    Collections.sort(namespaceList, namespaceComparator);
	}

	List getNamespaces() {
	    return Collections.unmodifiableList(namespaceList);
	}

	int size() {
	    return namespaceList.size();
	}

	Color getColor(Namespace n) {
	    return (Color)namespaceToColorMap.get(n);
	}

	void setColor(Namespace n, Color c) {
	    if (getColor(n) == null)
		throw new IllegalArgumentException
		    ("Can't set color for unknown namespace " + n);
	    Debug.expect(namespaceList.contains(n));
	    namespaceToColorMap.put(n, c);
	}

	int indexOf(Namespace n) {
	    return namespaceList.indexOf(n); // -1 if not present
	}

	Namespace getEntryNamespace(int i) {
	    return (Namespace)namespaceList.get(i);
	}

	Color getEntryColor(int i) {
	    return (Color)namespaceToColorMap.get(namespaceList.get(i));
	}

    }

    protected class NamespaceComparator implements Comparator {
	public int compare(Object a, Object b) {
	    Namespace na = (Namespace)a;
	    Namespace nb = (Namespace)b;
	    int c = na.getURI().compareTo(nb.getURI());
	    if (c == 0)
	    return na.getPrefix().compareTo(nb.getPrefix());
	    else
	    return c;
	}
    }

    protected class NamespaceFrame extends ToolFrame
	      implements ActionListener {

	TextAreaFrame helpFrame;
	Container contentPane;

	NamespaceFrame(String title) {
	    super(title);
	}

	void setupFrame() {
	    setIconImage(IconImage.getIconImage(this));
	    setJMenuBar(makeMenuBar());

	    table.getTableHeader().setReorderingAllowed(false);
	    table.setDefaultRenderer(Namespace.class,
				     new NamespaceCellRenderer());
	    table.setPreferredScrollableViewportSize
		(table.getPreferredSize());
	    table
		.addMouseListener(new TableMouseListener());

	    contentPane = getContentPane();

	    JScrollPane tableScroll = new JScrollPane(table);
	    tableScroll.setBorder
		(BorderFactory.createTitledBorder("Namespaces"));
	    contentPane.add(tableScroll, BorderLayout.CENTER);

	    contentPane.add(makeButtonPanel(), BorderLayout.SOUTH);

	    pack();
	    Dimension size = getSize();
	    setSize(new Dimension((int)Math.round(size.getWidth()),
				  200));
	}

	JMenuBar makeMenuBar() {
	    JMenuBar bar = new JMenuBar();

	    JMenu fileMenu = new JMenu("File");
	    fileMenu.add(makeMenuItem("Close"));

	    JMenu helpMenu = new JMenu("Help");
	    helpMenu.add(makeMenuItem("Help"));

	    bar.add(fileMenu);
	    bar.add(helpMenu);
	    return bar;
	}

	JMenuItem makeMenuItem(String text) {
	    JMenuItem item = new JMenuItem(text);
	    item.addActionListener(CatchingActionListener.listener(this));
	    return item;
	}

	JPanel makeButtonPanel() {
	    JPanel panel = new JPanel(); 	// defaults to FlowLayout
	    panel.setLayout(new BoxLayout(panel, BoxLayout.X_AXIS));
	    panel.add(Box.createHorizontalGlue());
	    panel.add(makeButton("Add Namespace"));
	    panel.add(Box.createHorizontalGlue());
	    return panel;
	}

	JButton makeButton(String command) {
	    JButton b = new JButton(command);
	    b.addActionListener(CatchingActionListener.listener(this));
	    return b;
	}

	/** Action interpreter */
	public void actionPerformed(ActionEvent e) {
	    String command = e.getActionCommand();
	    Debug.noteln("Namespace frame action:", command);
	    if (command.equals("Close")) {
		frame.setVisible(false);
	    }
	    else if (command.equals("Help")) {
		showHelp();
	    }
	    else if (command.equals("Add Namespace")) {
		doAddNamespace();
	    }
	    else
		throw new ConsistencyException
		    ("Nothing to do for " + command);
	}

	void showHelp() {
	    if (helpFrame == null) {
		helpFrame = new XMLTreeEditFrame.TreeHelpFrame
		    ("Namespace Manager Help");
		helpFrame.setText(makeHelpText());
	    }
	    helpFrame.setVisible(true);
	}

	String makeHelpText() {
	    return Strings.joinLines(helpText);
	}

	void doAddNamespace() {
	    String spec = JOptionPane.
		showInputDialog(table, "Enter namespace as prefix=URI");
	    if (spec == null)
		return;		// user cancelled
	    Namespace n = parseNamespaceSpec(spec);
	    if (n == Namespace.NO_NAMESPACE)
		return;		// spec was just whitespace
	    Debug.noteln("Adding Namespace", n);
	    noteNamespace(n);
	}

    }

    class NamespaceCellRenderer extends DefaultTableCellRenderer {

	NamespaceCellRenderer() { }

	public Component getTableCellRendererComponent
	                     (JTable table, Object value,
			      boolean isSelected, boolean hasFocus,
			      int row, int col) {
	    super.getTableCellRendererComponent
		(table, value, isSelected, hasFocus, row, col);
	    setBackground(Color.white);
	    setForeground(namespaces.getEntryColor(row));
	    return this;
	}

    }

    class NamespaceJTable extends JTable {

	NamespaceJTable(TableModel model) {
	    super(model);
	    setPreferredColumnWidths();
	}

	void setPreferredColumnWidths() {
	    for (int i = 0; i < model.getColumnCount(); i++) {
		TableColumn col = getColumnModel().getColumn(i);
		// /\/: Widths are times assumed number of pixels per char
		if (i == model.PREFIX_COL)
		    col.setPreferredWidth(5 * 9);
		else if (i == model.URI_COL)
		    col.setPreferredWidth(40 * 9);
		else
		    throw new ConsistencyException("Bogus column " + i);
	    }

	}

    }

    class NamespaceTableModel extends AbstractTableModel {

	final String[] columnName = {"Prefix", "URI"};
	final int PREFIX_COL = 0;
	final int URI_COL = 1;

	NamespaceTableModel() {
	    super();
	}

	public String getColumnName(int col) {
	    return columnName[col];
	}

	public int getColumnCount() {
	    return columnName.length;
	}

	public int getRowCount() {
	    return namespaces.size();
	}

        public Class getColumnClass(int c) {
	    return Namespace.class;
        }

	public boolean isCellEditable(int row, int col) {
	    return false;
	}

	public Object getValueAt(int row, int col) {
	    Namespace namespace = namespaces.getEntryNamespace(row);
	    switch(col) {
	    case PREFIX_COL:
		return namespace.getPrefix();
	    case URI_COL:
  		return namespace == Namespace.NO_NAMESPACE
		    ? "No namespace"
		    : namespace.getURI();
	    }
	    throw new ConsistencyException("Bogus column " + col);
	}

	public void setValueAt(Object value, int row, int col) {
	}

    }

    class TableMouseListener extends MouseAdapter {

	RowPopupMenu popup = new RowPopupMenu();

	TableMouseListener() { }

	public void mousePressed(MouseEvent e) {
	    if (SwingUtilities.isRightMouseButton(e)) {
		int row = table.rowAtPoint(e.getPoint());
		Debug.noteln("Table right press in row", row);
		popup.setRow(row);
		popup.show(e.getComponent(), e.getX(), e.getY());
	    }
	}

    }

    class RowPopupMenu extends JPopupMenu implements ActionListener {

	int row = -1;

	RowPopupMenu() {
	    super();
	    add(makeMenuItem("Change Color"));
	}

	JMenuItem makeMenuItem(String text) {
	    JMenuItem item = new JMenuItem(text);
	    item.addActionListener(CatchingActionListener.listener(this));
	    return item;
	}

	void setRow(int row) {
	    this.row = row;
	}

	public void actionPerformed(ActionEvent e) {
	    String command = e.getActionCommand();
	    Debug.noteln("Row popup command", command);
	    Debug.expect(row >= 0);
	    if (command.equals("Change Color")) {
		changeColor();
	    }
	    else
		throw new ConsistencyException
		    ("Nothing to do for " + command);
	}

	void changeColor() {
	    Namespace namespace = namespaces.getEntryNamespace(row);
	    Color color = JColorChooser.showDialog(this,
	        "Color for " + namespace,
		namespaces.getEntryColor(row));
	    if (color == null)
		return;		// user cancelled
	    namespaces.setColor(namespace, color);
	    model.fireTableRowsUpdated(row, row);
	    fireNamespaceEvent(new NamespaceEvent(namespace, color));
	}

    }

}
