/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Thu Mar 23 16:29:27 2006 by Jeff Dalton
 * Copyright: (c) 2002 - 2004, AIAI, University of Edinburgh
 */

package ix.util.xml;

import javax.swing.*;
import javax.swing.tree.*;
import javax.swing.event.*;

import java.awt.Color;
import java.awt.Container;
import java.awt.Component;
import java.awt.BorderLayout;
import java.awt.FlowLayout;
import java.awt.event.*;

import java.util.*;

// import java.io.StringReader;
import java.io.StringWriter;
import java.io.IOException;

// Imports for using JDOM
// import java.io.IOException;
import org.jdom.Document;
import org.jdom.Element;
import org.jdom.Attribute;
import org.jdom.Namespace;
import org.jdom.JDOMException;
import org.jdom.input.SAXBuilder;
import org.jdom.output.XMLOutputter;

import ix.iface.util.*;

import ix.util.*;
import ix.util.lisp.*;
import ix.util.reflect.*;
import ix.util.xml.event.*;

/**
 * A panel that contains an XML tree-editor.
 *
 * @see XMLTreeEditor
 * @see XMLTreeEditFrame
 */

public class XMLTreeEditPanel extends JPanel
       implements NamespaceListener,
		  ActionListener,
		  TreeSelectionListener {

    /** The document this panel is editing.
     */
    protected Document document;

    protected static final String ATTRIBUTES = "[attributes]";

    protected static EditorTree mostRecentlySelectedTree;

    protected TNodeFactory nodeFactory = new TNodeFactory();
    protected TNodeParser nodeParser = new TNodeParser();

    protected TemplateSyntax syntax = new TemplateSyntax();

    protected EditorTree docTree;
    protected DefaultTreeModel docModel;

    protected EditorTree templateTree;

    protected EditingTextArea editText;
    protected JButton submitButton;

    protected JSplitPane docSplit;	// document | templates
    protected JSplitPane textSplit;	// doc|tem | text

    public XMLTreeEditPanel() {
	// Default is FlowLayout.
	super(new BorderLayout());

	// Root node
	TNode docRoot = new TNode("root");
	docRoot.setShouldBeExpanded(true);

	// Tree model
	docModel = new DefaultTreeModel(docRoot);
	// docModel.setAsksAllowsChildren(true);

	// Doc Tree
	docTree = new EditorTree("document tree", docModel);
	// /\/: Not editable, because the bahaviour is too confusing
	// is buggy at least in Java 1.3.
	docTree.setEditable(false);
        docTree.getSelectionModel()
	    .setSelectionMode(TreeSelectionModel.SINGLE_TREE_SELECTION);
        docTree.setShowsRootHandles(true);
	docTree.addTreeSelectionListener(this);
	docTree.addMouseListener(new TreeMouseListener());

	// Template tree
	templateTree = makeTemplateTree();
	addTemplates();

	// Text editing
	editText = new EditingTextArea(0, 0);	// was 1, 30
	editText.setEditable(false);
	submitButton = makeButton("Enter"); 	// was "Submit Text"
	submitButton.setEnabled(false);

	// Scroll panes and borders
        JScrollPane docScroll = new JScrollPane(docTree);
	docScroll.setBorder
	    (BorderFactory.createTitledBorder("Document"));

	JScrollPane templateScroll = new JScrollPane(templateTree);
	templateScroll.setBorder
	    (BorderFactory.createTitledBorder("Templates"));

	JScrollPane textScroll = new JScrollPane(editText);
	JPanel textPanel = new JPanel();
	textPanel.setBorder
  	    (BorderFactory.createTitledBorder("Text"));

	textPanel.setLayout(new BorderLayout());
	textPanel.add(textScroll, BorderLayout.CENTER);
	Box enterPanel = Box.createVerticalBox();
	enterPanel.add(Box.createVerticalGlue());
	enterPanel.add(submitButton);
	textPanel.add(enterPanel, BorderLayout.EAST);

//  	textScroll.setBorder
//  	    (BorderFactory.createTitledBorder("Text"));

	// Split panes
	docSplit = new JSplitPane
	    (JSplitPane.HORIZONTAL_SPLIT, docScroll, templateScroll);
	docSplit.setOneTouchExpandable(true);
	// docSplit.setDividerSize(3);
	// docSplit.setResizeWeight(0.80);
	docSplit.setResizeWeight(1.00);

//  	textSplit = new JSplitPane
//  	    (JSplitPane.VERTICAL_SPLIT, docSplit, textScroll);
	textSplit = new JSplitPane
	    (JSplitPane.VERTICAL_SPLIT, docSplit, textPanel);
	// textSplit.setDividerSize(3);
	// textSplit.setResizeWeight(0.90);
	textSplit.setResizeWeight(1.00);

	// Panel contents
        add(textSplit, BorderLayout.CENTER);
	add(makeButtonPanel(), BorderLayout.SOUTH);

	// Register for Namespace events
	XMLTreeEditFrame.namespaces.addNamespaceListener(this);

    }

    protected void reset() {
	setDocRoot(new TNode("root"));
	editText.clear();
	// /\/: Need to reset template tree
    }

    protected XMLTreeEditFrame getEditFrame() {
	return (XMLTreeEditFrame)SwingUtilities.getRoot(this);
    }

    protected void setDocRoot(TNode root) {
	docModel.setRoot(root);
	root.setShouldBeExpanded(true);
    }

    public void editDocument(Document doc) {
	document = doc;
	setDocRoot(nodeFactory.nodeFrom(doc));
	if (mostRecentlySelectedTree == docTree)
	    editText.clear();
    }

    public Document getDocument() {
	document = nodeParser.documentFrom((TNode)docModel.getRoot());
	return document;
    }

    public void expandDocument(int depth) {
	TNode root = (TNode)docModel.getRoot();
	root.expandSubtree(depth);
	docTree.fixExpansions();
    }

    public void addRootChild(Document doc) {
	TNode root = (TNode)docModel.getRoot();
	TNode newNode = nodeFactory.nodeFrom(doc);
	docModel.insertNodeInto
	    (newNode, root, root.getChildCount());
	docTree.scrollPathToVisible(newNode.getTreePath());
    }

    public void editObject(Object object) {
	editDocument(syntax.xmlt().objectToDocument(object));
    }

    public Object getObject() {		// will select culprit node if error
	Document doc = getDocument();
	try {
	    return syntax.xmlt().objectFromDocument(doc);
	}
	catch (Exception e) {
	    Debug.noteException(e);
	    // Try to show and select the responsible node in the docTree.
	    Element probableBadElt = syntax.xmlt().getLastExaminedElement();
	    if (probableBadElt != null) {
		Debug.noteln("Bad elt", probableBadElt);
		TNode culprit = nodeParser.getNodeForElement(probableBadElt);
		if (culprit != null) {
		    TreePath path = culprit.getTreePath();
		    Debug.noteln("Bad node's path", path);
		    docTree.scrollPathToVisible(path);
		    docTree.setSelectionPath(path);
		}
	    }
	    throw new InvalidNode(Debug.describeException(e));
	}
    }

    /**
     * Finds text in the most recently selected tree even if the
     * tree is not in this panel.
     */
    public void findInTree(String targetText) {
	Debug.noteln("Edit panel asked to find " + Strings.quote(targetText));

	EditorTree destinationTree = mostRecentlySelectedTree;
	if (destinationTree == null) {
	    complain("No selected tree");
	    return;
	}

	DefaultTreeModel destinationModel =
	    (DefaultTreeModel)destinationTree.getModel();

	TNode root = (TNode)destinationModel.getRoot();
	TNode selected = destinationTree.getSelectedNode();

	// Go through the TNodes in "textual" order until we have
	// passed the selected node (if any), then continue going
	// through the nodes looking for the target text.
	boolean passedSelected = false;
	for (Enumeration e = root.preorderEnumeration();
	     e.hasMoreElements();) {
	    TNode at = (TNode)e.nextElement();
	    if (at == selected)
		passedSelected = true;
	    else if (passedSelected) {
		String text = (String)at.getUserObject();
		if (text.indexOf(targetText) >= 0) {
		    TreePath path = at.getTreePath();
		    Debug.noteln("Found text at path", path);
		    destinationTree.scrollPathToVisible(path);
		    destinationTree.setSelectionPath(path);
		    return;
		}
	    }
	}
	complain("Can't find " + Strings.quote(targetText));
    }

    protected Color getNamespaceColor(Namespace n) {
	return XMLTreeEditFrame.namespaces.getNamespaceColor(n);
    }

    protected void noteNamespace(Namespace n) {
	XMLTreeEditFrame.namespaces.noteNamespace(n);
    }

    public void namespaceEvent(NamespaceEvent e) {
	if (e.isColorChange())
	    repaint();
    }

    public void addTemplates() {
	addTemplatesFor(XML.config().treeEditorTemplateClassRoots());
	addTemplate(syntax.makeMapEntryNode());	//\/ ?
    }

    public void addTemplatesFor(Class[] classes) {
	DefaultTreeModel model = (DefaultTreeModel)templateTree.getModel();
	TNode root = (TNode)model.getRoot();

	List templates = syntax.makeTemplateNodes(classes);
	for (Iterator i = templates.iterator(); i.hasNext();) {
	    TNode t = (TNode)i.next();
	    root.add(t);
	}
	model.nodeStructureChanged(root);
    }

    public void addTemplate(TNode template) {
	DefaultTreeModel model = (DefaultTreeModel)templateTree.getModel();
	TNode root = (TNode)model.getRoot();
	root.add(template);
	model.nodeStructureChanged(root);
    }

    public void hideTemplates() {
	Debug.noteln("Hiding templates");
	docSplit.setDividerLocation(1.0);
    }

    protected EditorTree makeTemplateTree() {
	TNode root = new TNode("Templates");
	root.setShouldBeExpanded(true);
	EditorTree tree = new TemplateTree("template tree", root);
        tree.setEditable(false);
        tree.getSelectionModel()
	    .setSelectionMode(TreeSelectionModel.SINGLE_TREE_SELECTION);
        tree.setShowsRootHandles(true);
	tree.addTreeSelectionListener(this);
	tree.addMouseListener(new TreeMouseListener());
	return tree;
    }

    protected JPanel makeButtonPanel() {
	JPanel panel = new JPanel(); 	// defaults to FlowLayout
	panel.setLayout(new BoxLayout(panel, BoxLayout.X_AXIS));
	panel.add(Box.createHorizontalGlue());
	panel.add(makeButton("Add Child"));
	panel.add(makeButton("Copy Subtree"));
	panel.add(makeButton("Cut Subtree"));
	// panel.add(submitButton);
	panel.add(Box.createHorizontalGlue());
	return panel;
    }

    protected JButton makeButton(String command) {
	JButton b = new JButton(command);
	b.addActionListener(CatchingActionListener.listener(this));
	return b;
    }

    /**
     * Action interpreter for panel buttons
     */
    public void actionPerformed(ActionEvent e) {
	String command = e.getActionCommand();
	Debug.noteln("XMLTreeEditPanel action:", command);
	if (command.equals("Add Child")) {
	    addChild();
	}
	else if (command.equals("Copy Subtree")) {
	    copySubtree();
	}
	else if (command.equals("Cut Subtree")) {
	    cutSubtree();
	}
	else if (e.getSource() == submitButton) { // command text varies
	    editText.submit();
	}
	else
	    throw new ConsistencyException
		("Nothing to do for " + command);
    }

    /**
     * Adds a new node after the existing children of the selected
     * node.
     */
    public void addChild() {
	EditorTree tree = mostRecentlySelectedTree;
	if (tree == null) {
	    complain("No selected tree");
	    return;
	}
	TNode selected = tree.getSelectedNode();
	TNode newNode = new TNode("blank");
        DefaultTreeModel model = (DefaultTreeModel)tree.getModel();
	model.insertNodeInto
	    (newNode, selected, selected.getChildCount());
	tree.scrollPathToVisible(newNode.getTreePath());
    }

    /** 
     * Copy the subtree below and including the currently selected node.
     * This is essentially the same as a "Cut" without deletion, but it
     * also takes a copy of the subtree rather than taking those very
     * nodes.
     */ 
    public void copySubtree() {
	EditorTree tree = mostRecentlySelectedTree;
	if (tree == null) {
	    complain("No selected tree");
	    return;
	}
	TNode selected = tree.getSelectedNode();
	new CutFrame(selected.copySubtree(), "Copy");
    }

    /** 
     * "Cut" the subtree below and including the currently selected node.
     */ 
    public void cutSubtree() {
	EditorTree tree = mostRecentlySelectedTree;
	if (tree == null) {
	    complain("No selected tree");
	    return;
	}
	TNode selected = (TNode)tree.getSelectedNode();
	if (tree.cutNodeSubtree(selected))
	    new CutFrame(selected, "Cut");
    }

    protected void complain(Object message) {
	JOptionPane.showMessageDialog
	    (this, message, "Error", JOptionPane.ERROR_MESSAGE);
    }

    class InvalidCommand extends RuntimeException {
	InvalidCommand(String message) {
	    super(message);
	}
    }

    /**
     * Required by the TreeSelectionListener interface.
     *
     * Called for both the document and the template trees.
     */
    public void valueChanged(TreeSelectionEvent e) {
	EditorTree source = (EditorTree)e.getSource();
        TNode node = (TNode)source.getLastSelectedPathComponent();
        if (node == null)
	    return;
	Debug.noteln(source.getName() + " selection", node.getTreePath());
	setSelectedTree(source);
	editText.display(node);
    }

    void setSelectedTree(EditorTree tree) {
	if (mostRecentlySelectedTree != null
	        && mostRecentlySelectedTree != tree) {
	    Debug.noteln("Changing selected tree");
	    // Clear the current selection, so that there's only
	    // one selected node anywhere in any tree.  This makes
	    // it clear where the contents of a cutframe will go,
	    // though it means a "find" position may be lost.  /\/
	    mostRecentlySelectedTree.clearSelectionEtc();
	}
	mostRecentlySelectedTree = tree;
    }


    /*
     * Node classes
     */

    /**
     * A TNode in the namespace used for object class and
     * field names.
     */
    class ObjTNode extends TNode {
	ObjTNode(String userObject) {
	    super(userObject);
	    setNamespace(syntax.xmlt().getHomeNamespace());
	}
    }
    
    /**
     * A TreeNode for use in EditorTrees.
     */
    class TNode extends DefaultMutableTreeNode {

	Namespace namespace = Namespace.NO_NAMESPACE;
	boolean shouldBeExpanded = false;

	String renderingText = null;

	TNode(Object userObject) {
	    super(userObject);
	}

	Namespace getNamespace() {
	    return namespace;
	}

	void setNamespace(Namespace n) {
	    noteNamespace(n);
	    renderingText = null;
	    namespace = n;
	}

	boolean shouldBeExpanded() {
	    return shouldBeExpanded;
	}

	void setShouldBeExpanded(boolean sbe) {
	    shouldBeExpanded = sbe;
	}

	void setSubtreeShouldBeExpanded(boolean sbe) {
	    for (Enumeration e = preorderEnumeration();
		 e.hasMoreElements();) {
		TNode n = (TNode)e.nextElement();
		n.setShouldBeExpanded(sbe);
	    }
	}

	void expandSubtree(int depth) {
	    if (depth >= 0) {
		setShouldBeExpanded(true);
		for (Enumeration e = children(); e.hasMoreElements();) {
		    TNode child = (TNode)e.nextElement();
		    child.expandSubtree(depth - 1);
		}
	    }
	}

	public TreePath getTreePath() {
	    return new TreePath(getPath());
	}

	public void setUserObject(Object object) {
	    Debug.noteln("Setting user object to", object);
	    super.setUserObject(object);
	}

	TNode copySubtree() {
	    TNode result = new TNode(getUserObject());
	    result.namespace = namespace;
	    result.shouldBeExpanded = shouldBeExpanded;
	    for (Enumeration e = children(); e.hasMoreElements();) {
		TNode child = (TNode)e.nextElement();
		result.add(child.copySubtree());
	    }
	    return result;
	}

    }

    /**
     * Makes {@link XMLTreeEditPanel.TNode}s from JDOM Documents and Elements.
     */
    class TNodeFactory {

	TNodeFactory() { }

	TNode nodeFrom(Document doc) {
	    return nodeFrom(doc.getRootElement());
	}

	TNode nodeFrom(Element elt) {
	    TNode node = new TNode(elt.getName());
	    node.setNamespace(elt.getNamespace());

	    if (!elt.getAttributes().isEmpty()) {
		// Add attributes
		node.add(makeAttributesNode(elt));
	    }

	    // /\/: Doesn't yet handle mixed content.  If there are
	    // child elements, it's assumed they're all that matter.

	    if (elt.getContent().isEmpty()) {
		// If no content, we're done.
		return node;
	    }
	    else if (elt.getChildren().isEmpty()) {
		// No children - so take only the text.
		String text = elt.getText();
		if (text != null && !text.equals(""))
		    node.add(new TNode(text));
		return node;
	    }
	    else {
		// Add any children
		for (Iterator i = elt.getChildren().iterator(); i.hasNext();) {
		    Element fieldElt = (Element)i.next();
		    node.add(nodeFrom(fieldElt));
		}
		return node;
	    }
	}

	TNode makeAttributesNode(Element elt) {
	    TNode attributesNode = new TNode(ATTRIBUTES);
	    for (Iterator i = elt.getAttributes().iterator(); i.hasNext();) {
		Attribute fieldAttr = (Attribute)i.next();
		String fieldName = fieldAttr.getName();
		String fieldValue = fieldAttr.getValue();
		TNode attrNode = new TNode(fieldName + "=" + fieldValue);
		attrNode.setNamespace(fieldAttr.getNamespace());
		attributesNode.add(attrNode);
	    }
	    return attributesNode;
	}

    }

    /**
     * Responsible for understanding the object-like structure of tree
     * nodes and for converting nodes into JDOM Documents or Elements.
     */
    class TNodeParser {

	Map elementToNodeMap = new WeakHashMap();

	TNodeParser() { }

	Document documentFrom(TNode node) {
	    elementToNodeMap.clear();
	    Element rootElt = elementFrom(node);
	    return new Document(rootElt);
	}

	Element elementFrom(TNode node) {
	    Element result;
	    if (isAttributesList(node))
		throw syntaxError(node, "Misplaced attributes");
	    else if (isAtomicValue(node))
		result = valueElementFrom(node);
	    else if (isTextWithAttributes(node))
		result = textWithAttributesFrom(node);
	    else
		result = structElementFrom(node);
	    elementToNodeMap.put(result, node);
	    return result;
	}

	TNode getNodeForElement(Element elt) {
	    return (TNode)elementToNodeMap.get(elt);
	}

	Element valueElementFrom(TNode node) {
	    String name = (String)node.getUserObject();
	    TNode valueNode = (TNode)node.getChildAt(0);
	    String value = (String)valueNode.getUserObject();
	    Element elt = new Element(name, node.getNamespace());
	    elt.setText(value);
	    return elt;
	}

	Element textWithAttributesFrom(TNode node) {
	    // /\/: This is like an atomic value, but with
	    // attributes.  These things can't be translated
	    // to objects but we may as well allow them in XML.
	    // Note that we still don't handle full mixed content.
	    String name = (String)node.getUserObject();
	    Element elt = new Element(name, node.getNamespace());

	    Debug.expect(hasAttributes(node));
	    addAttributes(elt, node);
	    
	    TNode valueNode = (TNode)node.getChildAt(1);
	    String value = (String)valueNode.getUserObject();
	    elt.setText(value);
	    return elt;
	}

	Element structElementFrom(TNode node) {
	    String name = (String)node.getUserObject();
	    Enumeration children = node.children();

	    Element elt = new Element(name, node.getNamespace());
	    if (hasAttributes(node)) {
		addAttributes(elt, node);
		children.nextElement();		// remove atributes TNode
	    }
	    while (children.hasMoreElements()) {
		TNode child = (TNode)children.nextElement();
		elt.addContent(elementFrom(child));
	    }
	    return elt;

	}

	void addAttributes(Element elt, TNode node) {
	    TNode attributesNode = (TNode)node.getChildAt(0);
	    for (Enumeration e = attributesNode.children();
		 e.hasMoreElements();) {
		TNode attrNode = (TNode)e.nextElement();
		addAttribute(elt, attrNode);
	    }
	}

	void addAttribute(Element elt, TNode attrNode) {
	    String[] parts = parseAttribute(attrNode);
	    elt.setAttribute(parts[0], parts[1], attrNode.getNamespace());
	}

	String[] parseAttribute(TNode attrNode) {
	    if (!isAttribute(attrNode))
		throw syntaxError(attrNode, "Invalid attribute");
	    String text = (String)attrNode.getUserObject();
	    String[] parts = Strings.breakAtFirst("=", text);
	    if (attrNode.getChildCount() != 0
		    || parts[0].equals("")
		 /* || parts[1].equals("") */ )
		throw syntaxError(attrNode, "Invalid attribute");
	    return parts;
	}

	boolean hasAttributes(TNode node) {
	    return node.getChildCount() >= 1
		&& isAttributesList((TNode)node.getChildAt(0));
	}

	boolean isAttributesList(TNode node) {
	    return node.getUserObject().equals(ATTRIBUTES);
	}

	boolean isAttribute(TNode node) {
	    TNode parent = (TNode)node.getParent();
	    return parent != null
		&& isAttributesList(parent)
		&& node.getChildCount() == 0;
	}

	String getAttributeName(TNode node) {
	    String[] parts = parseAttribute(node);
	    return parts[0];
	}

	String getAttributeValue(TNode node) {
	    String[] parts = parseAttribute(node);
	    return parts[1];
	}

	TNode getAttributeContainingObjectNode(TNode attrNode) {
	    // The node that corresponds to the object that has
	    // the attribute as a field.
	    Debug.expect(isAttribute(attrNode));
	    TNode typeNode = (TNode)attrNode.getParent().getParent();
	    if (typeNode == null)
		throw syntaxError(attrNode, "Attribute outside object");
	    return typeNode;
	}

	boolean isAtomicValue(TNode node) {
	    return node.getChildCount() == 1
		&& !isAttributesList(node)
		&& !isAttributesList((TNode)node.getChildAt(0))
		&& looksLikeDataValue((TNode)node.getChildAt(0));
	}

//  	void setAtomicValue(TNode node, Object userObj) {
//  	    Debug.expect(isAtomicValue(node));
//  	    TNode valChild = (TNode)node.getChildAt(0);
//  	    valChild.setUserObject(userObj);
//  	}

	boolean isTextWithAttributes(TNode node) {
	    return node.getChildCount() == 2
		&& isAttributesList((TNode)node.getChildAt(0))
		&& looksLikeDataValue((TNode)node.getChildAt(1));
	}


	boolean looksLikeDataValue(TNode node) {
	    return node.getChildCount() == 0
		&& node.getNamespace() == Namespace.NO_NAMESPACE;
	}

	InvalidNode syntaxError(TNode culprit, String message) {
	    TreePath path = culprit.getTreePath();
	    docTree.scrollPathToVisible(path);
	    docTree.setSelectionPath(path);
	    return new InvalidNode(message);
	}

    }

    class InvalidNode extends RuntimeException {
	InvalidNode(String message) {
	    super(message);
	}
    }

    /**
     * Knows about the syntax of Java classes.
     */
    class TemplateSyntax extends XMLSyntax {

	TemplateSyntax() {
	    super();
	}

	List makeTemplateNodes(Class[] topClasses) {
	    // First find all relevant classes.
	    List relevantClasses = classSyntax.relevantClasses(topClasses);
	    // Create a template TNode for each relevant class
	    // if this will be useful and we know how.
	    List result = new LinkedList();
	    for (Iterator i = relevantClasses.iterator(); i.hasNext();) {
		Class c = (Class) i.next();
		TNode template = templateFrom(c);
		if (template != null) 		// useful and know how?
		    result.add(template);
	    }
	    return result;
	}

	TNode templateFrom(Class c) {
	    ClassDescr cd = getClassDescr(c);
	    if (cd.isStruct() && !cd.isAbstract())
		return structTemplateFrom(c);
	    else
		return null;
	}

	TNode structTemplateFrom(Class c) {
	    ClassDescr cd = getClassDescr(c);
	    String className = getElementName(cd);
	    List fields = cd.getFieldDescrs();
	    List attrFields = attributeFields(fields);
	    List eltFields = elementFields(fields);
	    Debug.expect(fields.size() == attrFields.size()
			                  + eltFields.size());
	    TNode result = new ObjTNode(className);
	    if (attrFields.isEmpty()) {
		if (eltFields.isEmpty()) return null;
	    }
	    else
		result.add(makeAttributesNode(attrFields));
	    for (Iterator ei = eltFields.iterator(); ei.hasNext();) {
		FieldDescr fd = (FieldDescr)ei.next();
		result.add(makeFieldNode(fd));
	    }
	    return result;
	}

	TNode makeAttributesNode(List attrFields) {
	    TNode attrNode = new TNode(ATTRIBUTES);
	    for (Iterator ai = attrFields.iterator(); ai.hasNext();) {
		FieldDescr fd = (FieldDescr)ai.next();
		String attrName = getElementName(fd);
		attrNode.add(new TNode(attrName + "="));
	    }
	    return attrNode;
	}

	TNode makeFieldNode(FieldDescr fd) {
	    TNode fieldNode = new ObjTNode(getElementName(fd));
	    ClassDescr ftype = fd.getTypeDescr();

	    if (ftype.isList()) {
		String listName = getElementName(List.class);
		TNode listNode = new ObjTNode(listName);
		listNode.add(new TNode(listName + "-element..."));
		fieldNode.add(listNode);
	    }
	    else if (ftype.isSet()) {
		String setName = getElementName(Set.class);
		TNode setNode = new ObjTNode(setName);
		setNode.add(new TNode(setName + "-element..."));
		fieldNode.add(setNode);
	    }
	    else if (ftype.isMap()) {
		String mapName = getElementName(Map.class);
		TNode mapNode = new ObjTNode(mapName);
		mapNode.add(makeMapEntryNode());
		fieldNode.add(mapNode);
	    }
	    else {
		String fieldClassName = getElementName(ftype);
		String valueName = fieldClassName + "-value";
		TNode fieldClassNode = new ObjTNode(fieldClassName);
		TNode fieldValueNode = ftype.isPrimitive()
		    ? new TNode(valueName)    // primitive -- no namespace
		    : new ObjTNode(valueName); // else I-X namespace
		fieldClassNode.add(fieldValueNode);
		fieldNode.add(fieldClassNode);
	    }
	    return fieldNode;
	}

	TNode makeMapEntryNode() {
	    ClassFinder cf = classSyntax.getClassFinder();
	    String entryName = cf.externalName("MapEntry");
	    TNode entryNode = new ObjTNode(entryName);
	    entryNode.add(new ObjTNode("key"));
	    entryNode.add(new ObjTNode("value"));
	    return entryNode;
	}

	XMLTranslator xmlt() {
	    return xmlt;
	}

	Class classForXmlName(String name) {
  	    return classSyntax.classForExternalName(name);
  	}


    }

    /*
     * Mouse listener
     *
     * /\/: There are separate listeners for the document and template
     * trees.  The main reason for using a separate class is that
     * it can extend MouseAdapter; however, for symmetry, it might
     * be better for the TreeSelectionListener to be handled in the
     * same way even though there isn't an adapter class to extend.
     */

    class TreeMouseListener extends MouseAdapter {

	public void mousePressed(MouseEvent e) {
	    EditorTree selTree = (EditorTree)e.getSource();
	    int selRow = selTree.getRowForLocation(e.getX(), e.getY());
	    TreePath selPath = selTree.getPathForLocation(e.getX(), e.getY());

	    if (selRow == -1)
		return;
	    if (!SwingUtilities.isRightMouseButton(e))
		return;

	    Debug.noteln("Right click in " + selTree.getName(), selPath);
	    TNode selNode = (TNode)selPath.getLastPathComponent();
	    JPopupMenu popup = makeNodePopupMenu(selTree, selNode);
	    popup.show(e.getComponent(), e.getX(), e.getY());
	}

    }

    /**
     * Factory method that makes a popup menu for right-press
     * on a tree node.
     */
    JPopupMenu makeNodePopupMenu(EditorTree tree, TNode node) {
	return new NodePopupMenu(tree, node);
    }

    /**
     * The default class of popup menu that appears when the user
     * does a right-button press on a tree node.
     *
     * @see #makeNodePopupMenu(XMLTreeEditPanel.EditorTree,
     *                         XMLTreeEditPanel.TNode)
     */
    class NodePopupMenu extends JPopupMenu
                        implements ActionListener {

	EditorTree tree;
	DefaultTreeModel model;
	TNode node;

	NodePopupMenu(EditorTree tree, TNode node) {
	    super();
	    this.tree = tree;
	    this.model = (DefaultTreeModel)tree.getModel();
	    this.node = node;
	    addMenuItems();
	}

	void addMenuItems() {
	    if (nodeParser.isAttribute(node))
		addValueMenuItemIfKnown();

	    add(makeMenuItem("Fully Expand", !node.isLeaf()));
	    add(makeMenuItem("Fully Collapse", !node.isLeaf()));

	    if (!nodeParser.isAttributesList(node))
		add(makeNamespaceMenu());

	    add(makeMenuItem("Add Child"));
	    add(makeMenuItem("Copy Subtree"));
	    add(makeMenuItem("Cut Subtree"));

	    if (nodeParser.isAttributesList(node))
		add(makeMenuItem("Delete Valueless Attributes"));
	    else if (node.getChildCount() > 0)
		add(makeMenuItem("Delete Closed Subnodes"));
	}

	JMenuItem makeMenuItem(String text) {
	    return makeMenuItem(text, true);
	}

	JMenuItem makeMenuItem(String text, boolean enabled) {
	    JMenuItem item = new JMenuItem(text);
	    item.addActionListener(CatchingActionListener.listener(this));
	    item.setEnabled(enabled);
	    return item;
	}

	void addValueMenuItemIfKnown() {
	    // We have to find the class of the object whose attribute
	    // it is, and then look up the class of the field that
	    // corresponds to the attribute.
	    TNode objNode = nodeParser
		.getAttributeContainingObjectNode(node);
	    String objClassName = (String)objNode.getUserObject();
	    Class objClass = syntax.classForXmlName(objClassName);
	    String fieldName = nodeParser.getAttributeName(node);
	    FieldDescr fd = syntax.getClassDescr(objClass)
		                .fieldForExternalName(fieldName);
	    if (fd != null && fd.getTypeDescr().isEnumeration()) {
		List values = syntax.getEnumerationValues(fd.getType());
		if (values != null) {
		    JMenu valueMenu = makeValueMenu(values);
		    add(valueMenu);
		}
	    }
	}

	JMenu makeValueMenu(List values) {
	    JMenu menu = new JMenu("Set Value");
	    for (Iterator i = values.iterator(); i.hasNext();) {
		Object value = i.next();
		JMenuItem item = makeMenuItem(value.toString());
		item.setActionCommand("setValue");
		menu.add(item);
	    }
	    return menu;
	}

	JMenu makeNamespaceMenu() {
	    JMenu menu = new JMenu("Set Namespace");
	    List namespaces = XMLTreeEditFrame.namespaces.getNamespaces();
	    for (Iterator i = namespaces.iterator(); i.hasNext();) {
		Namespace namespace = (Namespace)i.next();
		String prefix = namespace.getPrefix();
		String uri = namespace.getURI();
		Color color = XMLTreeEditFrame.namespaces
		    .getNamespaceColor(namespace);
		String spec = namespace == Namespace.NO_NAMESPACE
		    ? "No Namespace"
		    : (prefix.equals("") ? uri : prefix + "=" + uri); 
		JMenuItem item = makeMenuItem(spec);
		item.setForeground(color);
		item.setActionCommand("setNamespace");
		menu.add(item);
	    }
	    return menu;
	}

	public void actionPerformed(ActionEvent e) {
	    String command = e.getActionCommand();
	    Debug.noteln("Item popup command", command);
	    if (command.equals("Fully Expand")) {
		fullyExpand();
	    }
	    else if (command.equals("Fully Collapse")) {
		fullyCollapse();
	    }
	    else if (command.equals("Add Child")) {
		addChild();
	    }
	    else if (command.equals("setNamespace")) {
		JMenuItem item = (JMenuItem)e.getSource();
		setNamespace(item.getText());
	    }
	    else if (command.equals("Copy Subtree")) {
		copySubtree();
	    }
	    else if (command.equals("Cut Subtree")) {
		cutSubtree();
	    }
	    else if (command.equals("setValue")) {
		JMenuItem item = (JMenuItem)e.getSource();
		String value = item.getText();
		setValue(value);
	    }
	    else if (command.equals("Delete Valueless Attributes")) {
		deleteValuelessAttributes();
	    }
	    else if (command.equals("Delete Closed Subnodes")) {
		deleteClosedSubnodes();
	    }
	    else {
		throw new ConsistencyException
		    ("Nothing to do for " + command);
	    }
	}

	void fullyExpand() {
	    node.setSubtreeShouldBeExpanded(true);
	    tree.fixExpansions(node);
	}

	void fullyCollapse() {
	    node.setSubtreeShouldBeExpanded(false);
	    tree.fixExpansions(node);
	}

	void setNamespace(String spec) {
	    Namespace n = spec.equals("No Namespace")
		? Namespace.NO_NAMESPACE
		: XMLTreeEditFrame.namespaces.parseNamespaceSpec(spec);
	    node.setNamespace(n);
	    model.nodeChanged(node);
	}

	void addChild() {
	    TNode newChild = new TNode("blank");
	    model.insertNodeInto
		(newChild, node, node.getChildCount());
	    tree.scrollPathToVisible(newChild.getTreePath());
	}

	void copySubtree() {
	    new CutFrame(node.copySubtree(), "Copy");
	}

	void cutSubtree() {
	    if (tree.cutNodeSubtree(node))
		new CutFrame(node, "Cut");
	}

	void setValue(String value) {
	    String text = (String)node.getUserObject();
	    String typeName = Strings.beforeFirst("=", text);
	    node.setUserObject(typeName + "=" + value);
	    model.nodeChanged(node);
	}

	void deleteValuelessAttributes() {
	    // Must extract a separate list of the children so that
	    // we're not going through the node.children() Enumeration
	    // while we're deleting.
	    LList children = Seq.toLList(node.children());
	    for (Iterator i = children.iterator(); i.hasNext();) {
		TNode child = (TNode)i.next();
		if (nodeParser.getAttributeValue(child).trim().equals(""))
		    model.removeNodeFromParent(child);
	    }
	}

	void deleteClosedSubnodes() {
	    // Must extract a separate list of the children so that
	    // we're not going through the node.children() Enumeration
	    // while we're deleting.
	    LList children = Seq.toLList(node.children());
	    for (Iterator i = children.iterator(); i.hasNext();) {
		TNode child = (TNode)i.next();
		if (!child.shouldBeExpanded())
		    model.removeNodeFromParent(child);
	    }
	}

    }

    /**
     * JTree subclass used for all trees in the editor.
     */
    class EditorTree extends JTree {

	// Inherits get/setName from Component

	RecordingExpansionListener expansionListener
	    = new RecordingExpansionListener(this);

	EditorTree(String name, TreeNode root) {
	    this(name, new DefaultTreeModel(root));
	}

	EditorTree(String name, TreeModel model) {
	    super(model);
	    setName(name);
	    addTreeExpansionListener(expansionListener);
	    putClientProperty("JTree.lineStyle", "Angled");

	    // Here's how we would change the "folder" icons
	    // (Now TNodeRenderer makes them null -- see below.)
// 	    DefaultTreeCellRenderer tcr = new DefaultTreeCellRenderer();
// 	    tcr.setOpenIcon(Util.resourceImageIcon("ix-symbol-fold.gif"));
// 	    tcr.setClosedIcon(Util.resourceImageIcon("ix-symbol-unfold.gif"));
// 	    tcr.setLeafIcon(Util.resourceImageIcon("ix-symbol-unfold.gif"));
// 	    setCellRenderer(tcr);

	    DefaultTreeCellRenderer tcr = new TNodeRenderer();
	    setCellRenderer(tcr);

	}

	void clearSelectionEtc() {
	    clearSelection();
	    editText.clear();
	}

	boolean cutNodeSubtree(TNode node) {
	    TNode parent = (TNode)node.getParent();
	    if (parent == null)
		throw new InvalidCommand
		    ("A root cannot be cut");
	    DefaultTreeModel model = (DefaultTreeModel)getModel();
	    model.removeNodeFromParent(node);
	    return true;
	}

	TNode getSelectedNode() {
	    TNode selected = (TNode)getLastSelectedPathComponent();
	    if (selected == null)
		throw new InvalidCommand("No selected node");
	    else
		return selected;
	}

	void fixExpansions() {
	    fixExpansions((TNode)getModel().getRoot());
	}

	void fixExpansions(TNode node) {
    	    try {
		expansionListener.active = false;
		fixExpansionsRec(node);
	    }
	    finally {
		expansionListener.active = true;
	    }
	}

	private void fixExpansionsRec(TNode node) {
	    // Debug.noteln("Fixing expanion at", node);
	    // First, recursively process children
	    for (Enumeration e = node.children(); e.hasMoreElements();) {
		fixExpansionsRec((TNode)e.nextElement());
	    }
	    // Now this node
	    TreePath path = node.getTreePath();
	    if (node.shouldBeExpanded()) {
		if (!isExpanded(path))
		    expandPath(path);
	    }
	    else {
		if (!isCollapsed(path))
		    collapsePath(path);
	    }
	}

    }

    /**
     * A tree variety used for templates
     */
    class TemplateTree extends EditorTree {

	TemplateTree(String name, TreeNode root) {
	    super(name, root);
	}

	boolean cutNodeSubtree(TNode node) {
	    if (Util.dialogConfirms(XMLTreeEditPanel.this,
		     "Are you sure you want to cut a template?"))
		return super.cutNodeSubtree(node);
	    else
		return false;
	}

    }

    /**
     * A TreeExpansionListener that records whether a node should
     * be expanded or not.
     *
     * @see XMLTreeEditPanel.TNode#setShouldBeExpanded(boolean)
     */
    class RecordingExpansionListener implements TreeExpansionListener {

	JTree tree;
	boolean active = true;

	RecordingExpansionListener(JTree tree) {
	    this.tree = tree;
	}

	public void treeExpanded(TreeExpansionEvent e) {
	    if (active) {
		Debug.noteln("Tree expanding", e.getPath());
		theNode(e).setShouldBeExpanded(true);
	    }
	}

	public void treeCollapsed(TreeExpansionEvent e) {
	    if (active) {
		Debug.noteln("Tree collapsing", e.getPath());
		theNode(e).setShouldBeExpanded(false);
	    }
	}

	TNode theNode(TreeExpansionEvent e) {
	    Debug.expect(e.getSource() == tree);
	    return (TNode)e.getPath().getLastPathComponent();
	}

    }

    /**
     * A renderer that knows about namespaces.
     */
    class  TNodeRenderer extends DefaultTreeCellRenderer {

	public TNodeRenderer() {
	    super();
	    // Eliminate the icons.
	    setOpenIcon(null);
	    setClosedIcon(null);
	    setLeafIcon(null);
	}

	public Component getTreeCellRendererComponent
	                     (JTree tree,
			      Object value,
			      boolean sel,
			      boolean expanded,
			      boolean leaf,
			      int row,
			      boolean hasFocus) {

	    super.getTreeCellRendererComponent
		(tree, value, sel, expanded, leaf, row, hasFocus);

  	    TNode node = (TNode)value;
  	    Namespace namespace = node.getNamespace();
  	    if (namespace != null && node.renderingText == null) {
  		String prefix = namespace.getPrefix();
  		if (prefix != null && !prefix.equals(""))
  		    node.renderingText = prefix + ":" + getText();
  	    }
  	    if (node.renderingText != null)
  		setText(node.renderingText);
  	    setForeground(getNamespaceColor(namespace));

	    return this;
	}

    }

    /**
     * A text area for editing string values
     */
    class EditingTextArea extends JTextArea {

	TNode editingNode;
	TNode valueNode;
	boolean autoSelect = false;

	EditingTextArea(int height, int width) {
	    super(height, width);
	    addMouseListener(new MouseAdapter() {
		public void mousePressed(MouseEvent e) {
		    if (autoSelect) {
			autoSelect = false;
			setSelectionStart(0);
			setSelectionEnd(getText().length());
		    }
		}
	    });
	}

	void display(TNode node) {
	    editingNode = node;
	    setText((String)node.getUserObject());
	    setCaretPosition(0);
	    setEditable(true);
	    submitButton.setEnabled(true);
	}

	void clear() {
	    if (editingNode != null) {
		editingNode = null;
		setText("");
		setEditable(false);
		submitButton.setEnabled(false);
	    }
	}

	public void setText(String t) {
	    super.setText(t);
	    autoSelect = true;
	}

	void submit() {
	    TNode selected = mostRecentlySelectedTree.getSelectedNode();
	    Debug.expect(selected == editingNode,
			 "Text was not from selected node");
	    editingNode.setUserObject(getText());

	    DefaultTreeModel destinationModel = 
		(DefaultTreeModel)mostRecentlySelectedTree.getModel();
	    destinationModel.nodeStructureChanged(editingNode);
	    // clear();
	}

    }

    /**
     * A frame that shows a subtree cut from the main editor panel.
     */
    class CutFrame extends JFrame implements ActionListener {

	EditorTree subtree;
	TNode subtreeNode;

	EditorTree destinationTree;
	DefaultTreeModel destinationModel;

	Container contentPane = getContentPane();

	CutFrame(TNode subtreeNode, String title) {
	    super(title);
	    this.subtreeNode = subtreeNode;

	    // Tree
	    subtree = new EditorTree(title + " tree", subtreeNode);
	    subtree.setEditable(false);
	    subtree.setSelectionModel(null);
	    subtree.setShowsRootHandles(true);
	    subtree.fixExpansions();

	    // Frame contents
	    JScrollPane scrollPane = new JScrollPane(subtree);
	    scrollPane.setBorder
		(BorderFactory.createTitledBorder("Subtree"));
	    contentPane.add(scrollPane, BorderLayout.CENTER);
	    JPanel buttons = makeButtonPanel();
	    buttons.setBorder
		(BorderFactory.createTitledBorder("Insert ..."));
	    contentPane.add(buttons, BorderLayout.SOUTH);

	    // pack();
	    setSize(400, 300);	// width, height
	    setVisible(true);
	}

	JPanel makeButtonPanel() {
	    JPanel panel = new JPanel(); 	// defaults to FlowLayout
	    panel.setLayout(new BoxLayout(panel, BoxLayout.X_AXIS));
	    panel.add(Box.createHorizontalGlue());
	    panel.add(makeButton("As"));
	    panel.add(makeButton("Before"));
	    panel.add(makeButton("After"));
	    panel.add(makeButton("Child"));
	    panel.add(makeButton("Discard"));
	    panel.add(Box.createHorizontalGlue());
	    return panel;
	}

	JButton makeButton(String command) {
	    JButton b = new JButton(command);
	    b.addActionListener(CatchingActionListener.listener(this));
	    return b;
	}

	/**
	 * Action interpreter for cut-frame buttons
	 */
	public void actionPerformed(ActionEvent e) {
	    String command = e.getActionCommand();
	    Debug.noteln("CutFrame action:", command);

	    if (command.equals("Discard")) {
		subtreeNode = null;
		finished();
		return;
	    }

	    destinationTree = mostRecentlySelectedTree;
	    if (destinationTree == null) {
		complain("No selected tree");
		return;
	    }
	    destinationModel = (DefaultTreeModel)destinationTree.getModel();

	    TNode selected = destinationTree.getSelectedNode();

	    if (command.equals("As")) {
		insertAs(selected);
	    }
	    else if (command.equals("Before")) {
		insertBefore(selected);
	    }
	    else if (command.equals("After")) {
		insertAfter(selected);
	    }
	    else if (command.equals("Child")) {
		insertUnder(selected);
	    }
	    else
		throw new ConsistencyException
		    ("Nothing to do for " + command);
	}

	void insertAs(TNode selected) {
	    TNode parent = (TNode)selected.getParent();
	    if (parent == null) {
		destinationModel.setRoot(subtreeNode);
	    }
	    else {
		int i = destinationModel.getIndexOfChild(parent, selected);
		destinationModel.removeNodeFromParent(selected);
		destinationModel.insertNodeInto(subtreeNode, parent, i);
	    }
	    editText.clear();	// selected node was replaced
	    finished();
	}

	void insertBefore(TNode selected) {
	    TNode parent = (TNode)selected.getParent();
	    if (parent == null)
		complain("Cannot insert before root");
	    else {
		destinationModel.insertNodeInto
		    (subtreeNode,
		     parent,
		     destinationModel.getIndexOfChild(parent, selected));
		finished();
	    }
	}

	void insertAfter(TNode selected) {
	    TNode parent = (TNode)selected.getParent();
	    if (parent == null)
		complain("Cannot insert at same level as root");
	    else {
		destinationModel.insertNodeInto
		    (subtreeNode,
		     parent,
		     destinationModel.getIndexOfChild(parent, selected) + 1);
		finished();
	    }
	}

	void insertUnder(TNode selected) {
	    destinationModel.insertNodeInto(subtreeNode, selected, 0);
	    finished();
	}

	void finished() {
	    if (subtreeNode != null) {
		// Subtree was inserted somewhere
		destinationTree.fixExpansions();
		destinationTree.scrollPathToVisible(subtreeNode.getTreePath());
	    }
	    // /\/: Need to really get rid of this frame somehow.
	    // setVisible(false);
	    dispose();
	}

    }

}
// Issues:
// * Is it OK to make a new popup menu each time?
