/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Wed Feb 23 13:13:21 2005 by Jeff Dalton
 * Copyright: (c) 2002 - 2004, AIAI, University of Edinburgh
 */

package ix.util.xml;

import javax.swing.*;

import java.awt.Component;

import java.io.*;
import java.util.*;

// Imports for using JDOM
import org.jdom.Document;
import org.jdom.output.XMLOutputter;

import ix.iface.domain.DomainParser;

import ix.util.*;

/**
 * A utility for saving objects and XML documents to files, conducting
 * dialogs as required.  Objects can be saved in a variety of syntaxes,
 * not only as XML.  The available syntaxes are determined from the
 * object's class by a {@link FileSyntaxManager}.
 *
 * @see XML#fileSyntaxManager()
 */
public class XMLSaver {

    protected File libDir = DomainParser.getLibraryDirectory();
    protected XMLTranslator xmlt = XML.defaultTranslator();
    protected JFileChooser fileChooser = null;
    protected Component frame;		// for dialogs
    protected Class outputObjectClass;

    /**
     * Creates an XMLSaver that uses the default XML translator
     * that is in force at the time this constructor is called.
     * @see XML#defaultTranslator()
     */
    public XMLSaver(Component frame) {
	this(frame, Object.class);
    }

    public XMLSaver(Component frame, Class outputObjectClass) {
	this.frame = frame;
	this.outputObjectClass = outputObjectClass;
    }

    /**
     * Returns the XMLTranslator that is used by the saveObject method
     * to translate an object to a JDOM document.
     */
    public XMLTranslator getXMLTranslator() {
	return xmlt;
    }

    /**
     * Sets the XMLTranslator that is used by this XMLSaver
     * to translate an object to a JDOM document.
     */
    public void setXMLTranslator(XMLTranslator xmlt) {
	this.xmlt = xmlt;
    }

    /**
     * Writes an object to a file in a form that corresponds to the
     * selected file type or extension, conducting any necessary
     * dialogs along the way.  It repeatedly asks the user to select
     * a file until either the object has been successfully written
     * or the user decides to cancel the operation.  The user is
     * informed of any exceptions thrown while attempting to write,
     * and "success" means that no exceptions were thrown.
     *
     * <p>The default is to Convert the object to a JDOM Document
     * and then call the saveDocument method.
     *
     * @see #saveDocument(Document)
     * @see #agreeFileToWrite()
     */
    public void saveObject(Object obj) {
	// Loop until either the data has successfully been written
	// to a file or else the user has decided to "Cancel" rather
	// that select a(nother) file.
	while (true) {

	    // Ask user to select a file.
	    File file = agreeFileToWrite();
	    if (file == null)
		return;			// user cancels

	    // Write, then return if successful.
	    try {
		writeObjectToFile(obj, file);
		return;
	    }
	    catch (Throwable t) {	// was just IOException /\/
		Debug.noteException(t);
		JOptionPane.showMessageDialog(frame,
		    new Object[] {
		        "Problem saving to " + Strings.quote(file.getPath()),
		        Debug.foldException(t)},
		    "Trouble saving to file",
		    JOptionPane.ERROR_MESSAGE);
		// Now 'round the loop again
	    }

	}


    }

    protected void writeObjectToFile(Object obj, File file) 
	      throws IOException {
	Debug.noteln("XMLSaver writing object to", file);
	// /\/: The path string is later turned back into a File.
	XML.writeObject(obj, file.getPath());
    }

    /**
     * Writes a JDOM Document to a file selected by the user, conducting
     * any necessary dialogs along the way.  It repeatedly asks the user
     * to select a file until either the Document has been successfully
     * written or the user decides to cancel the operation.  The user is
     * informed of any exceptions thrown while attempting to write, and
     * "success" means that no exceptions were thrown.
     *
     * @see #saveObject(Object)
     * @see #agreeFileToWrite()
     */
    public void saveDocument(Document doc) {

	// We want a file-chooser that's only for writing XML.
	FileSyntaxManager fsm = XML.fileSyntaxManager();
	JFileChooser chooser = new JFileChooser(libDir);
	chooser.setFileFilter
	    (new SyntaxFileFilter(fsm.getSyntaxForType("xml")));
	fileChooser = chooser;

	// Loop until either the data has successfully been written
	// to a file or else the user has decided to "Cancel" rather
	// that select a(nother) file.
	while (true) {

	    // Ask user to select a file.
	    File file = agreeFileToWrite();
	    if (file == null)
		return;			// user cancels

	    // Write, then return if successful.
	    try {
		writeDocumentTo(file, doc);
		return;
	    }
	    catch (Throwable t) {	// was just IOException /\/
		JOptionPane.showMessageDialog(frame,
		    new Object[] {
		        "Problem saving " + Strings.quote(file.getPath()),
		        Debug.foldException(t)},
		    "Trouble saving to file",
		    JOptionPane.ERROR_MESSAGE);
		// Now 'round the loop again
	    }

	}

    }

    /**
     * Writes the document to the file.
     *
     * @see XML#makePrettyXMLOutputter()
     */
    protected void writeDocumentTo(File file, Document doc)
	      throws IOException {
	Debug.noteln("Writing XML Document to", file);
	// Try to write out the data.
	// Get a stream to write to.
	OutputStream out = 
	    new BufferedOutputStream(new FileOutputStream(file));
	// Create an outputter.
	XMLOutputter outputter = XML.makePrettyXMLOutputter();
	// Output the XML.
	outputter.output(doc, out);
	out.flush();
    }

    /**
     * Asks the user to select a file and, if it already exists,
     * whether to overwrite it.  If the user decides to overwrite
     * an existing file, the current version will first be renamed;
     * if the user decides not to overwrite, he or she will be
     * asked to select a different file.  This continues until
     * either the user has agreed to a file or else has elected
     * to cancel.
     *
     * #see Util#renameToBackup(File)
     */
    protected File agreeFileToWrite() {

	while (true) {

	    // Ask user to select a file.
	    File file = chooseFileToWrite();
	    if (file == null)
		return null;	// user cancels

	    // See if file already exists.
	    if (file.exists()) {
		// Yes -- ask if file should be overwritten.
		switch(JOptionPane.showConfirmDialog(frame,
			"Overwrite " + Strings.quote(file.getPath()),
			"Confirm",
			JOptionPane.YES_NO_OPTION)) {
		case JOptionPane.YES_OPTION:
		    // Rename the existing file.
		    Util.renameToBackup(file);
		    break;
		case JOptionPane.NO_OPTION:
		    // Leave the existing file alone and go around
		    // the loop again to get another selection.
		    continue;
		}
	    }
	    // Return the file as the user's choice.
	    return file;
	}
    }

    /**
     * Asks the user to choose a file.  Returns null if the user cancels.
     */
    protected File chooseFileToWrite() {
	if (fileChooser == null)
	    fileChooser = makeFileChooser();
	int option = fileChooser.showSaveDialog(frame);
	if (option == JFileChooser.APPROVE_OPTION)
	    return fileChooser.getSelectedFile();
	else
	    return null;
    }

    /**
     * Constructs a suitable file-chooser for this XMLSaver's
     * output object class.
     */
    public JFileChooser makeFileChooser() {
	JFileChooser chooser = new JFileChooser(libDir);
	FileSyntaxManager fsm = XML.fileSyntaxManager();
	Set extensions = new TreeSet();
	for (Iterator i = fsm.getAllSyntaxes().iterator(); i.hasNext();) {
	    FileSyntax syntax = (FileSyntax)i.next();
	    if (fsm.canWrite(syntax, outputObjectClass)) {
		extensions.addAll(syntax.getFileTypes());
	    chooser.addChoosableFileFilter(new SyntaxFileFilter(syntax));
	    }
	}
	SyntaxFileFilter classFilter =
	    new SyntaxFileFilter(outputObjectClass, extensions);
	chooser.setFileFilter(classFilter);
	return chooser;
    }

}
