/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Wed Aug 25 15:12:19 2010 by Jeff Dalton
 * Copyright: (c) 2003 - 2006, 2009, 2010, AIAI, University of Edinburgh
 */

package ix.util.xml;

import java.io.File;
import java.io.FileFilter;
import java.io.IOException;
import java.net.URL;
import java.net.MalformedURLException;
import java.util.*;

import ix.util.*;

/**
 * Maps file names, URLs, etc to {@link FileSyntax}es.
 */
public class FileSyntaxManager {

    protected List syntaxes = new LinkedList();
    protected FileSyntax defaultSyntax = null;

    public FileSyntaxManager() {
	addSyntaxes();
    }

    protected void addSyntaxes() {
	FileSyntax xml = new FileSyntax.XMLFiles();
	addSyntax(xml);
	addSyntax(new FileSyntax.RDFFiles());
	setDefault(xml);
    }

    public List getAllSyntaxes() {
	return Collections.unmodifiableList(syntaxes);
    }

    public void addSyntax(FileSyntax s) {
	syntaxes.add(s);
    }

    public void setDefault(FileSyntax s) {
	defaultSyntax = s;
    }

    public FileSyntax getSyntax(String filename) {
	return getSyntaxForType(getType(filename));
    }

    public FileSyntax getSyntax(File file) {
	return getSyntax(file.getName());
    }

    public FileSyntax getSyntax(URL url) {
	return getSyntax(url.getFile());
    }

    public String getType(String filename) {
	int i = filename.lastIndexOf('.');
	String type = i < 0 ? "" : filename.substring(i + 1);
	return type;
    }

    public String getExceptType(String filename) {
	int i = filename.lastIndexOf('.');
	return i < 0 ? filename : filename.substring(0, i);
    }

    public FileSyntax getSyntaxForType(String fileType) {
	// There can be only one.
	fileType = fileType.toLowerCase();
	for (Iterator i = syntaxes.iterator(); i.hasNext();) {
	    FileSyntax s = (FileSyntax)i.next();
	    if (s.getFileTypes().contains(fileType))
		return s;
	}
	return defaultSyntax;
    }

    public Set getInputTypesForClass(Class c) {
	Set types = new TreeSet();
	for (Iterator i = getAllSyntaxes().iterator(); i.hasNext();) {
	    FileSyntax syntax = (FileSyntax)i.next();
	    if (canRead(syntax, c)) {
		types.addAll(syntax.getFileTypes());
	    }
	}
	return types;
    }

    public Set getOutputTypesForClass(Class c) {
	Set types = new TreeSet();
	for (Iterator i = getAllSyntaxes().iterator(); i.hasNext();) {
	    FileSyntax syntax = (FileSyntax)i.next();
	    if (canWrite(syntax, c)) {
		types.addAll(syntax.getFileTypes());
	    }
	}
	return types;
    }

    public boolean canRead(FileSyntax syntax, Class c) {
	// /\/: This test is sufficiently complicated that we
	// provide a method for it.
	return syntax.isAvailable()
	    && syntax.canRead()
	    && allowsClass(c, syntax.readableClasses());
    }

    public boolean canWrite(FileSyntax syntax, Class c) {
	// /\/: This test is sufficiently complicated that we
	// provide a method for it.
	return syntax.isAvailable()
	    && syntax.canWrite()
	    && allowsClass(c, syntax.writableClasses());
    }

    private boolean allowsClass(Class c, Collection classes) {
	// If a syntax can handle a superclass of the desired class,
	// then it should be able to handle all instances of the desired
	// class.  If it can handle a subclass of the desired class,
	// then it should still be able to handle some instances of the
	// desired class (the ones that are also instances of the
	// subclass).
	for (Iterator i = classes.iterator(); i.hasNext();) {
	    if (!areDisjoint(c, (Class)i.next()))
		return true;
	}
	return false;
    }

    private boolean areDisjoint(Class c1, Class c2) {
	return !c1.isAssignableFrom(c2) && !c2.isAssignableFrom(c1);
    }

    public void addAboutInfo(List about) {
	if (syntaxes.isEmpty()) return;
	about.add("");
	about.add("File syntaxes:");
	for (Iterator i = syntaxes.iterator(); i.hasNext();) {
	    FileSyntax s = (FileSyntax)i.next();
	    about.add("   " + s.getFileTypeDescription() +
		      " " + s.getFileTypes());
	    if (!s.isAvailable())
		about.add("      Not available in this agent");
	    if (!s.canRead())
		about.add("      Output only");
	    if (!s.canWrite())
		about.add("      Input only");
	    if (s.canRead())
		about.addAll(aboutRestriction("Input", s.readableClasses()));
	    if (s.canWrite())
		about.addAll(aboutRestriction("Output", s.writableClasses()));
	}
    }

    private List aboutRestriction(String direction, List classes) {
	if (classes.size() == 1 && classes.get(0) == Object.class)
	    return Collections.EMPTY_LIST;
	else {
	    String classNames = "";
	    for (Iterator i = classes.iterator(); i.hasNext();) {
		Class c = (Class)i.next();
		String name = XML.nameForClass(c);
		classNames = classNames.equals("")
		    ? name
		    : (i.hasNext()
		         ? classNames + ", " + name
		         : classNames + " or " + name);
	    }
	    return Collections.singletonList
		("      " + direction + " must be "
		          + Strings.indefinite(classNames));
	}
    }

    // /\/: XMLLoader and XMLSaver ought to provide the convenient
    // methods for the case when the source or destination is known
    // and dialogs with the user are not desired, but they don't,
    // and they're already quite complicated, so we'll do it here,
    // at least for now.

    public Object readObject(Class desiredClass, String resourceName) {
	URL url = toURL(resourceName);
	if (url == null)
	    throw new IllegalArgumentException
		("Can't find " + aClass(desiredClass) +
		 " resource named " + Strings.quote(resourceName));
	return readObject(desiredClass, url);
    }

    public Object readObject(Class desiredClass, URL url) {
	FileSyntax syntax = getSyntax(url);
	checkSyntax(syntax, desiredClass, "input");
	try {
	    Object result = syntax.readObject(url);
	    if (result == null)
		throw new NullPointerException
		    ("null " + XML.nameForClass(desiredClass));
	    if (!desiredClass.isInstance(result))
		throw new ClassCastException
		    ("Required " + aClass(desiredClass) +
		     " but found " + aClass(result.getClass()));
	    return result;
	}
	catch (Exception e) {
	    Debug.noteException(e);
	    throw new RethrownException
		(e, "Cannot read " + aClass(desiredClass) +
		    " from " + Strings.quote(url.toString()) +
		    ": " + Debug.describeException(e));
	}
    }

    private String aClass(Class c) {
	return Strings.indefinite(XML.nameForClass(c));
    }

    /**
     * Returns a sorted-map from the Files in a specified directory
     * to objects that are instances of a desired class.  This method
     * assumes that file types are a reliable guide to contents.
     * If any file looks like it could contain a description
     * (in some known syntax) of an instance of the desired class,
     * then an exception is thrown if it instead contains a
     * description of some other kind of object.
     */
    public SortedMap readAllObjects(Class desiredClass, String directoryName) {
	File[] files = getFiles(desiredClass, directoryName);
	SortedMap result = new TreeMap();
	for (int i = 0; i < files.length; i++) {
	    Object obj = readObject(desiredClass, files[i].getPath());
	    result.put(files[i], obj);
	}
	return result;
    }

    /**
     * Returns an array of Files in a specified directory that
     * might contain instances of a desired class.  This method
     * assumes that file types are a reliable guide to contents.
     * If any file looks like it could contain a description
     * (in some known syntax) of an instance of the desired class,
     * then it is included.
     *
     * @throws IllegalArgumentException if the directoryName
     *    is not the name of a directory.
     *
     * @throws Warning if the directory does not contain any suitable files.
     */
    public File[] getFiles(Class desiredClass, String directoryName) {
	// Find all the files in the directory that might contain
	// an instance of the desired class.
	File dir = new File(directoryName);
	final Set types = getInputTypesForClass(desiredClass);
	File[] files = dir.listFiles(new FileFilter() {
	    public boolean accept(File f) {
		String name = f.getName();
		String type = Strings.afterLast(".", name);
		return type != name 
		    && types.contains(type.toLowerCase());
	    }
	});
	if (files == null)
	    throw new IllegalArgumentException
		(directoryName + " does not name a directory");
	if (files.length == 0)
	    throw new Warning
		(directoryName + " does not contain suitable files");
	return files;
    }

    public void writeObject(Object obj, String filename) {
	FileSyntax syntax = getSyntax(filename);
	checkSyntax(syntax, obj.getClass(), "output");
	try {
	    syntax.writeObject(obj, new File(filename));
	}
	catch (IOException e) {
	    Debug.noteException(e);
	    throw new RethrownException
		(e, "Cannot write to " + Strings.quote(filename) +
		    ": " + Debug.describeException(e));
	}
    }

    protected void checkSyntax(FileSyntax syntax, Class objClass,
			       String direction) {
	Debug.noteln("Checking syntax " + syntax + " for " +
		     objClass.getName() + " " + direction);
	if (!syntax.isAvailable())
	    throw new UnsupportedOperationException
		("The file syntax for " + syntax.getFileTypeDescription() +
		 " is not available in this agent.");
	if (direction.equals("input")) {
	    if (canRead(syntax, objClass)) return;
	}
	else if (direction.equals("output")) {
	    if (canWrite(syntax, objClass)) return;
	}
	else {
	    throw new ConsistencyException("Unknown direction", direction);
	}
	throw new UnsupportedOperationException
	    ("The file syntax for " + syntax.getFileTypeDescription() +
	     " cannot " + direction + " " + aClass(objClass) + ".");
    }

    /**
     * Converts the name to a URL where the name can be a URL,
     * the name of an existing file, or a resource accessible
     * via this FileSyntaxManager's class's class loader.
     * Those 3 possibilities are tried in turn, and if none
     * of them succeeds, the result is null.
     *
     * @return a URL or null
     */
    public URL toURL(String resourceName) {
	String name = resourceName;
	// Try it in turn as URL, file name, and resource.
	try {
	    // If it looks enough like a URL, assume it is.
	    return new URL(name);
	}
	catch(MalformedURLException e1) {
	    try {
		// Try it as a file name.
		File file = new File(name);
		if (file.exists()) {
		    return file.toURL();
		}
	    }
	    catch (MalformedURLException e2) {
		Debug.expect(false, "Can't convert file to URL", name);
	    }
	    catch (java.security.AccessControlException ace) {
		// Can happen in applets
		if (Parameters.isApplet()) {
                    if (Debug.isOn())
                        Debug.noteException(ace, false);
                }
		else
		    Debug.displayException
			("Can't see if file " + name + " exists",
			 ace);
	    }
	    // Maybe it's a resource
	    return this.getClass().getClassLoader().getResource(name);
	}
    }

    /**
     * Like {@link #toURL(String)} but throws an IllegalArgumentException
     * rather than returning null.
     */
    public URL requireURL(String resourceName) {
	URL url = toURL(resourceName);
	if (url == null)
	    throw new IllegalArgumentException
		("Can't find a resource named " + Strings.quote(resourceName));
	return url;
    }

}
