/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Mon May 11 16:15:25 2009 by Jeff Dalton
 * Copyright: (c) 2002 - 2005, 2007 - 2009, AIAI, University of Edinburgh
 */

package ix.util.xml;

import java.lang.reflect.*;
import java.util.*;

import java.net.URL;
import java.io.*;

// Imports for using JDOM
// import java.io.IOException;
import org.jdom.Document;
import org.jdom.Element;
import org.jdom.Attribute;
import org.jdom.Namespace;
import org.jdom.JDOMException;
import org.jdom.input.SAXBuilder;
import org.jdom.output.XMLOutputter;

import ix.util.*;
import ix.util.lisp.*;
import ix.util.reflect.*;

public class CompactXMLTranslator extends XMLTranslator {

    public CompactXMLTranslator() {
	super();
    }

    public CompactXMLTranslator(ClassSyntax syntax) {
	super(syntax);
    }

    /* ************************ *
     *      Objects to XML      *
     * ************************ */

    @Override    
    protected Element structToElement(Object obj) {
	Class c = obj.getClass();
	ClassDescr cd = classSyntax.getClassDescr(c);
	checkBodyUsage(cd);
	String cname = cd.getExternalName();
	Element elt = nsElement(cname);
	for (Iterator i = cd.getFieldDescrs().iterator(); i.hasNext();) {
	    FieldDescr fd = (FieldDescr)i.next();
	    visitField(obj, fd, elt);
	}
	return elt;
    }

    protected FieldDescr checkBodyUsage(ClassDescr cd) {
	boolean needNonBodyElements = false;
	FieldDescr bodyField = null;
	for (FieldDescr fd: cd.getFieldDescrs()) {
	    if (isAttributeClass(fd.getType()))
		continue;
	    else if (isBodyField(fd)) {
		if (needNonBodyElements)
		    throw new XMLException
			(cd.getName() + " needs elements for non-body fields");
		else if (bodyField == null)
		    bodyField = fd;
		else
		    throw new XMLException
			(cd.getName() + " has at least two body fields, " +
			 bodyField.getName() + " and " + fd.getName());
	    }
	    else {
		if (bodyField != null)
		    throw new XMLException
			(cd.getName() + " needs elements for non-body fields");
		else
		    needNonBodyElements = true;
	    }
	}
        if (bodyField == null)
            return null;
	ClassDescr valType = bodyField.getTypeDescr();
	if (!valType.isCollection())
	    throw new XMLException
		("Body field " + bodyField.getName() + " in " +
		 cd.getName() + " is not a collection.");
	return bodyField;
    }

    // /\/: Should we always use external names in exception messages?
    // In checkBodyUsage, we use only Java names.

    protected boolean isBodyField(FieldDescr fd) {
	return fd.getAnnotation(Body.class) != null;
    }

    protected boolean fieldElementSuffices(FieldDescr fd) {
	ClassDescr fieldType = fd.getTypeDescr();
	return fieldType.isFinal()
	    || (fieldType.isCollection()
                && fieldType.getEltType() != null);
    }

    @Override    
    protected void visitField(Object obj, FieldDescr fd, Element objElt) {
	Object value = getFieldValue(obj, fd);
	if (value == null) 
	    return;
	String fieldName = fd.getExternalName();
	Class valClass = fd.getType();
	if (isAttributeClass(valClass)) {
	    // Add attribute
	    // /\/: Need to check that value is a direct
	    // instance of the field class?  And make it
	    // an element if it isn't?
	    Stringer s = classSyntax.getStringer(valClass);
	    String text = s != null ? s.toString(value) : value.toString();
	    objElt.setAttribute(fieldName, text);
	}
	else if (isBodyField(fd)) {
	    for (Object item: ((Collection)value)) {
		objElt.addContent(objectToElement(item));
	    }
	}
	else {
	    // Add child element
            Element valueElt = objectToElement(value);
	    // Do we need both field and value elts?
            if (fieldElementSuffices(fd)) {
		// No -- turn value elt into field elt.
                valueElt.setName(fieldName);
                objElt.addContent(valueElt);
            }
            else {
		// Yes -- put value elt inside a field elt.
                Element fieldElt = nsElement(fieldName);
                objElt.addContent(fieldElt);
                fieldElt.addContent(valueElt);
            }
	}
    }

    /* ************************** *
     *      Objects from XML      *
     * ************************** */

    @Override
    protected Object structFromElement(Class c, Element elt) {
	// /\/: Should check for fields specified > once.
	// /\/: Should check namespace of field names.
	Object obj = makeInstance(c);
	ClassDescr cd = classSyntax.getClassDescr(c);
	// Process attributes
        processStructAttributes(obj, cd, elt);
	// See if there's a body field.
	FieldDescr bodyField = checkBodyUsage(cd);
	if (bodyField != null) {
	    ClassDescr bodyType = bodyField.getTypeDescr();
	    Class implDefault = bodyType.isList()
		? defaultListClass
		: defaultSetClass;
	    Class implClass = getImplClass(null, bodyType.theClass,
					   implDefault);
	    Collection fieldVal = collectionFromElement(implClass, elt);
	    setFieldValue(obj, bodyField, fieldVal);
	    return obj;
	}
	// Process subelements
        processStructSubelements(obj, cd, elt);
	return obj;
    }

    @Override
    protected void processStructSubelements(Object obj, ClassDescr cd,
                                            Element elt) {
	for (Iterator i = elt.getChildren().iterator(); i.hasNext();) {
	    Element fieldElt = (Element)i.next();
	    lastExaminedElement = fieldElt;
	    // Get our description of the field.
	    String fieldName = fieldElt.getName(); // external name
	    FieldDescr fd = cd.fieldForExternalName(fieldName);
	    if (fd == null) {
		handleUnknownFieldFromElement(obj, fieldName, fieldElt);
		continue;
	    }
            // Get the value to put in the field.
            Object fieldVal = null;
            if (fieldElementSuffices(fd)) {
                // Get the value from the field elt which, btw,
                // may have an implClass attribute.
                fieldVal = objectFromElement(fd.getType(), fieldElt);
            }
            else {
                // Get the value from the value elt that's the only
                // child of the field elt.  The val elt may have an
                // implClass attribute.
                requireEltSize(1, fieldElt);
                fieldVal = getFieldValue(fieldElt, fd);
            }
	    // Install the field value.
	    setFieldValue(obj, fd, fieldVal);
	}
    }

}
