/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Tue Aug 19 19:29:48 2008 by Jeff Dalton
 * Copyright: (c) 2002 - 2008, AIAI, University of Edinburgh
 */

package ix.util.xml;

import java.io.PrintStream;
import java.util.*;

import org.jdom.*;
import org.jdom.output.XMLOutputter;
import org.jdom.output.Format;

import ix.icore.domain.Constraint;
import ix.util.*;
import ix.util.lisp.*;
import ix.util.reflect.*;

/**
 * Describes the (XML) syntax of data objects in a BNF-like style.
 */
public class BNFSyntax extends XMLSyntax {

    protected XMLTranslator noNamespaceXmlt = XML.config().makeXMLTranslator();
    { noNamespaceXmlt.homeNamespace = null; }

    public BNFSyntax() {
	super();
    }

    public BNFSyntax(XMLTranslator xmlt) {
	super(xmlt);
    }

    /**
     * Prints the syntax for the indicated class and for any related
     * classes that can be found.  Related classes are the types of
     * fields, subclasses if known, and some special cases.  This
     * method is the main entry point when a BNFSyntax is used to
     * produce syntax descriptions.
     */
    public void describeClass(String className, PrintStream out) {
	Class c = classSyntax.classForExternalName(className);
	if (c == null) {
	    out.println("Can't find class named " + className);
	    return;
	}
	out.println("Syntax for " + className + " and related classes:");
	out.println("");

	List relevantClasses = relevantClasses(c);
	inheritance = new InheritanceTree(relevantClasses);
	printRuleSyntax(new RuleList(relevantClasses), out);
    }

    /** Prints the the BNF that corresponds to each rule. */
    void printRuleSyntax(RuleList list, PrintStream out) {
	for (Iterator i = list.getRules().iterator(); i.hasNext();) {
	    Rule rule = (Rule)i.next();
	    printRuleSyntax(rule, out);
	    out.println("");
	}
    }

    /** Prints the BNF that corresponds to an individual rule. */
    void printRuleSyntax(Rule rule, PrintStream out) {
	out.print(rule.nonterminal + " ::=");
	if (rule.rhs instanceof Alternatives)
	    printAlternatives((Alternatives)rule.rhs, out);
	else
	    printInstance((Instance)rule.rhs, out);
    }

    /** Prints the BNF for a RHS that is an Alternatives. */
    void printAlternatives(Alternatives alts, PrintStream out) {
	int i = 1;
	for (Iterator ai=alts.alts.iterator(); ai.hasNext(); i++) {
	    Instance inst = (Instance)ai.next();
	    String prefix = (i == 1) ? " " : " | ";
	    out.print(prefix);
	    out.print(inst.asValue());
	}
	out.println("");
    }

    /** Prints the BNF for a RHS that is an Instance -- a description
     * of an instance of a class. */
    void printInstance(Instance inst, PrintStream out) {
	out.println("");
	printIndented(out, 3, inst.asElement());
    }

    /** Utility that breaks text into lines and prints them with
     * the specified indentation. */
    void printIndented(PrintStream out, int indent, String text) {
	List lines = Strings.breakIntoLines(text);
	for (Iterator i = lines.iterator(); i.hasNext();) {
	    String line = (String)i.next();
	    out.print(Strings.repeat(indent, " "));
	    out.println(line);
	}
    }

    /**
     * Makes a description of an instance of a class.
     */
    Instance makeClassInstance(ClassDescr cd) {
	if (cd.isPrimitive()) {
	    return new SimpleInstance(cd);
	}
	else if (cd.isStruct()) {
	    return new StructInstance(cd);
	}
	else if (cd.isList())
	    return new ListInstance(cd);
	else if (cd.isSet())
	    return new SetInstance(cd);
	else if (cd.isMap())
	    return new MapInstance(cd);
	else if (cd.isInterface())
	    return new SimpleInstance(cd); // /\/ ? is that enough ?
	else
	    throw new ConsistencyException("Can't handle syntax of " + cd);
    }

    /**
     * Represents the grammar that is being produced.
     */
    class RuleList {
	List<Rule> rules = new LinkedList<Rule>();
	boolean hasMapEntryRule = false;
	RuleList(List classes) {
	    for (Iterator i = classes.iterator(); i.hasNext();) {
		Class c = (Class)i.next();
		ClassDescr cd = getClassDescr(c);
		if (cd.isEnumeration() || cd.isStruct())
		    addRule(cd);
		if (cd.theClass == Constraint.class)
		    addConstraintRules();
		if (needMapEntryRule(cd))
		    addMapEntryRule();
	    }
	}
	List<Rule> getRules() {
	    return rules;
	}
	void addRule(Rule r) {
	    rules.add(r);
	}
	void addRule(ClassDescr cd) {
	    if (!cd.isAbstract())
		addRule(new Rule(cd));
	    addInheritanceRule(cd);
	}
	void addInheritanceRule(ClassDescr cd) {
	    List subclasses = inheritance.getSubclasses(cd.theClass);
	    if (subclasses == null || subclasses.isEmpty())
		return;
	    List<Instance> instances = new LinkedList<Instance>();
	    for (Iterator si = subclasses.iterator(); si.hasNext();) {
		Class subclass = (Class)si.next();
		ClassDescr sd = getClassDescr(subclass);
		instances.add(new Nonterminal(sd));
	    }
	    addRule(new Rule(getNTName(cd), 
			     new Alternatives(instances)));
	}
	void addConstraintRules() {
	    String constraintNT = getNTName(Constraint.class);
	    List constraints = getConstraintSyntaxList();
	    if (constraints.isEmpty())
		return;
	    Nonterminal k = new Nonterminal("KNOWN-CONSTRAINT");
	    addRule(new Rule(constraintNT,
			     new Alternatives(k)));
	    for (Iterator i = constraints.iterator(); i.hasNext();) {
		Constraint c = (Constraint)i.next();
		addRule(new Rule("KNOWN-CONSTRAINT",
				 new ConstraintTemplate(c)));
	    }
	}
	boolean needMapEntryRule(ClassDescr cd) {
	    if (hasMapEntryRule)
		return false;
	    else if (cd.isMap())
		return true;
	    else if (cd.isStruct()) {
		List fields = cd.getFieldDescrs();
		for (Iterator fi = fields.iterator(); fi.hasNext();) {
		    FieldDescr fd = (FieldDescr)fi.next();
		    if (fd.getTypeDescr().isMap())
			return true;
		}
		return false;
	    }
	    else
		return false;
	}
	void addMapEntryRule() {
	    addRule(new Rule("MAP-ENTRY", new MapEntryInstance()));
	    hasMapEntryRule = true;
	}
    }

    /**
     * The right-hand side of a syntax rule.  Note that an RHS is not
     * always the entire right-hand side of a rule.  Sometimes an RHS
     * will appear inside a larger RHS instead.  In an {@link Alternatives},
     * for example.
     */
    abstract class RHS { }

    /**
     * A syntax rule.  The right-hand side is an instance of a RHS subclass;
     * the left-hand side is just (the name of) a nonterminal.
     */
    class Rule {
	String nonterminal;
	RHS rhs;
	Rule(String nt, RHS rhs) {
	    this.nonterminal = nt;
	    this.rhs = rhs;
	}
	Rule(ClassDescr cd) {
	    nonterminal = getNTName(cd);
	    rhs = makeRHS(cd);
	}
	RHS makeRHS(ClassDescr cd) {
	    if (cd.isEnumeration()) {
		List values = getEnumerationValues(cd.theClass);
		List<Instance> instances = new LinkedList<Instance>();
		for (Iterator i = values.iterator(); i.hasNext();) {
		    instances.add(new Literal(i.next().toString()));
		}
		return new Alternatives(instances);
	    }
	    else
		return makeClassInstance(cd);
	}
    }

    /**
     * A RHS that represents a disjunction of {@link Instance}s.
     * The instances should be ones that print as the name of a nonterminal
     * or as a simple literal value, when used as values.
     */
    class Alternatives extends RHS {
	private List<Instance> alts;
	Alternatives(Instance inst) {
	    alts = Collections.singletonList(inst);
	}
	Alternatives(List<Instance> alts) {
	    this.alts = alts;
	}
	List<Instance> getAlts() {
	    return alts;
	}
    }

    /**
     * A RHS that describes an instance of a class.
     *
     * <p>An instance can be used simply to represent a value.  In such
     * cases, it will be printed as the name of a nonterminal or as a
     * simple literal value.  For example "STRING" could be printed
     * to indicate that a string could appear in place of "STRING".</p>
     *
     * <p>Some instances can also be printed as a description of
     * an XML element.
     */
    abstract class Instance extends RHS {
	String asValue() {
	    throw new ConsistencyException
		("Can't express " + this + " as a value");
	}
	String asElement() {
	    throw new ConsistencyException
		("Can't express " + this + " as an element");
	}
    }

    /**
     * An Instance used when only the name of a nonterminal should appear.
     * A Nonterminal can be used only as a value, not as an element.
     */
    class Nonterminal extends Instance {
	// A ref to an instance of an nonterminal.
	String name;
	Nonterminal(ClassDescr cd) {
	    this(getNTName(cd));
	}
	Nonterminal(String name) {
	    this.name = name;
	}
	String asValue() {
	    return name;
	}
    }

    /**
     * An instance used when only a (simple) literal value shoud appear.
     * A Literal can be used only as a value, not as an element.
     */
    class Literal extends Instance {
	String value;
	Literal(String value) {
	    this.value = value;
	}
	String asValue() {
	    return value;
	}
    }

    /**
     * Represents an instance of a primitve or atomic class.
     *
     * <p>A SimpleInstance can be used both as a value and as an element.
     * As a value, it is the name of a nonterminal.  As an element type
     * T, it typically looks like this &lt;t&gt;T&lt;/t&gt;.</p>
     */
    class SimpleInstance extends Instance {
	ClassDescr cd;
	SimpleInstance(ClassDescr cd) {
	    this.cd = cd;
	}
	String asValue() {
	    return getNTName(cd);
	}
	String asElement() {
	    return tagged(getElementName(cd), getNTName(cd));
	}
    }

    /**
     * Represents an instance of a structure class.  Most such classes
     * have visible fields that will appear in XML as attributes or
     * elements.  When used as a value, a StructInstance appears
     * as the name of the nonterminal that corresponds to the class.
     * When used as a value, it appears as a description of an XML
     * element that includes field attributes and subelements.
     */
    class StructInstance extends Instance {
	ClassDescr cd;
	StructInstance(ClassDescr cd) {
	    this.cd = cd;
	}
	String asValue() {
	    return getNTName(cd);
	}
	String asElement() {
	    List fields = cd.getFieldDescrs();
	    List attrFields = attributeFields(fields);
	    List eltFields = elementFields(fields);
	    Debug.expect(fields.size() == attrFields.size() 
			                    + eltFields.size());
	    List lines = new LinkedList();
	    // Opening tag
	    lines.add("<" + getElementName(cd));
	    // Attributes
	    for (Iterator ai = attrFields.iterator(); ai.hasNext();) {
		FieldDescr fd = (FieldDescr)ai.next();
		String f = "      " + new InstanceField(fd).description();
		lines.add(f);
	    }
	    lines.add(lines.remove(lines.size()-1) + ">");
	    // Subelements
	    for (Iterator ei = eltFields.iterator(); ei.hasNext();) {
		FieldDescr fd = (FieldDescr)ei.next();
		String f = "   " + new InstanceField(fd).description();
		lines.add(f);
	    }
	    // Closing tag
	    lines.add("</" + getElementName(cd) + ">");
	    return Strings.joinLines(lines);
	}
    }

    /** Represents a field of a structure class. */
    class InstanceField {
	FieldDescr fd;
	boolean isAttribute;
	InstanceField(FieldDescr fd) {
	    this.fd = fd;
	    this.isAttribute = xmlt.isAttributeClass(fd.getType());
	}
	String description() {
	    ClassDescr valFD = fd.getTypeDescr();
	    Instance valInst = makeClassInstance(valFD);
	    String eltName = getElementName(fd);
	    String valDescr = valInst.asValue();
	    if (isAttribute)
		return fd.getExternalName() + "=" + Strings.quote(valDescr);
	    // A field represented as an element
	    String oneLine = tagged(eltName, valDescr);
	    if (oneLine.length() <= 75)
		return oneLine;
	    // Need to break it over several lines
	    List lines = new LinkedList();
	    lines.add("<" + eltName + ">");
	    lines.add("      " + valDescr);
	    lines.add("   </" + eltName + ">");
	    return Strings.joinLines(lines);
	}
    }

    /**
     * An instance that's already XML in the form of a JDOM Element.
     * It can be used only as an element, not as a value.
     *
     * <p>TemplateInstances allow us to have grammar rules that aren't
     * just derived from class definitions.</p>
     */
    abstract class TemplateInstance extends Instance {
	Element template;
	TemplateInstance() {
	}
	TemplateInstance(Element template) {
	    this.template = template;
	}
	String asElement() {
	    XMLOutputter out = XML.makePrettyXMLOutputter();
	    Format format = out.getFormat(); // returns a copy
	    format.setOmitDeclaration(true);
	    format.setIndent("   ");
	    out.setFormat(format);
	    return out.outputString(template);
	}
    }

    /**
     * A TemplateInstance based on a {@link Constraint}.
     * The constraint's parameters are interpreted as descriptions
     * of the values that belong there.  A parameter in such a
     * template constraint is either in instance of the appropriate
     * class or a class literal.
     */
    class ConstraintTemplate extends TemplateInstance {
	ConstraintTemplate(Constraint c) {
	    super();
	    // N.B. Can't call a method of this instance in a constructor.
	    this.template = constraintTemplate(c);
	}
	Element constraintTemplate(Constraint c) {
	    ClassDescr con_cd = getClassDescr(Constraint.class);
	    FieldDescr par_fd = con_cd.fieldForName("parameters");
	    FieldDescr ann_fd = con_cd.fieldForName("annotations");
	    String parametersName = getElementName(par_fd);
	    String annotationsName = getElementName(ann_fd);
	    String constraintName = getElementName(Constraint.class);
	    String listName = getElementName(List.class);
	    String mapName = getElementName(Map.class);
	    List parameters = new LinkedList();
	    for (Iterator i = c.getParameters().iterator(); i.hasNext();) {
		Object parameter = i.next();
		Class pc = parameter instanceof Class ? (Class)parameter
		             : parameter.getClass();
		ClassDescr pcd = getClassDescr(pc);
		parameters.add(new Text(getNTName(pcd)));
		if (i.hasNext())
		    parameters.add(new Text(" "));
	    }
	    Element template =
	      new Element(constraintName)
		.setAttribute("type", c.getType().toString())
		.addContent
		    (new Element(parametersName)
		         .addContent
		             (new Element(listName)
			          .setContent(parameters)));
	    if (c.getRelation() != null) {
		template.setAttribute("relation", c.getRelation().toString());
	    }
	    // World-state constraints sometimes have condition-type
	    // annotations that are significant.  /\/
	    if (c.getAnnotations() != null) {
		Element ann = 
		    noNamespaceXmlt.objectToElement(c.getAnnotations());
		template
		    .addContent(new Element(annotationsName)
				    .addContent(ann));
	    }
	    return template;
	}
    }

    /**
     * Represents an instance of a collection class.  For some purposes
     * here, collections include maps.  CollectionInstances are printed
     * as XML element descriptions even when they appear as values.
     */
    abstract class CollectionInstance extends Instance {
	ClassDescr cd;
	CollectionInstance(ClassDescr cd) {
	    this.cd = cd;
	}
	String asValue() {
	    return asElement();
	}
	String asElement() {
	    Class tagClass = 
		cd.isList() ? List.class
		: cd.isSet() ? Set.class
		: null;
	    if (tagClass == null)
		throw new ConsistencyException
		              ("Unknown collection type " + cd);
	    ClassDescr eltType = cd.getEltType();
	    return tagged(getElementName(tagClass),
			  eltType == null
			  ? "..."
			  : getNTName(eltType) + "...");
	}
    }

    class ListInstance extends CollectionInstance {
	ListInstance(ClassDescr cd) {
	    super(cd);
	}
    }

    class SetInstance extends CollectionInstance {
	SetInstance(ClassDescr cd) {
	    super(cd);
	}
    }

    class MapInstance extends CollectionInstance {
	MapInstance(ClassDescr cd) {
	    super(cd);
	}
	String asElement() {
	    return tagged(getElementName(Map.class),
			  "MAP-ENTRY...");
	}
    }

    class MapEntryInstance extends TemplateInstance {
	MapEntryInstance() {
	    super();
	    // N.B. Can't call a method of this instance in a constructor.
	    this.template = mapEntryTemplate();
	}
	Element mapEntryTemplate() {
	    String objectNT = getNTName(Object.class);
	    return new Element("map-entry")
		.addContent(new Element("key")
			        .addContent(new Text(objectNT)))
		.addContent(new Element("value")
			        .addContent(new Text(objectNT)));
	}
    }

    /** Returns a string representing an XML element. */
    String tagged(String tag, String contents) {
	return "<" + tag + ">" + contents + "</" + tag + ">";
    }


    /**
     * Test loop that repeatedly asks the user for a class name
     * and prints a description of the syntax for objects of
     * that class.
     */
    public static void main(String[] argv) {
	BNFSyntax syntax = argv.length > 0 && argv[0].equals("java")
	    ? new BNFSyntax	// Java syntax
	          (new XMLTranslator
		       (new ClassSyntax
			    (new ClassFinder
			         (XML.config().defaultClassFinder()))))
	    : new BNFSyntax();
	// Try preloading the class-finder.
	XML.config().defaultClassFinder()
	    .preLoad(new XMLSyntax().relevantClasses(Object.class));
	for (;;) {
	    String in = Util.askLine("Class name:");
	    if (in.equals("bye"))
		return;
	    syntax.describeClass(in, System.out);
	    System.out.println("");
	}
    }

}

// Issues:
// * Should specify the PrintStream when constructing rather than
//   passing it from method to method?
