/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Mon Aug 18 15:20:25 2008 by Jeff Dalton
 * Copyright: (c) 2003, 2008, AIAI, University of Edinburgh
 */

package ix.util.reflect;

import java.lang.reflect.*;
import java.lang.annotation.*;
import java.util.*;

import ix.util.*;

/**
 * A view or description of a field.
 *
 * <p>N.B. The constructors are deliberately not public so that something
 * wanting a FieldDescr must go through a {@link ClassSyntax} object.
 */
public class FieldDescr {

    protected String name;		// Java name
    protected String externalName;
    protected Class type;
    protected Type genericType;
    protected ClassDescr typeDescr;
    protected Annotation[] annotations;
    protected Method getter;
    protected Method setter;

    FieldDescr(ClassSyntax syntax, Field f) {
	this.name = f.getName();
	this.externalName = syntax.externalNameForField(f.getName());
	this.type = f.getType();
	this.genericType = f.getGenericType();
	this.annotations = f.getAnnotations();
	// this.typeDescr is filled-in later and may not be the
	// ClassDescr of type.  For example, it might be known
	// that a List is a list of Issues, and the typeDescr
	// would reflect that.
    }

    public String getName() {
	return name;
    }

    public String getExternalName() {
	return externalName;
    }

    public void renameTo(String javaName, String externalName) {
	this.name = javaName;
	this.externalName = externalName;
    }

    public Class getType() {
	return type;
    }

    public Class determineElementType() {
	if (! (genericType instanceof ParameterizedType))
	    return null;
	ParameterizedType pType = (ParameterizedType)genericType;
	Type[] args = pType.getActualTypeArguments();
	if (args.length == 1 && args[0] instanceof Class)
	    return (Class)args[0];
	else
	    return null;
    }

    public ClassDescr getTypeDescr() {
	return typeDescr;
    }

    public Object getValue(Object obj)
	throws IllegalAccessException,
	       IllegalArgumentException,
	       InvocationTargetException {
	return getter.invoke(obj, new Object[0]);
    }

    public void setValue(Object obj, Object newVal)
	throws IllegalAccessException,
	       IllegalArgumentException,
	       InvocationTargetException {
	setter.invoke(obj, new Object[] { newVal });
    }

    public void checkValue(Object val) {
        // System.out.println(name + " ----> " + genericType);
        if (!typeDescr.valueIsOk(val))
            throw new ClassCastException
                (Util.aClass(val.getClass()) + " cannot be used in " +
                 genericType + " field " + name);
    }

    public <A extends Annotation> A getAnnotation(Class<A> anClass) {
	if (annotations != null)
	    for (Annotation an: annotations)
		if (anClass.isInstance(an))
		    return (A)an;
	return null;
    }

    public String toString() {
	return "FieldDescr[" + type.getName() + " " + name + "]";
    }

}

// Issues:
// * getType() returns a Class, so that getTypeDescr() has to be
//   called to get the ClassDescr, but the ClassDescr get...Type()
//   methods such as getEltType() all return ClassDescrs.
