/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Thu Sep  8 18:02:57 2005 by Jeff Dalton
 * Copyright: (c) 2002, 2003, 2005, AIAI, University of Edinburgh
 */

package ix.util.reflect;

import java.util.*;

import ix.util.*;

/**
 * <p>Converts between class names (Strings) and classes.  Note that a
 * single ClassFinder maps in both directions to make it easier to
 * create mappings that are consistent.</p>
 *
 * <p>Explicit mappings and "imports" may be specified; otherwise
 * fully qualified names must be used.</p>
 *
 * <p>A ClassFinder may use a different naming convention than Java
 * provided that there are String-to-String mappings in both directions
 * between any special "external" names and ordinary Java names.</p>
 *
 * <p><b>This class does not yet handle array classes.</b></p>
 */

public class ClassFinder {

    protected List imports = new LinkedList();
    protected Map nameToClass = new HashMap();
    protected Map classToName = new HashMap();
    protected Map nameToClassCache = new HashMap();
    protected Map classToNameCache = new HashMap();

    public ClassFinder() {
	addInitialNames();
	addInitialImports();
    }

    public ClassFinder(ClassFinder base) {
	addInitialNames();	// take from base? /\/
	// /\/: N.B. same, modifiable List as the base.
	this.imports = base.imports;
    }

    /**
     * Adds the initial set of name mappings.
     */
    protected void addInitialNames() {
	// Explicitly specify the mapping for primitive types,
	// because Class.forName(String) will not work.
//  	addName("byte",    Byte.TYPE);
//  	addName("char",    Character.TYPE);
//  	addName("short",   Short.TYPE);
//  	addName("int",     Integer.TYPE);
//  	addName("long",    Long.TYPE);
//  	addName("float",   Float.TYPE);
//  	addName("double",  Double.TYPE);
//  	addName("boolean", Boolean.TYPE);
//  	addName("void",    Void.TYPE);
    }

    /**
     * Adds the initial set of imports.  These are:
     * <pre>
     *   java.lang.*
     *   java.util.*
     * </pre>
     */
    protected void addInitialImports() {
	// Standard imports
	addImport("java.lang.*");
	addImport("java.util.*");
    }

    /**
     * Records an import that allows a class name or names to be used without
     * package-qualification.  The import is specified as either a fully
     * qualified Java class name or a package name suffixed by ".*", as in
     * the Java <code>import</code> statement.
     */
    public void addImport(String name) {
	// /\/: Perhaps we shouldn't allow names to be given more than once,
	// but sometimes I-DE wants to add one that the XMLConfig already has.
	if (!haveImport(name))
	    imports.add(new Import(name));
	nameToClassCache.clear();         // Caches may no longer be valid
	classToNameCache.clear();
    }

    private boolean haveImport(String name) {
	for (Iterator i = imports.iterator(); i.hasNext();) {
	    Import imp = (Import)i.next();
	    if (imp.name.equals(name))
		return true;
	}
	return false;
    }

    /**
     * Adds an explicit mapping in both directions between a name
     * and a class.  This takes precedence over all other ways of
     * determining those relationships.
     *
     * @param name  an external name for a class.
     * @param c     a class
     */
    public void addName(String name, Class c) {
	nameToClass.put(name, c);
	classToName.put(c, name);
	// Caches may no longer be valid.  Used to do:
	//   nameToClassCache.clear();
	//   classToNameCache.clear();
	// We could just call cacheBothWays here ...  /\/
	nameToClassCache.remove(name);
	classToNameCache.remove(c);
    }

    /**
     * Caches the mapping from name to class and from class to name.
     * This method must be used with care, because it's possible
     * for more than one name to map to the same class: for example,
     * a fully-qualified name, and a short, packageless name.
     * There may even be more than one short name that works.
     * A ClassFinder that has lower-case (e.g. "dash-syntax") external
     * names may convert to Java names by capitalizing some letters,
     * but leaving any existing capitals in place.  It might thus
     * accept both "Issue" and "issue".
     */
    protected void cacheBothWays(String name, Class c) {
	// Debug.noteln("Cache both ways: " + name + " <--> " + c);
	nameToClassCache.put(name, c);
	classToNameCache.put(c, name);
    }

    public void preLoad(List expectedClasses) {
	for (Iterator i = expectedClasses.iterator(); i.hasNext();) {
	    Class c = (Class)i.next();
	    nameForClass(c);	// will cache both ways
	}
    }

    /**
     * Returns the Java name that corresponds to an external class name.
     * The method in the ClassFinder class returns the name unchanged, but
     * it may be overridden in subclasses.
     */
    public String javaName(String externalName) {
	return externalName;
    }

    /**
     * Returns the Java name that corresponds to an external field name.
     * The method in the ClassFinder class just calls the method
     * {@link #javaName(String)}, but it may be overridden in subclasses
     * in which the conversion for class names cannot be used for fields.
     */
    public String javaFieldName(String externalName) {
	return javaName(externalName);
    }

    /**
     * Returns the external name that corresponds to a Java class name.
     * The method in the ClassFinder class returns the name unchanged, but
     * it may be overridden in subclasses.
     */
    public String externalName(String javaName) {
	return javaName;
    }

    /**
     * Returns the external name that corresponds to a Java field name.
     * The method in this class calls {@link #externalName(String)},
     * but it may be overridden in subclasses in which the conversion
     * for class names cannot be used for fields.
     */
    public String externalFieldName(String javaName) {
	return externalName(javaName);
    }

    /**
     * Returns the class that corresponds to the name, as determined
     * by any imports or explicit mappings.  Fully qualified names
     * may also be used.  The mapping from name to class is cached,
     * but not the mapping the other way, because more than one
     * name might map to the same class.
     * (See {@link #cacheBothWays(String, Class)} for an explanation.)
     *
     * @param name  an external name for a class.
     * @return a Class or null.
     */
    public Class classForName(String name) {

	// See if the external name is already cached.
	Class c = (Class)nameToClassCache.get(name); // cached?
	if (c != null)
	    return c;

	// See if the mapping was explicitly given
	c = (Class)nameToClass.get(name);  	// explicitly specified?
	if (c != null) {
	    nameToClassCache.put(name, c);	// yes - cache
	    return c;				// and return
	}

	// Try to find the class from the Java name.
	String javaName = javaName(name);
	boolean hasPackage = javaName.indexOf(".") >= 0;
	Debug.expect(c == null);
	if (hasPackage)				// if a package is specified,
	    c = tryClassForName(javaName); 	//   try as-is
	else					// else
	    c = findImportClass(javaName); 	//   try the imports

	if (c != null)
	    nameToClassCache.put(name, c);	// cache if found

	return c;
    }

    /**
     * Tries to find a class by asking each import in turn.
     *
     * @param name  a Java name for a class.
     * @return a class or null.
     * @throws RuntimeException if more than one import can find a class.
     */
    protected Class findImportClass(String name) {
	Class result = null;
	Import source = null;
	for (Iterator i = imports.iterator(); i.hasNext();) {
	    Import imp = (Import)i.next();
	    Class c = imp.tryName(name);
	    if (c != null) {
		if (result == null) {
		    result = c;
		    source = imp;
		}
		else {
		    throw new RuntimeException
			("Ambiguous name " + Util.quote(name) +
			 " imported via both " + source + " and " + imp);
		}
	    }
	}
	return result;
    }

    /**
     * Like <code>Class.forName(String)</code> but returns null
     * if the class cannot be found.
     *
     * @param name  a Java name for a class.
     * @return a class or null.
     */
    public static Class tryClassForName(String name) {
	// Debug.noteln("tryClassForName", name);
	try {
	    return Class.forName(name);
	}
	catch (ClassNotFoundException e) {
	    return null;
	}
	catch (SecurityException e) {
	    if (Parameters.isApplet()) {
		Debug.noteException(e, false);
		if (e.getMessage().startsWith("Prohibited package name"))
		    return null;
	    }
	    throw e;
	}
	catch (ClassFormatError e) {
	    if (Parameters.isApplet()) {
		// /\/: Happens in applets.
		Debug.noteException(e, false);
		return null;
	    }
	    throw e;
	}
    }

    /**
     * Returns the best name that this ClassFinder would map to the
     * specified class, where a plain class name counts as better than
     * a package-qualified one.  Since the result is the best name
     * and hence we'd return it again if called again, we cache it as
     * the name for this class; and since we know it's a name for the
     * class, we can cache in the name-to-class direction as well.
     *
     * @return an external name for the class.
     *
     * @see #cacheBothWays(String, Class)
     */
    public String nameForClass(Class c) {
	// First see if cached.
	String name = (String)classToNameCache.get(c);
	if (name != null)
	    return name;

	// See if the mapping has been explicitly specified
	name = (String)classToName.get(c);
	if (name != null) {
	    Debug.expect(nameToClass.get(name) == c);
	    classToNameCache.put(c, name);	 // cache the result
	    return name;
	}

	// Otherwise, determine whether we need the fully qualified name
	// or can give a simple name because of imports.  /\/: Note that
	// a primitive class such as Integer.TYPE will have a name such
	// as "int" that is not prefixed by a package name, so that the
	// full and short names will be the same.  However, it may not
	// work to use that name to look up the class.  For example,
	// classForName("long") may return Long.class, not Long.TYPE
	// (depending on the imports etc).  So that is a case where
	// classForName(nameForClass(c)) == c may be false.  Is it the
	// only case?  /\/
	String fullName = c.getName();
	if (fullName.startsWith("[")) {
	    throw new RuntimeException("Can't handle array class" + fullName);
	}
	String shortName = Strings.afterLast(".", fullName);
	String shortExternal = externalName(shortName);
	// See if the short external name is mapped to the desired class.
	if (classHasShortName(c, shortExternal)) {
	    // Yes -- the short name can be returned,
	    name = shortExternal;
	}
	else {
	    // No -- must return the full name
	    name = externalName(fullName);
	    // Check that the full name will work.
	    Debug.expect(classForName(name) == c || c.isPrimitive(),
			 "Can't get back class when using name", name);
	}
	cacheBothWays(name, c);	 // cache the result
	return name;
    }

    protected boolean classHasShortName(Class c, String shortExternal) {
	// This could just do: return classForName(shortExternal) == c;
	// But that's too inefficient if the classes are "distant",
	// for instance when loading classes for an applet.

	// First try the name-to-class cache and the explicit map.
	Class cached = (Class)nameToClassCache.get(shortExternal);
	if (cached != null)
	    return c == cached;
	Class given = (Class)nameToClass.get(shortExternal);
	if (given != null)
	    return c == given;

	// Try the imports, but only the ones for the right package.
	String pack = Strings.beforeLast(".", c.getName());
	String classOnly = Strings.afterLast(".", c.getName());
	String javaName = javaName(shortExternal);
	if (!javaName.equals(classOnly) && !c.isPrimitive())
	    // The classOnly for int will be int, but the javaName is Int.
	    // int can appear as a field type.
	    throw new ConsistencyException
		("External name maps to " + javaName +
		 " but the actual class name is " + classOnly);
	for (Iterator i = imports.iterator(); i.hasNext();) {
	    Import imp = (Import)i.next();
	    if (imp.packageName().equals(pack)) {
		Class imported = imp.tryName(javaName);
		if (imported == c)
		    return true;
		else if (imported == null)
		    continue;
		else
		    throw new ConsistencyException
			(imp + " found " + imported +
			 " instead of " + c);
	    }
	}
	return false;
    }

    /**
     * Internal class that performs the name-to-class mapping that
     * corresponds to a single import specification.
     */
    protected static class Import {
	boolean star;
	String name;
	String prefix;

	public Import(String name) {
	    this.name = name;
	    star = name.endsWith("*");
	    if (star) {
		prefix = name.substring(0, name.length() - 1);
		Debug.expect(prefix.endsWith("."), "invalid import", name);
	    }
	    else
		prefix = name;
	}

	public String packageName() {
	    Debug.expect(name.indexOf(".") > 0, "Bogus import name", name);
	    return Strings.beforeLast(".", name);
	}

	/**
	 * Tries to find a class for the name.
	 *
	 * @return a Class or null.
	 */
	public Class tryName(String name) {
	    if (star) 
		return tryClassForName(prefix + name);
	    else if (prefix.equals(name))
		return tryClassForName(prefix);	// in case name is fully qual
	    else if (prefix.endsWith("." + name))
		return tryClassForName(prefix); // explicit import
	    else
		return null;
	}

	public String toString() {
	    return "import[" + name + "]";
	}

    }

    /**
     * Simple main program for testing.  This method calls
     * {@link #do_main(ClassFinder)} with an instance of
     * this class as the finder.
     */
    public static void main(String[] argv) {
	do_main(new ClassFinder());
    }

    /**
     * A test loop that asks the user for a name and prints
     * the corresponding class (or else null) as determined
     * by the ClassFinder.  When the result is not null,
     * it also prints the ClassFinder's "best" name for that
     * class.
     */
    protected static void do_main(ClassFinder cf) {
	for (;;) {
	    String in = Util.askLine("Class name:");
	    if (in.equals("bye"))
		return;
	    Class c = cf.classForName(in);
	    System.out.println(c);
	    if (c != null)
		System.out.println("Best name is " + 
				   Util.quote(cf.nameForClass(c)));
	}
    }

}
