/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Sun Sep 28 14:16:04 2008 by Jeff Dalton
 * Copyright: (c) 2003, 2005, 2008, AIAI, University of Edinburgh
 */

package ix.util.reflect;

import java.lang.reflect.*;
import java.util.*;

import ix.util.*;

import ix.util.xml.LiteralDocument;

/**
 * A view or description of a class.
 *
 * <p>N.B. The constructors are deliberately not public so that something
 * wanting a ClassDescr must go through a {@link ClassSyntax} object.
 */
public class ClassDescr {

    protected ClassSyntax syntax;

    /** The class that this describes. */
    public final Class theClass;

    protected ClassDescr eltType = null;
    protected ClassDescr keyType = null;
    protected ClassDescr valType = null;

    protected int type;

    protected FieldMap fields;

    protected String externalName;
    protected String upperName;

    ClassDescr(ClassSyntax s, Class c) {
	this.syntax = s;
	this.theClass = c;
	init(c);
    }

    ClassDescr(ClassSyntax s,
	       Class collectionClass, Class eltClass) {
	this.syntax = s;
	this.theClass = collectionClass;
	this.eltType = syntax.getClassDescr(eltClass);
	init(collectionClass);
    }

    ClassDescr(ClassSyntax s, 
	       Class mapClass, Class keyClass, Class valClass) {
	this.syntax = s;
	this.theClass = mapClass;
	this.keyType = syntax.getClassDescr(keyClass);
	this.valType = syntax.getClassDescr(valClass);
	init(mapClass);
    }

    protected void init(Class c) {
	type = determineType(c);
	externalName = syntax.externalNameForClass(c);
	upperName = syntax.upperNameForClass(c);
    }


    /*
     * Class
     */

    /**
     * Returns the described class.
     */
    public Class getDescribedClass() {
	return theClass;
    }

    public ClassDescr getEltType() { return eltType; }
    public ClassDescr getKeyType() { return keyType; }
    public ClassDescr getValType() { return valType; }

    /**
     * Returns the superclass of the described class.  As the JDK
     * javadoc explains,
     * <blockquote>
     *   If this Class represents either the Object class, an interface,
     *   a primitive type, or void, then null is returned.  If [it]
     *   represents an array class then the Class object representing
     *   the Object class is returned.
     * </blockquote>
     */
    Class getSuperclass() {
	return theClass.getSuperclass();
    }

    public boolean valueIsOk(Object val) {
        return Fn.objectClass(theClass).isInstance(val)
            && (isCollection() && eltType != null
                ? elementsAreAllOk((Collection)val)
                : true);
    }

    private boolean elementsAreAllOk(Collection val) {
        Class eltClass = eltType.getDescribedClass();
        for (Object elt: val)
            if (!eltClass.isInstance(elt))
                return false;
        return true;
    }


    /*
     * Class type
     */

    /** Internal class type */
    protected static final int
	INTERFACE = 0,		// other than List, Set, Map, ...
	PRIMITIVE = 1,
	LIST = 2,
	SET = 3,
	MAP = 4,
	STRUCT = 5,
	XML = 6;		// embedded XML

    /** Internal type names */
    protected static final String[] typeName = {
	"INTERFACE",
	"PRIMITIVE",
	"LIST",
	"SET",
	"MAP",
	"STRUCT",
	"XML"
    };

    protected int determineType(Class c) {
	if (isPrimitive(c))
	    return PRIMITIVE;
	else if (List.class.isAssignableFrom(c))
	    return LIST;
	else if (Set.class.isAssignableFrom(c))
	    return SET;
	else if (Map.class.isAssignableFrom(c))
	    return MAP;
	else if (Collection.class.isAssignableFrom(c))
	    throw new ConsistencyException
		("Class of unknown collection type", c);
	else if (c.isInterface())
	    return INTERFACE;
	else if (LiteralDocument.class.isAssignableFrom(c))
	    return XML;
	else
	    return STRUCT;
    }

    public boolean isInterface() {
	// N.B. incluides List, Set, and Map
	return type == INTERFACE || theClass.isInterface();
    }

    public boolean isPrimitive() {
	return type == PRIMITIVE;
    }

    protected boolean isPrimitive(Class c) {
	// Used to determine the class type initially.
	// Thereafter the test is type == PRIMITIVE.
	// Incluides enumerations (EnumeratedValues),
	// because they're semi-primitives.
	// /\/: Ought to include Character.
	return c.isPrimitive()
	    || c == String.class
	    || Number.class.isAssignableFrom(c)
	    || SemiPrimitive.class.isAssignableFrom(c)
	    || Boolean.class.isAssignableFrom(c)
	    || syntax.getStringer(c) != null;
    }

    public boolean isFinal() {
	int modifiers = theClass.getModifiers();
	return Modifier.isFinal(modifiers);
    }

    public boolean isAbstract() {
	int modifiers = theClass.getModifiers();
	return Modifier.isAbstract(modifiers);
    }

    public boolean isEnumeration() {
	return EnumeratedValue.class.isAssignableFrom(theClass);
    }

    public boolean isCollection() {
	return type == LIST || type == SET
            || Collection.class.isAssignableFrom(theClass);
    }

    public boolean isList() {
	return type == LIST;
    }

    public boolean isTypedList() {
	return type == LIST && eltType != null;	// n.b. elt class might be Obj
    }

    public boolean isSet() {
	return type == SET;
    }

    public boolean isMap() {
	return type == MAP;
    }

    public boolean isStruct() {
	return type == STRUCT;
    }

    public boolean isXML() {
	return type == XML;
    }

    public void visitClass(ClassVisitor visitor) {
	switch(type) {
	case INTERFACE:   visitor.visitInterface(this);
	    break;
	case PRIMITIVE:   visitor.visitPrimitive(this);
	    break;
	case LIST:        visitor.visitList(this, eltType);
	    break;
	case SET:         visitor.visitSet(this, eltType);
	    break;
	case MAP:         visitor.visitMap(this, keyType, valType);
	    break;
	case STRUCT:      visitor.visitStruct(this);
	    break;
	case XML:         visitor.visitXML(this);
	    break;
	default:
	    throw new ConsistencyException
		("Unknown type for class " + theClass + type);
	}
    }

    public void visitObject(Object obj, ObjectVisitor visitor) {
	if (!theClass.isInstance(obj))
	    throw new IllegalArgumentException
		(obj + " is not an instance of " + this);
	switch(type) {
	case INTERFACE:
	    throw new ConsistencyException
		("Instance class appears to be interface " + this);
	case PRIMITIVE:   visitor.visitPrimitive(obj, this);
	    break;
	case LIST:        visitor.visitList((List)obj, this);
	    break;
	case SET:         visitor.visitSet((Set)obj, this);
	    break;
	case MAP:         visitor.visitMap((Map)obj, this);
	    break;
	case STRUCT:      visitor.visitStruct(obj, this);
	    break;
	case XML:         visitor.visitXML((LiteralDocument)obj, this);
	    break;
	default:
	    throw new ConsistencyException
		("Unknown type for class " + theClass + type);
	}
    }

    public Object mapObject(Object obj, ObjectMapper mapper) {
	if (!theClass.isInstance(obj))
	    throw new IllegalArgumentException
		(obj + " is not an instance of " + this);
	switch(type) {
	case INTERFACE:
	    throw new ConsistencyException
		("Instance class appears to be interface " + this);
	case PRIMITIVE:   return mapper.mapPrimitive(obj, this);
	case LIST:        return mapper.mapList((List)obj, this);
	case SET:         return mapper.mapSet((Set)obj, this);
	case MAP:         return mapper.mapMap((Map)obj, this);
	case STRUCT:      return mapper.mapStruct(obj, this);
	case XML:         return mapper.mapXML((LiteralDocument)obj, this);
	default:
	    throw new ConsistencyException
		("Unknown type for class " + theClass + type);
	}
    }


    /*
     * Class names
     */

    public String getName() {
	return theClass.getName();
    }

    public String getExternalName() {
	return externalName;
    }

    /** Returns an all-upper-case version of the external name. */
    public String getUpperName() {
	return upperName;
    }


    /*
     * Class fields
     */

    // One reason the field-descriptions aren't collected until
    // getFieldDescrs() is called is to break a possible infinite
    // recursion.  The typeDescr in a FieldDescr is a ClassDescr,
    // and constructing the FieldDescrs for a class C might involve
    // a field of type C.  If constructing the ClassDescr for C
    // also constructed the FieldDescrs, constructing the FieldDescr
    // for a field of type C would resursively call for the
    // ClassDescr for C.

    /** Returns a List of {@link FieldDescr}s. */
    public List<FieldDescr> getFieldDescrs() {
	ensureFields();
	return fields.getFields();
    }

    protected void ensureFields() {
	if (fields == null)
	    fields = syntax.makeFieldDescrs(theClass);
    }

    public FieldDescr fieldForName(String javaName) {
	ensureFields();
	return fields.fieldForName(javaName);
    }

    public FieldDescr fieldForExternalName(String externalName) {
	ensureFields();
	return fields.fieldForExternalName(externalName);
    }

    public void visitFields(FieldVisitor fv) {
	for (Iterator fi = getFieldDescrs().iterator(); fi.hasNext();) {
	    FieldDescr fd = (FieldDescr)fi.next();
	    fv.visitField(this, fd);
	}
    }


    /*
     * toString() - for debug output
     */

    public String description() {
	if (isList()) {
	    if (eltType == null)
		return "list";
	    else
		return "list of " + eltType.getExternalName();
	}
	else if (isSet()) {
	    if (eltType == null)
		return "set";
	    else
		return "set of " + eltType.getExternalName();
	}
	else if (isMap()) {
	    return "map";	// for now /\/
	}
	else {
	    return externalName;
	}
    }

    public String toString() {
	String about = description();
	if (isCollection() || isMap())
	    // include actual class
	    about = externalName + ", a " + about;
	return "ClassDescr[" + typeName[type] + ": " + about + "]";
    }


}
