/* File: SimpleMatcher.java
 * Contains: A simple pattern-matcher
 * Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Created: January 1998
 * Updated: Sun Apr 10 21:05:14 2005 by Jeff Dalton
 * Copyright: (c) 1998, 2005, AIAI, University of Edinburgh
 */

package ix.util.match;

import java.util.*;
import ix.util.*;
import ix.util.lisp.*;

import ix.icore.Variable; // sigh /\/


/**
 * A less simple pattern-matcher.
 */

public final class Matcher {

    /**
     * emptyEnv is used to return non-null without allocating an env.
     */
    public static final MatchEnv emptyEnv = new MatchEnv() {
	public Object put(Object key, Object Value) {
	    throw new ConsistencyException("emptyEnv can't be modified");
	}
	public Object checkedPut(Object key, Object value) {
	    throw new ConsistencyException("emptyEnv can't be modified");
	}
	public Object get(Object key) {
	    Debug.expect(isEmpty(), "Empty env isn't empty");
	    return null;
	}
    };

    public static final Symbol REST = Symbol.intern("&rest");

    public static final Symbol ITEM_WILD = Symbol.intern("??");

    public static final 
    MatchEnv mustMatch(Object pat, Object dat) {
	return mustMatch(pat, dat, null);
    }

    public static final
    MatchEnv mustMatch(Object pat, Object dat, MatchEnv env) {
	MatchEnv result = match(pat, dat, env);
	Debug.expect(result != null, "Failed to match", pat);
	return result;
    }

    /**
     * Match tries to match a pattern against an object. <p>
     *
     * Tries to handle Variables as well as ItemVars.
     *
     * @see SimpleMatcher
     * @see MatchEnv
     * @see ix.util.lisp.ItemVar
     * @see ix.icore.Variable
     */

    public static final MatchEnv match(Object pat, Object dat) {
	return match(pat, dat, null);
    }

    public static final MatchEnv match(Object pat, Object dat, MatchEnv env) {
	// N.B. Some cases could be handled more elegantly by recursion,
	// but we don't want to suggest that cases such as chains of
	// variables bound to variables are meant to work.  /\/

	if (pat == ITEM_WILD || dat == ITEM_WILD)
	    return success(env);

	// Protect emptyEnv from accidental modification
	if (env == emptyEnv) env = null;

	// Get rid of some cases that involve variables.

	// Debug.expect(!(dat instanceof ItemVar), "Match data has " + dat);
	if (dat instanceof ItemVar) {
	    if (pat instanceof ItemVar)
		return success(env);         //\/ should unify ...
	    else
		return match(dat, pat, env); //\/ for now ...
	}

	if (pat instanceof Variable) {
	    Object val = deref((Variable)pat, env);
	    if (val != null)
		pat = val;
	}
	if (dat instanceof Variable) {
	    Object val = deref((Variable)dat, env);
	    if (val != null)
		dat = val;
	}

	// Try matching ...
	if (pat == dat) {
	    return success(env);
	}
	else if (pat instanceof ItemVar) {
	    if (env == null)
		env = new MatchEnv();
	    Object existingValue = env.get(pat);
	    if (existingValue == null) {
		env.checkedPut(pat, dat);
		return env;
	    }
	    if (existingValue instanceof Variable) {
		Object val = deref((Variable)existingValue, env);
		if (val != null) {
		    existingValue = val;
		    // N.B. no return here
		}
		else {
		    // existingValue is an unbound Variable, so we
		    // bind both it and the item var to the data.
		    if (dat instanceof Variable) {
			Debug.noteln("Asked to unify value " + existingValue +
				     " and " + dat);
			// return null;
			return success(env);
		    }
		    env.checkedPut(pat, dat);
		    env.checkedPut(existingValue, dat);
		    return env;
		}
	    }
	    // Could call match recursively here.  /\/
	    return existingValue.equals(dat) ? env : null;
	}
	else if (pat instanceof Variable) {
	    // We know from code above that the Variable does not
	    // yet have a value.
	    if (dat instanceof Variable) {
		Debug.noteln("Asked to unify " + pat + " and " + dat);
		// return null;
		return success(env);
	    }
	    if (env == null)
		env = new MatchEnv();
	    env.checkedPut(pat, dat);
	    return env;
	}
	else if (dat instanceof Variable) {
	    // We know from code above that the Variable does not
	    // yet have a value.
	    if (env == null)
		env = new MatchEnv();
	    env.checkedPut(dat, pat);
	    return env;
	}
	else if (pat instanceof Cons && ((Cons)pat).car() == REST) {
	    return matchRest(pat, dat, env);
	}
	else if (pat instanceof Number) {
	    return when(pat.equals(dat), env);
	}
	else if (pat.getClass() != dat.getClass()) {
	    return null;
	}
	else if (pat instanceof Symbol) {
	    return null;	// matching symbols would have been ==
	}
	else if (pat instanceof String) {
	    return when(((String)pat).equals((String)dat), env);
	}
	else if (pat instanceof Cons) {
	    Cons p = (Cons)pat;
	    Cons d = (Cons)dat;
	    MatchEnv e = match(p.car(), d.car(), env);
	    if (e == null)
		return null;
	    else if (e == emptyEnv)
		return match(p.cdr(), d.cdr(), null);
	    else
		return when(match(p.cdr(), d.cdr(), e) != null,
			    e);
	}
	else {
	    return null;
	}

    }

    public static final Object deref(Variable var, MatchEnv env) {
	Object val = var.getValue();
	if (val != null) {
	    Debug.expect(env == null || env.get(var) == null);
	    return val;
	}
	return env == null ? var : env.get(var);
    }

    private static final MatchEnv matchRest(Object pat, Object dat,
					    MatchEnv env) {
	if (!(dat instanceof LList))
	    return null;	// no match
	Cons p = (Cons)pat;
	Object var = p.cdr().car();
	if (var == Lisp.NIL)	 	// (&rest) matches any List
	    return success(env);
	Debug.expect(p.cdr().cdr() == Lisp.NIL,
		     "(&rest ?var) was the end of pattern", pat);
	return match(var, dat, env);
    }

    private static final MatchEnv when(boolean cond, MatchEnv env) {
	if (cond)
	    return success(env);
	else
	    return null;
    }

    private static final MatchEnv success(MatchEnv env) {
	if (env == null)
	    return emptyEnv;
	else
	    return env;
    }

}

// Issues:
// * If we don't want to handle Variable-Variable matches (yet), what should
//   we do?  If we make the match fail, that gives us some false negatives;
//   but if we make it succeed without recording anything, we forget that
//   the variables must have the same value.  At present, we make it succeed,
//   because most of our matches are "possibility tests" (so that false
//   negatives are undesirable) that are reconsidered when more Variables
//   are bound.  [W/o any V-V cases, we expect binding to rule out some
//   matches that seemed ok earlier, not to make new ones possible; so
//   it makes sense to have V-V matches fit that pattern.]
//
//   It also means that certain cases will work ... For example,
//   suppose (f ?x ?x ?x) is entered as an activity, so that the
//   variables all become Variables, and there's a refinement
//   with an expands pattern (f ?y ?y apple).  That will match
//   with Variable[from ?x] bound to apple.
