/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Wed Nov 10 22:02:00 2004 by Jeff Dalton
 * Copyright: (c) 2003, AIAI, University of Edinburgh
 */

package ix.util.match;

import java.util.*;

import ix.icore.Variable;
import ix.util.*;

/**
 * A wrapper around a Map from {@link Variable}s to their values.
 */
public class Bindings {

    Map map;

    public Bindings(Map valuation) {
	this.map = valuation;
    }

    public static List mapsToBindings(List maps) {
	List result = new LinkedList();
	for (Iterator i = maps.iterator(); i.hasNext();) {
	    Map m = (Map)i.next();
	    if (!Collect.every(m.keySet(), Fn.isInstanceOf(Variable.class)))
		throw new IllegalArgumentException
		    ("Map has a key that isn't a Variable: " + m);
	    result.add(new Bindings(m));
	}
	return result;
    }

    public static List bindingsToMaps(List bindings) {
	List result = new LinkedList();
	for (Iterator i = bindings.iterator(); i.hasNext();) {
	    Bindings b = (Bindings)i.next();
	    result.add(b.map);
	}
	return result;
    }

    public Set getVariables() {
	return map.keySet();
    }

    public Object valueOf(Variable v) {
	return map.get(v);
    }

    public boolean isConsistentWith(Bindings b) {
	return Collect.areConsistent(map, b.map);
    }

    public boolean isConsistentWith(MatchChoice c) {
	// /\/: Looks at the MatchChoice's liveBranches.
	List live = c.getLiveBranches();
	if (live.isEmpty()) {
	    // The MatchChoice doesn't allow any values for any
	    // of its variables.
	    return Collect.haveCommonElements(getVariables(), c.getVariables())
		? false : true;
	}
	for (Iterator i = live.iterator(); i.hasNext();) {
	    Bindings b = (Bindings)i.next();
	    if (Collect.areConsistent(map, b.map)) {
		// Debug.noteln("Consistent " + map + ", " + b.map);
		return true;
	    }
	}
	// Debug.noteln("Inconsistent " + map + ", " + live);
	return false;
    }

    public String toString() {
	return "Bindings" + map.toString();
    }

}

// Issues:
// * It is important that equals is object identity (==)
//   for these objects.  We use the collection contains method
//   to see if a MatchChoice contains a Bindings, but we want
//   to allow ==-different elements that equals(Object) says
//   are the same.

