/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Mon Jan  7 17:24:30 2008 by Jeff Dalton
 */

package ix.util.lisp;

import java.util.*;

import ix.util.*;

/**
 * Converts objects to printed representations.
 */
public class LispPrinter {

    private boolean printUnprintablesUsingToString = false;

    private String stringForNil = "nil";

    public LispPrinter() {
    }

    public void setPrintUnprintablesUsingToString(boolean v) {
	printUnprintablesUsingToString = v;
    }

    public void setStringForNil(String s) {
	stringForNil = s;
    }

    public String printToString(Object a) {
	if (a == null)
	    return "#<null>";
	else if (a instanceof DelimitedSymbol)
	    return quotedAndEscaped('|', a.toString());
	else if (a instanceof Symbol)
	    return a.toString();
	else if (a instanceof Number)
	    return a.toString();
	else if (a instanceof String)
	    return quotedAndEscaped('"', (String)a);
	else if (a instanceof Null)
	    return stringForNil;
	else if (a instanceof Cons) {
	    LList at = (Cons)a;
	    String s = "(" + printToString(at.car());
	    for (; at.cdr() instanceof Cons; at = at.cdr()) {
		s += " " + printToString(at.cdr().car());
	    }
	    return s + ")";
	}
	else if (a instanceof List) {
	    StringBuffer b = new StringBuffer();
	    b.append("(");
	    for (Iterator i = ((List)a).iterator(); i.hasNext();) {
		b.append(printToString(i.next()));
		if (i.hasNext())
		    b.append(" ");
	    }
	    b.append(")");
	    return b.toString();
	}
	else {
	    return unprintableToString(a);
	}
    }

    protected String unprintableToString(Object a) {
	// Debug.warn("printToString got a " + a.getClass() + ":" + a);
	// return "#<" + a.getClass() + ">";
	// return "#<" /* + a.getClass() +": " */ + a.toString() + ">";
	// For a long time, it was:
	// return a.toString();
	if (printUnprintablesUsingToString)
	    return a.toString();
	else if (a instanceof ix.icore.Variable)
	    return a.toString(); 		// /\/ hack
	else
	    return Lisp.hashName(a);
    }

    public boolean isFullyPrintable(Object a) {
	return a instanceof Symbol
	    || a instanceof String
	    ||(a instanceof Cons && isFullyPrintableCons((Cons)a))
	    || a instanceof Null
	    || a instanceof Long
	    || a instanceof Double
	    || a instanceof Integer
	    || a instanceof Float;
    }
    private boolean isFullyPrintableCons(Cons c) {
	for (LList tail = c; tail != Lisp.NIL; tail = tail.cdr()) {
	    if (!isFullyPrintable(tail.car()))
		return false;
	}
	return true;
    }

    /**
     * Returns a string containing the speficied object(s), converted
     * to strings by calling {@link #printToString(Object)} and with
     * a single space as the separator.
     */
    public String elementsToString(Object a) {
	if (!(a instanceof List))
	    return printToString(a);
	StringBuffer buf = new StringBuffer();
	for (Iterator i = ((List)a).iterator(); i.hasNext();) {
	    buf.append(printToString(i.next()));
	    if (i.hasNext())
		buf.append(" ");
	}
	return buf.toString();
    }

    /**
     * Adds quote marks around a string and escape sequences inside it.
     */
    public String quotedAndEscaped(char quote, String s) {
	int len = s.length();
	// See how many escape chars we'll need to add
	int increase = 0;
	for (int i = 0; i < len;) {
	    char c = s.charAt(i);
	    if (c == quote || c == '\\' || c == '\n')
		increase++;
	    else if (c == '\r') {
		increase++;
		// deal with possible CRLF
		int j = i + 1;
		if (j < len) {
		    char c2 = s.charAt(j);
		    if (c2 == '\n')
			i = j;
		}
	    }
	    i++;
	}
	// Make a buffer large enough to hold the text, the escape chars,
	// and a quote at each end.
	SimpleStringBuffer buf = new SimpleStringBuffer(len + increase + 2);
	buf.append(quote);
	for (int i = 0; i < len;) {
	    char c = s.charAt(i);
	    if (c == quote || c == '\\' || c == '\n')
		buf.append('\\');
	    else if (c == '\r') {
		buf.append('\\');
		// deal with possible CRLF
		int j = i + 1;
		if (j < len) {
		    char c2 = s.charAt(j);
		    if (c2 == '\n') {
			buf.append(c);
			i = j;
			c = c2;
		    }
		}
	    }
	    buf.append(c);
	    i++;
	}
	buf.append(quote);
	return buf.toString();
    }

    /** 
     * Returns the number of characters in the textual represpetation of
     * an object.
     */
    public int printLength(Object a) {
	//\/: Numbers other than Longs are done inefficiently.
	//\/: Doesn't count the "/"s that will appear in escape
	//    sequences inside strings and delimited symbols.
	if (a == null)
	    return printedLen(a);
	else if (a instanceof DelimitedSymbol)
	    return a.toString().length() + 2;
	else if (a instanceof Symbol)
	    return a.toString().length();
	else if (a instanceof Long) {
	    Long n = ((Long)a).longValue();
	    int len = 0;
	    if (n < 0) { len++; n = -n; }
	    while (n > 0) { len++; n = n / 10; }
	    return len;
	}
	else if (a instanceof Number)
	    return printedLen(a);
	else if (a instanceof String)
	    return ((String)a).length() + 2;
	else if (a instanceof Null)
	    return printedLen(a);
	else if (a instanceof List) {
	    int len = 1;	// for the (
	    for (Iterator i = ((List)a).iterator(); i.hasNext();) {
		len += printLength(i.next());
		if (i.hasNext())
		    len++;	// for a space
	    }
	    return len + 1;	// + 1 for the )
	}
	else {
	    return printedLen(a);
	}
    }

    private int printedLen(Object a) {
	return printToString(a).length();
    }

}
