/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Mon Jan  7 17:26:46 2008 by Jeff Dalton
 */

package ix.util.lisp;

import java.io.*;
import java.net.URL;
import java.util.*;

import ix.util.*;
import ix.util.xml.XML;


/** The class for static Lisp utilities. */

public class Lisp {

    private Lisp() { }		// can't instantiate


    public static final Null NIL = new Null();

    public static final Object EOF = new UniqueObject("EOF");

    private static final LispPrinter printer = new LispPrinter();


    /* cons */

    public static final Cons cons(Object car, LList cdr) {
	// /\/: maybe Object cdr and cast to LList.
	return new Cons(car, cdr);
    }


    /* list */

    public static final Null list() {
	return NIL;
    }

    public static final Cons list(Object a) {
	return new Cons(a, NIL);
    }

    public static final Cons list(Object a, Object b) {
	return new Cons(a, new Cons(b, NIL));
    }

    public static final Cons list(Object a, Object b, Object c) {
	return new Cons(a, new Cons(b, new Cons(c, NIL)));
    }

    public static final Cons list(Object a, Object b, Object c, Object d) {
	return new Cons(a, new Cons(b, new Cons(c, new Cons(d, NIL))));
    }

    public static final Cons list(Object a, Object b, Object c,
				  Object d, Object e) {
	return new Cons(a, new Cons(b, new Cons(c, 
                 new Cons(d, new Cons(e, NIL)))));
    }

    public static final Cons list(Object a, Object b, Object c,
				  Object d, Object e, Object f) {
	return new Cons(a, new Cons(b, new Cons(c, 
                 new Cons(d, new Cons(e, new Cons(f, NIL))))));
    }


    /* equal */

    public static final boolean equal(Object a, Object b) {
	if (a == b)
	    return true;
	else if (a instanceof Cons)
	    return b instanceof Cons
		   ? equal(((Cons)a).car, ((Cons)b).car) &&
		     equal(((Cons)a).cdr, ((Cons)b).cdr)
		   : false;
	else if (a instanceof String)
	    return b instanceof String ? a.equals(b) : false;
	else if (a instanceof Number)
	    return b instanceof Number ? a.equals(b) : false;
	else
	    return false;
    }

    public static LispReader openForInput(URL url) throws IOException {
	Reader reader = Util.openURLReader(url);
	return new LispReader(new BufferedReader(reader));
    }

    public static LispReader openForInput(String resourceName)
	   throws IOException {
	URL url = XML.toURL(resourceName);
	if (url == null)
	    throw new IllegalArgumentException("Can't find " + resourceName);
	else
	    return openForInput(url);
    }

    /* readFromString */

    public static Object readFromString(String s) {
	LispReader lr = new LispReader(s);
	try {
	    Object result = lr.readObject();
	    if (lr.readObject() != Lisp.EOF)
		throw new LispReadException(
		    s.trim().startsWith("(")
		    ? "Extra close paren"
		    : "More than one object");
	    return result;
	}
	catch (LispReadException e) {
	    throw new LispReadException(
	      e.getMessage() + " in string " + Strings.quote(s));
	}
    }

    public static LList elementsFromString(String s) {
	// return (LList)Lisp.readFromString("(" + s + ")");
	LListCollector result = new LListCollector();
	LispReader lr = new LispReader(s);
	try {
	    while (true) {
		Object elt = lr.readObject();
		if (elt == Lisp.EOF)
		    return result.contents();
		else
		    result.add(elt);
	    }
	}
	catch (LispReadException e) {
	    throw new LispReadException(
	      e.getMessage() + " in string " + Strings.quote(s));
	}
    }


    /* printToString */

    public static String printToString(Object a) {
	return printer.printToString(a);
    }

    public static boolean isFullyPrintable(Object a) {
	return printer.isFullyPrintable(a);
    }

    /**
     * Returns a string containing the speficied object(s), converted
     * to strings as if by calling {@link #printToString(Object)} and with
     * a single space as the separator.
     */
    public static String elementsToString(Object elts) {
	return printer.elementsToString(elts);
    }

    /**
     * Adds quote marks around a string and escape sequences inside it.
     */
    public static String quotedAndEscaped(char quote, String s) {
	return printer.quotedAndEscaped(quote, s);
    }

    /** 
     * Returns the number of characters in the textual represpetation of
     * an object.
     */
    public static int printLength(Object a) {
	return printer.printLength(a);
    }


    /* object hash and unhash */

    private static ObjectHash defaultHash = new ObjectHash();

    public static int hash(Object obj) {
	return defaultHash.hash(obj);
    }

    public static Object unhash(int h) {
	return defaultHash.unhash(h);
    }

    public static String hashName(Object obj) {
	return defaultHash.hashName(obj);
    }

    public static Object parseHashName(String name) {
	return defaultHash.parseHashName(name);
    }

}
