/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Thu Apr 24 14:33:07 2008 by Jeff Dalton
 * Copyright: (c) 2001, 2008, AIAI, University of Edinburgh
 */

package ix.util.context;

import java.util.*;

/**
 * An element in a chain of associations between contexts and values. <p>
 *
 * A chain of ContextLinks is used to map contexts to values (Objects).
 * The public constructors create a 1-link chain.  This normally specifies
 * a default, initial, or top-level value (typically one associated with
 * the root context).  Further links are normally added by calling the
 * Context.setInContext method; this ensures that the resulting chain
 * can be processed correctly by the Context.getInContext method,
 * which requires that entries for newer (higher-numbered) contexts
 * appear before those for older ones.
 *
 * @see Context#getInContext(ContextLink cl, Context c)
 * @see Context#setInContext(ContextLink cl, Context c, Object value)
 */
public final class ContextLink<T> {

    long contextNumber;
    T value;
    ContextLink next;

    /**
     * Creates a link that specifies a value in the root context.
     */
    public ContextLink(T value) {
	this(Context.rootContext, value, null);
    }

    /**
     * Creates a link that specifies a value for a given context.
     */
    public ContextLink(Context context, T value) {
	this(context, value, null);
    }

    /**
     * Constructs a new link at the head of a chain.
     */
    ContextLink(Context context, T value, ContextLink<T> next) {
	this.contextNumber = context.number;
	this.value = value;
	this.next = next;
    }

    /**
     * Constructs a new link at the head of a chain.
     */
    ContextLink(long contextNumber, T value, ContextLink<T> next) {
	this.contextNumber = contextNumber;
	this.value = value;
	this.next = next;
    }

}
