/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Fri Nov 19 21:55:15 2004 by Jeff Dalton
 * Copyright: (c) 2004, AIAI, University of Edinburgh
 */

package ix.util;

/**
 * A class that encapsulates the tricks needed to use a "finally"
 * clause that might throw an exception.  A throw (or return) out
 * of a "finally" clause will cause any exception from the "try"
 * clause to be lost.  So there has to be another try-catch
 * inside the "finally", and up to two exceptions might have
 * to be handled.
 *
 * <p>Instead of creating a plain Runnable, like this:
 * <pre>
 *     new Runnable() {
 *         public void run() {
 *             ... do something ...
 *         }
 *     }
 * </pre>
 * you can instead create a WithCleanup:
 * <pre>
 *     new WithCleanup() {
 *         public void body() {
 *             ... do something ...
 *         }
 *         public void cleanup() {
 *             ... do some cleanup ...
 *         }
 *     }
 * </pre>
 * </p>
 * <p>When the WithCleanup's {@link #run()} method is called,
 * it will call {@link #body()} inside a "try", and {@link #cleanup()}
 * inside the corresponding "finally".</p>
 *
 * <p>If any exception (any Throwable) is thrown out of the body or
 * cleanup, this method throws a {@link RethrownException}.  If only
 * the body, or only the cleanup, throws an exception, then the
 * RethrownException is a simple wrapper around that exception.
 * If both body and cleanup throw an exception, the RethrownException
 * is wrapped around the body's exception, but its message describes
 * both exceptions.  In addition, {@link Debug#describeException(Throwable)}
 * is called on cleanup's exception to print its backtrace.</p>
 */
public abstract class WithCleanup implements Runnable {

    /**
     * Calls {@link #body()} in a "try", {@link #cleanup()}
     * in the corresponding "finally", and ensures that any
     * exception thrown by {@link #body()} is rethrown.
     * See the class's javadoc for a more complete explanation.
     *
     * <p>Subclasses should not normally redefine this method.</p>
     */
    public void run() {
	Throwable thrown = null;
	try {
	    this.body();
	}
	catch (Throwable t) {
	    thrown = t;
	    throw new RethrownException(t);
	}
	finally {
	    try {
		this.cleanup();
	    }
	    catch (Throwable t2) {
		if (thrown == null)
		    throw new RethrownException(t2);
		else {
		    Debug.noteln("Exception during cleanup after " +
				 Debug.describeException(thrown));
		    Debug.noteException(t2);
		    throw new RethrownException
			(thrown,
			 Debug.describeException(thrown) + ". \n\n" +
			 "Second exception during cleanup: " +
			 Debug.describeException(t2));
		}
	    }
	}
    }

    /**
     * The main part of what this object should do when its
     * {@link #run()} method is called.
     */
    public abstract void body() throws Exception;

    /**
     * Whatever this object should always do after its
     * {@link #body()} method has returned.  cleanup() is
     * called in a "finally" that goes with a "try"-"catch"
     * that's around the call to {@link #body()}.
     */
    public abstract void cleanup() throws Exception ;

}
