/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Mon May 12 17:05:21 2008 by Jeff Dalton
 * Copyright: (c) 2003, 2005 - 2008, AIAI, University of Edinburgh
 */

package ix.util;

import java.util.*;

import ix.util.reflect.*;
import ix.util.xml.XML;
import ix.util.lisp.*;

/**
 * Compares two objects to see whether they have the same values
 * in the same places.  This is simpler than a Comparator, because
 * it doesn't have to determine which of two unequal objects is
 * the greater.  However, what counts as the same value, or the
 * same place, might very, so that more than one class of this
 * sort could be defined.  Another variation could be the ability
 * to handle circularities.
 *
 * <p>This class
 * <ul>
 * <li>Requires that equal objects be of exactly the same class.
 * <li>Allows elements to appear in different orders when comparing
 *     Sets or Maps.
 * <li>Uses a {@link ClassSyntax} to determine which fields
 *     to compare in structures.
 * <li>Doesn't handle arrays.
 * </ul></p>
 */
public class StructuralEquality {

    protected ClassSyntax syntax;

    /**
     * Creates an equality test that used the default {@link ClassSyntax}.
     *
     * @see XML
     */
    public StructuralEquality() {
	this(XML.config().defaultClassSyntax());
    }

    /**
     * Creates an equality test that uses the specified {@link ClassSyntax}
     * as a source for {@link ClassDescr}s.
     */
    public StructuralEquality(ClassSyntax syntax) {
	this.syntax = syntax;
    }

    /**
     * Compares two objects, returning true if they are structurally
     * equal and false otherwise.
     *
     * @throws IncomparableException if a comparison of corresponding
     *    parts of the two objects that should be possible cannot be
     *    performed.
     */
    public boolean equal(Object a, Object b) {
	if (a == b)
	    return true;
	else if (a.getClass() != b.getClass() && !haveEquivalentClasses(a, b))
	    return false;
	else if (a instanceof String)
	    return a.equals(b);
	else if (a instanceof Number) {
	    if (a instanceof Comparable)
		return ((Comparable)a).compareTo(b) == 0;
	    else
		throw new IncomparableException(a, b);
	}
	else if (a instanceof List)
	    return equalLists((List)a, (List)b);
	else if (a instanceof Set)
	    return equalSets((Set)a, (Set)b);
	else if (a instanceof Map)
	    return equalMaps((Map)a, (Map)b);
//  	else if (a instanceof Name) 		// irritating special case /\/
//  	    return a.equals(b);
	else if (a instanceof SemiPrimitive)
	    return a.equals(b);
	else {
	    ClassDescr cd = syntax.getClassDescr(a.getClass());
	    if (cd.isStruct())
		return equalStructs(a, b, cd);
	    else if (a instanceof Comparable)  // handles various random things
		return ((Comparable)a).compareTo(b) == 0;
	    else if (a instanceof Boolean)
		return a.equals(b);
	    else
		return a == b;	// but we know they're not == from test above
	}
    }

    protected boolean haveEquivalentClasses(Object a, Object b) {
	return a.getClass() == b.getClass();
    }

    protected boolean equalLists(List a, List b) {
	Iterator i = a.iterator(), j = b.iterator();
	while (i.hasNext() && j.hasNext()) {
	    if (!equal(i.next(), j.next()))
		return false;
	}
	return !i.hasNext() && !j.hasNext();
    }

    protected boolean equalSets(Set a, Set b) {
	// Tricky, because the elements might not be in the same order.
	if (a.size() != b.size())
	    return false;
	List b_elts = new LinkedList(b);
    a_loop:
	for (Iterator i = a.iterator(); i.hasNext();) {
	    Object ai = i.next();
	b_loop:
	    for (Iterator j = b_elts.iterator(); j.hasNext();) {
		if (equal(ai, j.next())) {
		    j.remove();	// needn't look at it again.
		    continue a_loop;
		}
	    }
	    return false;
	}
	Debug.expect(b_elts.isEmpty());
	return true;
    }

    protected boolean equalMaps(Map a, Map b) {
	// Tricky, because the entries might not be in the same order.
	if (a.size() != b.size())
	    return false;
	List b_ents = new LinkedList(b.entrySet());
    a_loop:
	for (Iterator i = a.entrySet().iterator(); i.hasNext();) {
	    Map.Entry a_e = (Map.Entry)i.next();
	b_loop:
	    for (Iterator j = b_ents.iterator(); j.hasNext();) {
		Map.Entry b_e = (Map.Entry)j.next();
		if (equal(a_e.getKey(), b_e.getKey())
		       && equal(a_e.getValue(), b_e.getValue())) {
		    j.remove();	// needn't look at it again.
		    continue a_loop;
		}
	    }
	    return false;
	}
	Debug.expect(b_ents.isEmpty());
	return true;
    }

    protected boolean equalStructs(Object a, Object b, ClassDescr cd) {
	List fields = cd.getFieldDescrs();
	try {
	    for (Iterator fi = fields.iterator(); fi.hasNext();) {
		FieldDescr fd = (FieldDescr)fi.next();
		Object f_a = fd.getValue(a);
		Object f_b = fd.getValue(b);
		if (f_a == null && f_b == null)
		    continue;
		else if (f_a == null || f_b == null) {
		    if (!missingFieldIsOk(cd, fd, f_a, f_b))
			return false;
		}
		else if (!equal(f_a, f_b))
		    return false;
	    }
	    return true;
	}
	catch(IncomparableException e) {
	    throw e;
	}
	catch(Exception e) {
	    throw new IncomparableException(a, b, e);
	}
    }

    protected boolean missingFieldIsOk(ClassDescr cd, FieldDescr fd,
				       Object f_a, Object f_b) {
	return false;
    }

    public static class IncomparableException
	   extends IllegalArgumentException {

	public IncomparableException(Object a, Object b) {
	    this(a, b, null);
	}

	public IncomparableException(Object a, Object b, Throwable reason) {
	    super("Cannot compare " + a.getClass().toString() +
		    " instances 1: " + a + " and 2: " + b +
		    (reason == null ? "" : " because " + reason),
		  reason);
	}

    }

}
