/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Mon May  4 11:52:13 2009 by Jeff Dalton
 * Copyright: (c) 2009, AIAI, University of Edinburgh
 */

package ix.util;

import java.io.*;
import java.util.*;

/**
 * Automatically log property values to a file.  The property,
 * object, and property-value must all be strings.
 */
public class PropertyLogger {

    private String logFileName;

    private Map<String,LoggingMap> propNameToObjectValueMap =
        new LinkedHashMap<String,LoggingMap>();

    private Writer logWriter = null;

    /**
     * Make a logger for the named file.
     */
    public PropertyLogger(String logFileName) {
        this.logFileName = logFileName;
    }

    /**
     * Returns this logger's LoggingMap for the named property,
     * or else null of no such map yet exists.
     */
    public LoggingMap getMap(String propName) {
        return propNameToObjectValueMap.get(propName);
    }

    /**
     * Returns this logger's LoggingMap for the named property,
     * creating it if it does not already exist.
     */
    public LoggingMap ensureMap(String propName) {
        LoggingMap map = getMap(propName);
        if (map == null) {
            map = new LoggingMap(propName);
            propNameToObjectValueMap.put(propName, map);
        }
        return map;
    }

    /**
     * Returns the object's value of the named property, or else
     * else if no such value has been assigned.
     */
    public String get(String propName, String objectName) {
        LoggingMap map = getMap(propName);
        return map != null? getMap(propName).get(objectName) : null;
    }

    /**
     * Records a value for the named object and property.  The first
     * time this method is called, this logger's log file will be
     * renamed, if it exists, and a new log file will be created.
     * Any values that had been read by calling {@link #readLogIfExists()}
     * will be recorded in the new file, followed by the new
     * property value.
     */
    public void put(String propName, String objectName, String propValue) {
        try {
            if (logWriter == null)
                openLogFile();
            logProp(propName, objectName, propValue);
            logWriter.flush();
        }
        catch (IOException e) {
            throw new RethrownIOException(e);
        }
        ensureMap(propName).__record(objectName, propValue);
    }

    /**
     * Records a value for the named object and property, but only if
     * the value is new.  "New" means either that no value had been
     * recorded for that object and property combination before, or
     * that a different value had been recorded.  "Different" is
     * determined by calling 'equals'.  Recording a value affacts
     * the log file in the same way as {@link #put(String,String,String)}.
     *
     * @return true if the information was new and false otherwise.
     */
    public boolean putIfNew(String propName, String objectName,
                            String propValue) {
        return ensureMap(propName).putIfNew(objectName, propValue);
    }

    /**
     * A Map-like object that records any property value that is
     * given to its 'put' method.  The value is recorded in this
     * logger's log file as if the logger's
     * {@link PropertyLogger#put(String, String, String)}
     * method had been called.
     */
    public class LoggingMap {

        // /\/: Deliberately does not implement Map so that the implementation
        // can be simple without exposing functionality that won't ensure
        // that property-value changes are recorded.

        private Map<String,String> map = new LinkedHashMap<String,String>();

        private String propName;

        protected LoggingMap(String propName) {
            this.propName = propName;
        }

        public String get(String objectName) {
            return map.get(objectName);
        }

        public void put(String objectName, String propValue) {
            PropertyLogger.this.put(propName, objectName, propValue);
        }

        public boolean putIfNew(String objectName, String propValue) {
            String existingValue = map.get(objectName);
            if (existingValue != null && existingValue.equals(propValue))
                return false;
            else {
                put(objectName, propValue);
                return true;
            }
        }

        private void __record(String objectName, String propValue) {
            map.put(objectName, propValue);
        }

        private Set<Map.Entry<String,String>> __entrySet() {
            return map.entrySet();
        }

        public Set<Map.Entry<String,String>> entrySet() {
            return Collections.unmodifiableMap(map).entrySet();
        }

        public String toString() {
            return "LoggingMap[" + propName + "]";
        }

    }

    /**
     * Reads and remembers property values from this logger's log file
     * if the file exists.  This method does not change the file.
     *
     * <p>When the first new property value is assigned, the file will
     * be renamed (if it exists), and a new file will be created.
     * The old property values (if any), plus the new value, will
     * then be written to the new file.
     */
    public void readLogIfExists() {
        try {
            File log = new File(logFileName);
            if (log.exists()) {
                Debug.noteln("Reading propert values from", logFileName);
                final BufferedReader r = new BufferedReader
                                                (new FileReader(log));
                Util.run(new WithCleanup() {
                    public void body() throws IOException {
                        while (true) {
                            String line = r.readLine();
                            if (line == null)
                                break;
                            readProp(line);
                        }
                    }
                    public void cleanup() throws IOException {
                        r.close();
                    }
                });
            }
        }
        catch (IOException e) {
            throw new RethrownIOException(e);
        }
    }

    private void readProp(String line) {
        // Syntax is propName(objectName)=propValue
        List<String> parts = Strings.breakAtAny("()=", line);
        if (parts.size() != 4 || !parts.get(2).equals(""))
            throw new IllegalArgumentException
                ("Systax error in property line: " + line);
        String propName = parts.get(0);
        String objectName = parts.get(1);
        String propValue = parts.get(3);
        Debug.noteln("Property: " + propName + "(" + objectName + ")=" +
                     propValue);
        ensureMap(propName).__record(objectName, propValue);
    }

    private void logProp(String propName, String objectName, String propValue)
            throws IOException {
        logWriter.write(propName + "(" +objectName + ")=" + propValue + "\n");
    }

    private void openLogFile() throws IOException {
        File log = new File(logFileName);
        if (log.exists())
            Util.renameToBackup(log);
        logWriter = new FileWriter(log);
        // Start by recording everything we already know.
        for (Map.Entry<String,LoggingMap> e1:
                 propNameToObjectValueMap.entrySet()) {
            String propName = e1.getKey();
            LoggingMap loggingMap = e1.getValue();
            for (Map.Entry<String,String> e2: loggingMap.__entrySet()) {
                logProp(propName, e2.getKey(), e2.getValue());
            }
        }
    }

    /**
     * Simple main program for testing.  Takes one command-line
     * argument which should be the name of the log file.
     */
    public static void main(String[] argv) {
        String logFileName = argv[0];
        PropertyLogger logger = new PropertyLogger(logFileName);
        logger.readLogIfExists();
        while (true) {
            String line = Util.askLine("prop object value:");
            if (line.equals("bye"))
                break;
            List<String> parts = Strings.breakAt(" ", line);
            if (parts.size() == 3) {
                if (logger.putIfNew(parts.get(0), parts.get(1), parts.get(2)))
                    System.out.println("Thanks.  That was new.");
                else
                    System.out.println("I knew that.");
            }
            else
                System.out.println("Syntax error");
        }
    }

}
