/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Fri Sep 29 16:15:58 2006 by Jeff Dalton
 * Copyright: (c) 2001 - 2005, AIAI, University of Edinburgh
 */

package ix.util;

import java.util.*;
import java.io.*;
import java.net.*;

import ix.icore.Sendable;
import ix.icore.Annotated;
import ix.icore.AbstractAnnotatedObject;

import ix.util.ipc.*;
import ix.util.xml.*;

// import ix.util.lisp.Lisp;	// for Lisp.readFromString(String)
import ix.util.lisp.*;

/**
 * Support for interprocess communication in a framework that allows
 * different communication strategies to be used in a uniform way.
 * Some simple strategies, such as one that sends serialized objects
 * via sockets, are included. <p>
 *
 * The framework specifies certain entities only a in very general
 * way.  Communication strategies that are meant to be interchangeable
 * must therefore follow shared conventions - not enforced by the
 * framework - chiefly concerning the objects used to represent
 * "destinations" and implementations of the InputMessage interface. <p>
 *
 * Note that a "destination" need not contain all the information
 * needed for communication with the corresponding agent.  Instead,
 * the destination may be merely a name, perhaps as a String, with
 * the rest of the information stored elsewhere.
 */

public class IPC {

    private IPC() { }		// can't instantiate

    /**
     * The communication strategy used by the static methods of this
     * class that do not have a communication strategy as a parameter.
     */
    static CommunicationStrategy communicationStrategy
	= new EmptyCommunicationStrategy() {};

    static Object thisAgentAsDestination = null;

    /**
     * Listeners that can observe messages as they're sent and
     * received.
     */
    static List ipcListeners = new LinkedList();

    /**
     * Returns the communication strategy used by the static methods of this
     * class that do not have a communication strategy as a parameter.
     */
    public static synchronized CommunicationStrategy 
	    getCommunicationStrategy() {
	return communicationStrategy;
    }

    /**
     * Sets the communication strategy used by the static methods of this
     * class that do not have a communication strategy as a parameter.
     */
    public static synchronized void setCommunicationStrategy
	    (CommunicationStrategy s) {
	Debug.noteln("Setting global communication strategy to", s);
	communicationStrategy = s;
    }

    /**
     * Sets the communication strategy used by the static methods of this
     * class that do not have a communication strategy as a parameter.
     * The <code>strategyName</code> is interpreted by the 
     * {@link #makeCommunicationStrategy(String)} method.
     */
    public static synchronized void 
	    setCommunicationStrategy(String strategyName) {
	setCommunicationStrategy
	    (makeCommunicationStrategy(strategyName));
    }

    /**
     * Returns a CommunicationStrategy based on a name, <i>name</i>,
     * that is either an expression containing a colon, a special
     * abbreviation or the name of a class.  For the expression case,
     * see {@link #evalStrategyExpr(String)}; otherwise, the
     * algorithm is as follows, with <i>Name</i> standing for
     * <i>name</i> with its first character changed to upper case:
     * <ol>
     * <li>if <i>name</i> is <tt>""</tt> or <tt>"simple"</tt>,
     *     use <code>SimpleIXCommunicationStrategy</code>;
     * <li>else if <i>name</i> is <tt>"xml"</tt>,
     *     use <code>SimpleIXXMLCommunicationStrategy</code>;
     * <li>else if a class named <i>name</i> exists,
     *     use that class;
     * <li>else if a class named 
     *        <tt>ix.</tt><i>name.Name</i><tt>CommunicationStrategy</tt>
     *        exists, use that class;
     * <li>else if a class named
     *        <tt>ix.util.ipc.<i>Name</i>Strategy</tt>
     *        exists, use that class;
     * <li>else throw an <code>IllegalArgumentException</code>.
     * </ol>
     *
     * @see IPC.SimpleIXCommunicationStrategy
     * @see IPC.SimpleIXXMLCommunicationStrategy
     * @see #setCommunicationStrategy(String strategyName)
     */
    public static CommunicationStrategy
	          makeCommunicationStrategy(String strategyName) {

	// See if it's an expression
	if (strategyName.indexOf(":") >=0)
	    return evalStrategyExpr(strategyName);

	// Try special case abbreviations
	if (strategyName.equals("") || strategyName.equals("simple"))
	    return new SimpleIXCommunicationStrategy();
	else if (strategyName.equals("xml"))
	    return new SimpleIXXMLCommunicationStrategy();

	// Try it as the name, or abbreviated name, of a class.
	Class c;
	if ( // First try name as-is
	     (c = findClassForName(strategyName)) != null
	     ||
	     // Then as ix.name.NameCommunicationStrategy
	     (c = findClassForName
	             ("ix." + strategyName + "."
		      + Strings.capitalize(strategyName)
		      + "CommunicationStrategy")) != null
	     ||
	     // Then as ix.util.ipc.NameStrategy
	     (c = findClassForName
	             ("ix.util.ipc."
		      + Strings.capitalize(strategyName)
		      + "Strategy")) != null)
	    // We seem to have a class ...
	    try {
		return (CommunicationStrategy)c.newInstance();
	    }
	    catch (Exception e) {
		Debug.noteException(e);
		throw new IllegalArgumentException
		    ("Can't make communication strategy from " + strategyName +
		     " because " + e);
	    }
	else
	    throw new IllegalArgumentException
		("Can't find communication strategy class specified by " +
		 Util.quote(strategyName));
    }

    /**
     * Evaluate the description of a {@link ParameterizedCommStrategy}.
     * The syntax is <i>name</i>:<i>args</i> where <i>name</i>
     * is the name of a ParameterizedCommStrategy and <i>args</i> is
     * a comma-separated list.  The interpretation of the arguments
     * is up to the parameterized comm strategy, but usually the
     * arguments will be comm strategy names or expressions.
     * Some strategies that expect more than one strategy as parameters
     * use the syntax <tt>default:</tt><i>strategyExpr</i> to say
     * which strategy is the default.
     *
     * <p>Examples:
     * <pre>
     *    tracing:xml
     *    wrapper:somepack.subpack.SomeCommunicationStrategy
     *    dispatching:jabber,default:xml,tracing:xml
     *    dispatching:jabber,xml,default:tracing:xml
     *    random:this,might,be,anything
     * </pre>
     *
     * There's an obvious problem if one of the parameter strategies
     * expects muliple parameters of its own.  For example,
     * "<code>tracing:dispatching:a,b,c</code>" will give
     * the tracing strategy parameters "<tt>dispatching:a</tt>",
     * "<tt>b</tt>", and "<tt>c</tt>".  If this becomes a problem
     * in practice, we'll define a parser that can handle
     * parenthesized subexpressions.
     */
    protected static CommunicationStrategy evalStrategyExpr(String e) {
	String[] parts = Strings.breakAtFirst(":", e);
	String strategyName = parts[0];
	List argStrings = Strings.breakAt(",", parts[1]);
	CommunicationStrategy outer = makeCommunicationStrategy(strategyName);
	if (outer instanceof ParameterizedCommStrategy) {
	    ParameterizedCommStrategy pcs = (ParameterizedCommStrategy)outer;
	    return pcs.apply(Strings.toArray(argStrings));
	}
	else
	    throw new IllegalArgumentException
		("Cannot use " + Strings.quote(strategyName) +
		 " communication strategy with parameters.  " +
		 "Parameters were " + argStrings);
    }

    private static Class findClassForName(String name) {
	Debug.noteln("Looking for class", name);
	try { return Class.forName(name); }
	catch (ClassNotFoundException e) { return null; }
    }

    /**
     * Sends an object to the specified destination using the global
     * communication strategy.  The agents who are sent these objects
     * must be using a compatible strategy.  The recipient should in
     * effect get a copy of the object that was sent, but the exact
     * relationship between the two objects depends on the communication
     * strategy.
     *
     * <p>If the object being sent implements the {@link Sendable}
     * interface and has a null sender-id, the sender-id will be
     * set to the "destination" recorded by the <tt>setupServer</tt>
     * method.  (The sender-id is actually set in a clone of the
     * original object so that the original isn't modified.)
     *
     * @see #setupServer(Object destination, MessageListener listener)
     */
    public static void sendObject(Object destination, Object contents) {
	if (contents instanceof Sendable) {
	    Sendable sendable = (Sendable)contents;
	    if (sendable.getSenderId() == null) {
		Debug.noteln("Filling in sender-id of " + sendable +
			     " to be " + thisAgentAsDestination);
		Sendable copy = (Sendable)Util.clone(sendable);
		copy.setSenderId(Name.valueOf(thisAgentAsDestination));
		contents = copy;
	    }
	}
	fireSendingMessage(destination, contents);
	getCommunicationStrategy().sendObject(destination, contents);
	fireMessageSent(destination, contents);
    }

    /**
     * Does what is required for this agent to receive messages sent
     * using the global communication strategy.  Here the "destination"
     * is the agent in which <tt>setupServer</tt> was called, not a
     * remote agent.  The (remote) agents sending messages to the server
     * must be using a compatible communication strategy.  Note that
     * <tt>setupServer</tt> does not assume that all messages are
     * "sending an object" in the sense of the <tt>sendObject</tt> method.
     *
     * <p>The destination is recorded and will be used by the
     * <tt>sendObject</tt> method to fill in the sender-id of
     * {@link Sendable} objects.
     */
    public static void setupServer(Object destination,
				   MessageListener listener) {
	getCommunicationStrategy().setupServer(destination, listener);
	thisAgentAsDestination = destination;
    }


    /**
     * An object that determines how various IPC operations are
     * performed.
     */
    public static interface CommunicationStrategy {
	public void sendObject(Object destination, Object contents);
	public void setupServer(Object destination,
				MessageListener listener);
    }

    /**
     * An object that represents an incoming message.  The InputMessage
     * provides a place for information that is not part of the message
     * contents but may be used for debugging or for operations such as
     * sending replies.  In later versions, this interface may include
     * methods for accessing such information.
     */
    public static interface InputMessage extends Annotated {
	public Object getContents();
    }

    /**
     * A minimal implementation of InputMessage.
     */
    public static class BasicInputMessage extends AbstractAnnotatedObject
                                          implements InputMessage {
	protected Object contents;
	public BasicInputMessage(Object contents) {
	    this.contents = contents;
	}
	public Object getContents() { return contents; }
    }


    /**
     * An object that is notified when a message is received.
     */
    public static interface MessageListener {
	public void messageReceived(InputMessage message);
    }


    /**
     * A MessageListener that enqueues messages for processing in a
     * separate thread of the listener's own.
     *
     * <p>There are two ways to use this class.  An instance of this class
     * can be used as a wrapper around an existing "inner" MessageListener.
     * Or an instance of a subclass can be used without an inner listener
     * if the subclass redefines the {@link #handleMessage(IPC.InputMessage)}
     * method.
     *
     * <p>Note that in this class the messageReceived method only enqueues
     * a message for later processing, and that it's the handleMessage
     * method that actually processes the message, after its been
     * removed from the queue by the BufferedMessageListener's thread.
     *
     * <p>Here's why the class is useful.  Suppose an ordinary 
     * MessageListener is invoked in a thread T by calling its 
     * messageReceived method.  T cannot do anything else until
     * that call returns.  /\/: ...
     *
     * @see ix.util.MessageQueue
     * @see ix.util.TransferThread
     */
    public static class BufferedMessageListener implements MessageListener {
	protected MessageListener innerListener;
	protected TransferThread thread = new TransferThread() {
	    public void handle(Object message) {
		handleMessage((InputMessage)message);
	    }
	};
	/** 
	 * Constructs an (outer) listener and starts the thread
	 * that will pass messages to the inner listener.
	 */
	public BufferedMessageListener(MessageListener innerListener) {
	    this.innerListener = innerListener;
	    thread.start();
	}
	/**
	 * Constructs a listener without an inner listener.
	 * This should be used only for subclasses that redefine
	 * the {@link #handleMessage(IPC.InputMessage)} method.
	 */
	protected BufferedMessageListener() {
	    this(null);
	}
	/**
	 * Enqueues a message for later processing by the
	 * {@link #handleMessage(IPC.InputMessage)} method, which
	 * will be called in this BufferedMessageListener's thread.
	 */
	public void messageReceived(InputMessage message) {
	    thread.take(message);
	}
	/**
	 * Called in this BufferedMessageListener's thread to process
	 * the the next message from the queue.  The version of the
	 * method in this class does that by calling the inner
	 * listener's messageReceived method, and so it must be
	 * refefined in subclasses that do not have an inner listener.
	 *
	 * @see IPC.MessageListener#messageReceived(IPC.InputMessage)
	 */
	protected void handleMessage(InputMessage message) {
	    innerListener.messageReceived(message);
	}
    }


    /**
     * A CommunicationStrategy that provides "connections" analogous
     * to sockets and a visible mapping from destination names to the
     * data needed to establish a connection.
     */
    public static interface SocketlikeCommunicationStrategy 
      extends CommunicationStrategy {
	public Object getDestinationData(Object destination);
	public void setDestinationData(Object destination, Object data);
	public Connection connectTo(Object destination);
	public Connection getConnection(Object destination);
    }


    /**
     * An object that can send and receive.
     */
    public static interface Connection {
	public Object getDestination();
	public void send(Object contents);
	public Object receive();
	public void close();
    }


    /**
     * A mapping from destination names to the data needed to establish
     * connections with the corresponding agents.
     */
    public static interface DestinationTable {
	public Object get(Object destination);
	public Object put(Object destination, Object data);
    }

    /**
     * A HashMap implementation of the DestinationTable interface.
     */
    public static class BasicDestinationTable extends HashMap
	implements DestinationTable {
	public BasicDestinationTable() { }
    }

    /**
     * A strategy that just throws an exception whenever the
     * agent tries to send.
     */
    public static class EmptyCommunicationStrategy
	   implements CommunicationStrategy {
	public void sendObject(Object destination, Object contents) {
	    throw new IPCException
		("Cannot send to " + destination +
		 " because no communication strategy has been specified.");
	}
	public void setupServer(Object destination,
				MessageListener listener) {
	}
    }

    /**
     * An ObjectStream communication strategy for I-X agents.
     */
    public static class SimpleIXCommunicationStrategy
	extends SerializedCommunicationStrategy {
	public SimpleIXCommunicationStrategy() { 
	}
    }


    /**
     * An XMLObjectStream communication strategy for I-X agents.
     * It encodes the message contents in XML rather than serializing.
     */
    public static class SimpleIXXMLCommunicationStrategy
	extends XMLObjectStreamCommunicationStrategy {
	public SimpleIXXMLCommunicationStrategy() { 
	}
    }


    /**
     * A version of ObjectStreamCommunicationStrategy that encodes
     * the message contents in XML rather than serializing.
     */
    public static class XMLObjectStreamCommunicationStrategy 
	extends SerializedCommunicationStrategy {

	public XMLObjectStreamCommunicationStrategy() { }

	public Object preEncode(Object contents) {
	    return XML.objectToXMLString(contents);
	}

	public Object postDecode(Object contents) {
	    return XML.objectFromXML((String)contents);
	}

    }


    /**
     * The exception thrown by IPC methods.  These exceptions are
     * thrown in place of whatever exceptions are thrown internally
     * (such as IOExceptions) in order to hide the internal details.
     */
    public static class IPCException extends RuntimeException {
	Throwable reason;
	public IPCException() {
	    super();
	}
	public IPCException(String message) {
	    super(message);
	}
	public IPCException(Throwable reason) {
	    super("Caused by " + Debug.describeException(reason));
	    this.reason = reason;
	}
	public IPCException(String message, Throwable reason) {
	    super(message + " caused by " + Debug.describeException(reason));
	    this.reason = reason;
	}
	public Throwable getReason() {
	    return reason;
	}
    }

    public static class BrokenConnectionException extends IPCException {
	public BrokenConnectionException(String message) {
	    super(message);
	}
	public BrokenConnectionException(String message, Throwable reason) {
	    super(message, reason);
	}
	public BrokenConnectionException(Connection c, Throwable reason) {
	    super("Broken connection", reason);
	}
    }

    /* --- IpcListeners ---*/

    public static void addIPCListener(IPCListener listener) {
	ipcListeners.add(listener);
    }

    public static void fireSendingMessage(Object dest, Object contents) {
	for (Iterator i = ipcListeners.iterator(); i.hasNext();) {
	    IPCListener listener = (IPCListener)i.next(); 
	    listener.sendingMessage(dest, contents);
	}
    }

    public static void fireMessageSent(Object dest, Object contents) {
	for (Iterator i = ipcListeners.iterator(); i.hasNext();) {
	    IPCListener listener = (IPCListener)i.next(); 
	    listener.messageSent(dest, contents);
	}
    }

    public static void fireMessageReceived(IPC.InputMessage message) {
	for (Iterator i = ipcListeners.iterator(); i.hasNext();) {
	    IPCListener listener = (IPCListener)i.next(); 
	    listener.messageReceived(message);
	}
    }

}

// Issues:
// * The name SocketlikeCommunicationStrategy may be too specific;
//   what it really does, we might say, is to expose more of the
//   details.  Connections objects are the most socket-like part;
//   the mapping from destination names to more elaborate "information"
//   objects is much more general.
//
// * Consider renaming "destination" to "homeDestination" (or some
//   similar name) when it refers to the agent setting up a server
//   to receive messages rather than referring to a remote agent.
//
// * Should we make the receiving threads daemon threads?
