/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Fri Nov 19 18:11:24 2004 by Jeff Dalton
 * Copyright: (c) 2004, AIAI, University of Edinburgh
 */

package ix.util;

import java.util.*;
import java.io.*;
import java.net.*;

/**
 * Class for useful static I/O methods that don't belong anywhere else.
 */
public final class IO {

    private IO() {}	// block instantiation

    public static void withReader(Reader r, Runnable thunk) {
	withClosable(r, thunk);
    }

    public static void withWriter(Writer w, Runnable thunk) {
	withClosable(w, thunk);
    }

    public static void withInputStream(InputStream s, Runnable thunk) {
	withClosable(s, thunk);
    }

    public static void withOutputStream(OutputStream s, Runnable thunk) {
	withClosable(s, thunk);
    }

    public static void withClosable(final Object closable,
				    final Runnable thunk) {
	Util.run(new WithCleanup() {
	    public void body() { thunk.run(); }
	    public void cleanup() throws IOException { close(closable); }
	});
    }

    private static void close(Object o) throws IOException {
	Debug.noteln("Trying to close", o);
	if (o instanceof Reader)
	    ((Reader)o).close();
	else if (o instanceof Writer)
	    ((Writer)o).close();
	else if (o instanceof InputStream)
	    ((InputStream)o).close();
	else if (o instanceof OutputStream)
	    ((OutputStream)o).close();
	else
	    throw new IllegalArgumentException
		("Cannot close " + o +
		 " because it is not a Reader, Writer, or stream");
    }

}
