/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Mon May 12 14:42:36 2008 by Jeff Dalton
 * Copyright: (c) 2003, 2004, 2007, 2008, AIAI, University of Edinburgh
 */

package ix.util;

import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.*;

import ix.icore.IXAgent;
import ix.util.lisp.Symbol;
import ix.test.SecureRandomTest; // /\/
import ix.test.LongToBytes;	 // /\/

/**
 * Name and ID generator a la gensym.
 *
 * <p>Relevant parameters:
 * <dl>
 *  <dt>allow-random-device=<i>boolean</i>
 *  <dd>Use /dev/random, if it exists
 *  <dt>use-long-ids=<i>boolean</i>
 *  <dt>use-hash-ids=<i>boolean</i>
 * </dl>
 *
 * @see Parameters
 */
public class Gensym {

    private Gensym() {}

    private static Generator defaultGenerator = new Generator();
    private static FutureValue futureRandom = null;

    public static synchronized Generator getDefaultGenerator() {
	return defaultGenerator;
    }

    public static synchronized void setDefaultGenerator(Generator g) {
	defaultGenerator = g;
    }

    /**
     * Calls the default generator's method
     * {@link Gensym.Generator#useUniquePrefix()}.
     */
    public static synchronized void useUniquePrefix() {
	defaultGenerator.useUniquePrefix();
    }

    /**
     * Calls the default generator's method
     * {@link Gensym.Generator#usingUniquePrefix()}.
     */
    public static synchronized boolean usingUniquePrefix() {
	return defaultGenerator.usingUniquePrefix();
    }

    /**
     * Calls the default generator's method
     * {@link Gensym.Generator#nextString(String)}.
     */
    public static synchronized String nextString(String base) {
	return defaultGenerator.nextString(base);
    }

    /**
     * Calls the default generator's method
     * {@link Gensym.Generator#nextName(String)}.
     */
    public static synchronized Name nextName(String base) {
	return defaultGenerator.nextName(base);
    }

    /**
     * Calls the default generator's method
     * {@link Gensym.Generator#nextSymbol(String)}.
     */
    public static synchronized Symbol nextSymbol(String base) {
	return defaultGenerator.nextSymbol(base);
    }

    /**
     * Calls the default generator's method
     * {@link Gensym.Generator#nextNewSymbol(String)}.
     *
     * <p>Remember that the base string might be one that results in an
     * instance of a subclass of Symbol such as {@link ix.util.lisp.Keyword}
     * or {@link ix.util.lisp.ItemVar}.
     */
    public static synchronized Symbol nextNewSymbol(String base) {
	return defaultGenerator.nextNewSymbol(base);
    }

    /**
     * A name and ID generator.
     */
    public static class Generator {

	protected boolean useUnique = false;
	protected boolean useHash =
	    Parameters.getBoolean("use-hash-ids", false);

	protected Map counters = new HashMap();

	protected long initialNumber = 0;

	protected String separator = "-";

	protected String prefix = null;
	
	public Generator() { }

	public void setInitialNumber(long n) {
	    initialNumber = n;
	}

	public void setSeparator(String s) {
	    separator = s;
	}

	public void setPrefix(String s) {
	    prefix = s;
	}

	public boolean usingUniquePrefix() {
	    return useUnique;
	}

	public void useUniquePrefix() {
	    // If it looks like we will have to use the secure random
	    // generator, start asking it now, so the answer might be
	    // ready when we want it.  Some, at least, of the algorithms
	    // can take a noticeably long time to produce their first
	    // value.  /\/: Note that all Generators will use the same
	    // random value.
	    synchronized(Gensym.class) {
		if (futureRandom == null) {
		    if (!haveRandomSourceDevice()) {
			futureRandom = new SecureRandomTest.FutureRandom();
			futureRandom.start();
		    }
		}
	    }
	    // Remember that this generator should use a unique prefix.
	    // /\/: It's unique to this agent/JVM, but not necessarily
	    // to each generator, because the secure random future
	    // produces only one value and the current time (see
	    // makeUniquePrefix) may not have advanced enough.
	    // However, if /dev/random is used, each generator
	    // probably will have a different prefix.
	    useUnique = true;
	    prefix = null;
	}

	private String ensurePrefix() {
	    if (prefix == null) {
		if (useUnique) {
		    prefix = makeUniquePrefix();
		    return prefix;
		}
		else return "";
	    }
	    else return prefix;
	}

	protected String makeUniquePrefix() {
	    return useHash ? makeHashPrefix() : makeLongPrefix();
	}

	protected String makeLongPrefix() {
	    String symName = getSymbolName();
	    return "http://" + getHostName()
		+ (symName == null ? "" : "/" + symName)
		+ "/" + LongToBytes.encode(System.currentTimeMillis())
		+ "/" + LongToBytes.encode(getRandomLong())
		+ "/";
	}

	protected String makeHashPrefix() {
	    IXAgent agent = IXAgent.getAgent();
	    Date startup = agent.getAgentStartupDate();
	    MessageDigest md = makeMessageDigest();
	    md.update(getHostName().getBytes());
	    md.update(agent.getAgentSymbolName().getBytes());
	    md.update(LongToBytes.longToBytes(startup.getTime()));
	    md.update(LongToBytes.longToBytes(System.currentTimeMillis()));
	    md.update(LongToBytes.longToBytes(getRandomLong()));
	    byte[] digest = md.digest();
	    return "http://www.aiai.ed.ac.uk/project/ix/id/"
		+ LongToBytes.bytesToLongStrings(digest)
		+ "/";
	}

	public String nextString(String base) {
	    Long count = (Long)counters.get(base);
	    if (count == null) {
		count = new Long(initialNumber);
		counters.put(base, count);
	    }
	    long c = count.longValue();
	    counters.put(base, new Long(c + 1));
	    return (prefix == null ? ensurePrefix() : prefix)
		+  base + separator + c;
	}

	public Name nextName(String base) {
	    return Name.valueOf(nextString(base));
	}

	public Symbol nextSymbol(String base) {
	    return Symbol.intern(nextString(base));
	}

	public Symbol nextNewSymbol(String base) {
	    // N.B. Uses base as-is if the corresponding symbol
	    // does not already exist.
	    String b = base;
	    while (Symbol.exists(b)) {
		b = nextString(b);
	    }
	    return Symbol.intern(b);
	}

    }

    /*
     * Private utilities
     */

    private static String getHostName() {
	try { return Util.getHostName(); }
	catch (java.net.UnknownHostException e) {
	    throw new RethrownException(e);
	}
    }

    private static String getSymbolName() {
	IXAgent agent = IXAgent.getAgent();
	if (agent != null)
	    return agent.getAgentSymbolName();
	else
	    return null;
    }

    private static boolean haveRandomSourceDevice() {
	if (!Parameters.getBoolean("allow-random-device", false))
	    return false;
	try { return new java.io.File("/dev/random").canRead(); }
	catch (SecurityException e) {
	    Debug.noteException(e);
	    return false;
	}
    }

    private static synchronized Long getLongFromRandomSourceDevice() {
	java.io.FileInputStream f = null;
	try {
	    f = new java.io.FileInputStream("/dev/random");
	    long r = new java.io.DataInputStream(f).readLong();
	    Debug.noteln("Random long from /dev/random " + r);
	    return new Long(r);
	}
	catch (java.io.FileNotFoundException e) {
	    // No need to note this in the debugging output.
	}
	catch (java.io.IOException e) {
	    Debug.noteException(e, false);
	}
	finally {
	    if (f != null) {
		try { f.close(); }
		catch (Exception e) { Debug.noteException(e); }
	    }
	}
	return null;
    }

    /**
     * Returns a random long, from a random source device such
     * as /dev/random if one is known and available, otherwise
     * from a secure random number generator.  This method should
     * be called only once.
     *
     * <p>At present, /dev/random is the only such device known to
     * this class,  and so the secure generator will be used except
     * on Unix systems that provide /dev/random.
     */
    private static long getRandomLong() {
	if (haveRandomSourceDevice()) {
	    Long r = getLongFromRandomSourceDevice();
	    if (r != null)
		return r.longValue();
	}
	// Failed with /dev/random, so try secure random
	synchronized(Gensym.class) {
	    if (futureRandom == null) {
		Debug.noteln("Unexpected null futureRandom");
		futureRandom = new SecureRandomTest.FutureRandom();
		futureRandom.start();
	    }
	}
	Long r = (Long)futureRandom.getValue();
	Debug.noteln("Random long from secure random", r);
	return r.longValue();
    }

    private static MessageDigest makeMessageDigest() {
	try {
	    return MessageDigest.getInstance("MD5");
	}
	catch (NoSuchAlgorithmException e) {
	    throw new RethrownException(e);
	}
    }

    /**
     * For testing.
     */
    public static void main(String[] argv) {
	Parameters.processCommandLineArguments(argv);
	defaultGenerator = new Generator(); // now, so cmd-line args visible
	Debug.noteln("An id", Gensym.nextString("issue"));
	Gensym.useUniquePrefix();
	IXAgent a = new IXAgent() {};
	a.mainStartup(argv);
	Debug.noteln("Unique id", Gensym.nextString("issue"));
    }
    
}
