/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Thu Aug 19 01:56:45 2004 by Jeff Dalton
 * Copyright: (c) 2004, AIAI, University of Edinburgh
 */

package ix.util;

import java.util.*;

/**
 * The transitive closure of a relation (cycles are allowed).
 */
public class FullTransitiveClosure implements TransitiveClosure {

    DirectedGraph baseGraph;

    Map successorMap = new HashMap(); // vertex -> Set of vertices

    public FullTransitiveClosure(DirectedGraph g) {
	this.baseGraph = g;
	computeTransitiveClosure();
    }

    /*
     * DirectedGraph methods
     */

    public Collection getAllNodes() {
	return baseGraph.getAllNodes();
    }

    public Collection getRoots() {
	return baseGraph.getRoots();
    }

    public Collection getSuccessors(Object node) {
	return (Set)successorMap.get(node);
    }

    public boolean isBefore(Object v1, Object v2) {
	return getSuccessors(v1).contains(v2);
    }

    /*
     * The algorithm
     */

    /**
     * Constructs the transitive closure using Warshall's algorithm.
     */
    protected void computeTransitiveClosure() {
	Collection allNodes = baseGraph.getAllNodes();
	// First give each node its direct successors as a Set.
	for (Iterator i = allNodes.iterator(); i.hasNext();) {
	    Object u = i.next();
	    successorMap.put(u, new HashSet(baseGraph.getSuccessors(u)));
	}
	// Construct the transitive closure.
	for (Iterator i = allNodes.iterator(); i.hasNext();) {
	    Object u = i.next();
	    Set su = (Set)successorMap.get(u);
	inner:
	    for (Iterator j = allNodes.iterator(); j.hasNext();) {
		Object v = j.next();
		if (u.equals(v))
		    continue inner;
		Set sv = (Set)successorMap.get(v);
		if (sv.contains(u))
		    sv.addAll(su);
	    }
	}
    }

}
