/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Sat Feb  8 00:54:41 2003 by Jeff Dalton
 * Copyright: (c) 2002, 2003, AIAI, University of Edinburgh
 */

package ix.util;

import java.io.Serializable;
import java.lang.reflect.*;
import java.util.*;

/**
 * Base/utility class for type-safe enumerated values.
 */
public abstract class EnumeratedValue
    implements Serializable, SemiPrimitive {

    protected String name;

    protected static void recordValue(List values, EnumeratedValue value) {
	value.name = value.name.intern();
	if (getValue(values, value.name) == null)
	    values.add(value);
	else
	    throw new Error
		("Attempt to create 2 instances of " + value.getClass()
		 + " with the name " + Util.quote(value.name));
    }

    protected static EnumeratedValue getValue(List values, String name) {
	String unique = name.intern();
	for (Iterator i = values.iterator(); i.hasNext();) {
	    EnumeratedValue val = (EnumeratedValue)i.next();
	    if (val.name == unique)
		return val;
	}
	return null;
    }

    protected static EnumeratedValue 
	      getValueElseError(List values, String name) {
	EnumeratedValue val = getValue(values, name);
	if (val != null)
	    return val;
	else {
	    Object example = values.get(0);
	    String className = 
		Strings.afterLast(".", example.getClass().getName());
	    throw new IllegalArgumentException
		("There is no " + className +
		 " value for name " + Util.quote(name));
	}
    }

    /**
     * Ensures that the correct (==) objects are obtained when
     * deserializing.
     */
    protected Object readResolve() throws java.io.ObjectStreamException {
	// Deserialization magic to avoid multiple (visible)
	// instances of objects that should be ==.  The instances
	// are created (it seems) but then this method is called ...
	try {
	    // Since the valueOf method is static, we either have
	    // to define readResolve in every concrete class or
	    // else use reflection...
	    Class c = this.getClass();
	    Method m = c.getMethod("valueOf", new Class[] {String.class});
	    return m.invoke(this, new Object[] {this.name});
	}
	catch (Exception e) {
	    Debug.noteException(e);
	    // Not sure what to do here ...
	    throw new Error("Can't readresolve because "
			    + Debug.describeException(e));
	}
    }

}
