/* File: Debug.java
 * Contains: Debugging tools
 * Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Created: January 1998
 * Updated: Wed Mar  3 19:17:05 2010 by Jeff Dalton
 * Copyright: (c) 1998 - 2010, AIAI, University of Edinburgh
 */

package ix.util;

import java.util.*;
import java.io.*;
import javax.swing.*;
import java.awt.Component;

import java.lang.reflect.InvocationTargetException;

/** Class for useful static debugging tools */

public abstract class Debug {

    private Debug() { }		// no instantiation


    /*
     * Simple debugging output
     */

    /** 
     * Global on/off control over the debugging output produced by
     * the note and noteln methods.  Initially true.
     */
    public static boolean on = true;

    /**
     * The output destination used by note and noteln.  Do not
     * assign directly to this variable.  Call setNoteStream or
     * setNoteFile instead.  The initial value is System.out.
     */
    public static PrintStream out = System.out;

    private static FileDescriptor outFD = FileDescriptor.out;

    /**
     * True iff the "noteln" methods should give the name of the
     * current thread at the start of the line.  Initially false.
     */
    public static boolean noteThreads = false;

    /**
     * True iff the "noteln" methods should give the name of the
     * calling class and method at the start of the line.
     * Initially false.
     */
    public static boolean noteCaller = false;

    /**
     * The GUI frame to use by default when displaying popups.
     */
    private static Component mainAgentFrame = null;

    private static volatile Fn1<Throwable,String> exceptionDescriber = null;

    /**
     * Turns on debugging output.
     *
     * @return the previous value.
     * @see #off()
     */
    public static boolean on() {
	return setDebug(true);
    }

    /**
     * Turns off debugging output except for exception-reporting.
     *
     * @return the previous value.
     * @see #on()
     */
    public static boolean off() {
	return setDebug(false);
    }

    /**
     * Returns true if debugging is on and false otherwise.
     */
    public static synchronized boolean isOn() {
	return on;
    }

    /**
     * Turns off debugging output on or off.
     *
     * @return the previous value.
     *
     * @see #on()
     * @see #off()
     */
    public static synchronized boolean setDebug(boolean v) {
	boolean was = on;
	on = v;
	return was;
    }

    /**
     * Records the GUI frame to use by default when displaying popups.
     * This frame will be returned by {@link #probableMainFrame()}.
     */
    public static synchronized void recordMainFrame(Component frame) {
        mainAgentFrame = frame;
    }

    public static void setExceptionDescriber(Fn1<Throwable,String> d) {
        exceptionDescriber = d;
    }

    /**
     * Observes parameters as they're set and implements their
     * effects (if any) on debugging output.
     *
     * @see Parameters#setParameter(String pname, String value)
     */
    public static synchronized void processParameter
	                                (String pname, String value) {
	if (pname.equals("debug")) {
	    on = Parameters.getBoolean("debug", on);
	}
	else if (pname.equals("interactive"))
	    Parameters.setIsInteractive
		(Parameters.getBoolean("interactive"));
    }

    /**
     * note writes a string to Debug.out if Debug.on is true.
     * It is intetended to replace System.out.print calls, for
     * debugging output.<p>
     *
     * Unlike noteln, note does not print a newline after the message
     * and does not have variants that take different arguments.
     * The typical use of note is to use several calls to write a line
     * of output.  One potential problem with using note is that
     * a different thread may print some other output between those
     * calls.
     */
    public static void note(String message) {
	if (on) printNote(message);
    }

    /**
     * noteln writes a string, followed by a newline, to Debug.out
     * if Debug.on is true.  It is intended to replace System.out.println
     * calls, for debugging output.  noteln has variants that take more
     * than one argument.  The arguments will be printed as strings
     * with single spaces between.  The aim here is to avoid string
     * concatenation in the calls to noteln, so that the concatenation
     * will not occur when debugging output is turned off.
     */
    public static void noteln(String message) {
	if (on) printlnNote(message);
    }

    public static void noteln(String message, Object whatever) {
	if (on) printlnNote(message + " " + whatever);
    }

    // Some things are not objects, alas

    public static void noteln(Object part1, Object part2) {
	if (on) printlnNote(part1 + " " + part2);
    }

    public static void noteln(String message, int i) {
	if (on) printlnNote(message + " " + i);
    }

    public static void noteln(String message, boolean b) {
	if (on) printlnNote(message + " " + b);
    }

    public static synchronized void notelines(String[] lines) {
	if (on) {
	    for (int i = 0; i < lines.length; i++) {
		printlnNote(lines[i]);
	    }
	}
    }

    /**
     * Produces output even when debug output is off.
     */
    public static void forceln(String message) {
        printlnNote(message);
    }

    // The following messages are synchronized, to ensure that each call
    // to note or noteln completes before a new one starts, and to ensure
    // that Debug.out isn't changed during one of these calls.  The
    // stream classes may already provide enough synchronization for
    // the first case.

    private static synchronized void printNote(String message) {
	out.print(message);
	out.flush();
        outSync();
    }

    private static void outSync() {
        if (outFD != null) {
            try {
                outFD.sync();
            }
            catch (SyncFailedException e) {
                outFD = null;
                // throw new RuntimeException("Sync failed", e);
                noteln("Debug sync failed.");
            }
        }
    }

    private static synchronized void printlnNote(String message) {
	String prefix = "";
	if (noteThreads)
	    prefix = Thread.currentThread().getName() + ": ";
	if (noteCaller) {
	    StackTraceElement[] trace = Thread.currentThread().getStackTrace();
	    for (int i = 0; i < trace.length; i++) {
		StackTraceElement e = trace[i];
		String className = e.getClassName();
		if (! className.equals("ix.util.Debug")) {
		    String methodName = e.getMethodName();
		    if (! (methodName.equals("getStackTrace")
			   && className.equals("java.lang.Thread"))) {
			prefix = prefix + className + "." + methodName + ": ";
			break;
		    }
		}
	    }
	}
	out.println(prefix == "" ? message : prefix + message);
	// Do we need out.flush() when we've called out.println()?
        out.flush();
        outSync();
    }


    /*
     * Debugging output redirection
     */

    /**
     * Sets the output destination for debugging notes.
     */
    public static synchronized void setNoteStream(PrintStream s) {
	out = s;
        outFD = null;
    }

    /**
     * Sets the output destination for debugging notes.
     */
    public static synchronized void setNoteFile(String filename) {
	try {
	    // First, a stream to the file.
	    FileOutputStream fout = new FileOutputStream(filename);
	    // Then a PrintStream with flush-on-newline = true.
	    out = new PrintStream(fout, true);
            outFD = fout.getFD();
	}
	catch (IOException e) {
	    Debug.warn("Failed to setNoteFile to \"" + filename + "\"\n"
		       + "because " + e + "\n"
		       + "so switching to System.out.");
	    out = System.out;
            outFD = FileDescriptor.out;
	}
    }


    /*
     * Special-purpose debugging output routines
     */

    /**
     * Note an exception or error, together with a backtrace.  The note
     * is always printed to System.out or System.err, in addition to
     * Debug.out, even when debug output has been turned off or redirected
     * to a file.
     */
    public static synchronized void noteException(Throwable e) {
	noteException(e, true);
    }

    /**
     * Note an exception or error, optionally with a backtrace.  The note
     * is always printed to System.out or System.err, in addition to
     * Debug.out, even when debug output has been turned off or redirected
     * to a file.
     */
    public static synchronized void noteException(Throwable e,
						  boolean backtrace) {
	String descr;
	try { descr = describeException(e); }
	catch (Throwable t) {
	    descr = e.toString();
	}
	if (on) {
	    noteln("\nException:", descr);
	    if (backtrace) e.printStackTrace(out);
	    noteln("");
	}
	if (!on || out != System.out) {
	    if (noteThreads)
		System.err.println
		    ("\n" + Thread.currentThread().getName() + 
		     ": Exception: " + descr);
	    else
		System.err.println("\nException: " + descr);
	    if (backtrace) e.printStackTrace(System.err);
	    System.err.println("");
	}
    }

    /**
     * Calls {@link #displayException(String, Throwable)}
     * with a null pre-message.
     */
    public static void displayException(Throwable except) {
	displayException(null, except);
    }

    /**
     * Puts up a description of the exception in a message dialog,
     * making sure that this happens in the AWT event thread.
     * If the call is not already in the event thread, it is
     * moved to it by calling SwingUtilities.invokeAndWait.
     *
     * <p>The pre-message, if not null, is prefixed to a description
     * of the exception, with ": " as a separator.
     *
     * <p>Calls {@link #noteException(Throwable)} first (and before
     * any thread change) so you don't have to.  That ensures that
     * a backtrace is printed.</p>
     *
     * <p>If {@link Parameters#isInteractive()} returns false,
     * none of the above applies, and the method simply calls
     * {@link #noteException(Throwable)}.
     *
     * @see #describeException(Throwable)
     */
    public static void displayException(final String preMessage,
					final Throwable except) {
	if (preMessage != null)
	    note(preMessage + ": ");
	noteException(except);
	if (!Parameters.isInteractive()) // no GUI, maybe even no user
	    return;
	if (SwingUtilities.isEventDispatchThread())
	    do_displayException(preMessage, except);
	else {
	    try {
		SwingUtilities.invokeAndWait(new Runnable() {
		    public void run() {
			do_displayException(preMessage, except);
		    }
		});
	    }
	    catch (Throwable t) {
		noteln("Exception in displayException method");
		noteException(t);
	    }
	}
    }

    private static void do_displayException(String preMessage,
					    Throwable except) {
	String descr = 
	    (preMessage == null ? "" : preMessage + ": ")
	    + Debug.describeException(except);
	JOptionPane.showMessageDialog
            (probableMainFrame(),
             new Object[] {Strings.foldLongLine(descr)},
             "Exception",
             JOptionPane.ERROR_MESSAGE);
    }

    /**
     * Returns the frame that is likely to be the agent's main GUI frame.
     *
     * @see #recordMainFrame(JFrame)
     */
    public static synchronized Component probableMainFrame() {
        return mainAgentFrame;
    }

    /**
     * Returns a relatively readable description of an exception.
     * Since an error or exception's message is not always sufficiently
     * meaningful on its own, it is prefixed by the class name (minus
     * the package name).  If the exception is an InvocationTargetException,
     * a description of its target exception is included.
     */
    public static String describeException(Throwable t) {
        if (exceptionDescriber != null)
            return exceptionDescriber.funcall(t);
        else
            return defaultDescribeException(t);
    }

    public static String defaultDescribeException(Throwable t) {
	String className = t.getClass().getName();
	String message = t.getMessage();
	String classPrefix =
	    // Remove package and, if an inner class, the names of
	    // the enclosing classes.  /\/: Wrong for "anonymous"
	    // classes, because they end up with names like
	    // this: "ix.ip2.ItemEditor$1".
	    Strings.afterLast("$",
			      Strings.afterLast(".", className))
	    + ": ";
	// /\/: Can RethrownException be made more similar to other
	// "chained exceptions"?
	if (t instanceof RethrownException) {
	    // /\/: Ignore classPrefix
	    return message;
	}
	Throwable cause = t.getCause();
	if (cause == null) {
	    return classPrefix + message;
	}
	// Non-null cause - looks like we have part of an exception chain.
	else if (message != null) {
	    return classPrefix + message + " - " + describeException(cause);
	}
	// Null message
	else if (t instanceof InvocationTargetException) {
	    // Omit any mention of InvocationTargetException in this case.
	    return describeException(cause);
	}
	else {
	    return classPrefix + describeException(cause);
	}
    }

    /**
     * Calls {@link #describeException(Throwable)} then breaks the result
     * into lines if it would otherwise be too long.
     *
     * @see Strings#foldLongLine(String)
     */
    public static String foldException(Throwable t) {
	return Strings.foldLongLine(describeException(t));
    }

    /** 
     * Use this to tell the user about minor problems.  Warn prints
     * a message to System.err, followed by a backtrace for the current
     * thread, and also displays the message in a dialog.  If method
     * {@link Parameters#isInteractive()} returns false, the dialog
     * does not occur and instead a RuntimeException is thrown.
     */
    public static void warn(String message) {
	System.err.println("\nWarning: " + message + "\n");
	Thread.dumpStack();
	System.err.println("");
	if (Parameters.isInteractive())
	    Util.displayAndWait(null,
				Strings.foldLongLine("Warning: " + message));
	else
	    throw new Warning(message);
    }

    /**
     * Use this to tell the user about problems that should not be
     * ignored and are not handled locally.
     *
     * @throws RuntimeException as notification of the problem.
     * @deprecated  As of I-X 4.0, just throw instead.
     */
    public static void error(String message) {
	System.err.println("\nError: " + message + "\n");
	throw new RuntimeException(message);
    }


    /**
     * Numbers and prints the elements of an Enumeration on separate lines.
     */
    public static synchronized void noteEnumeration(Enumeration e) {
	Debug.note("[");
	for (int i = 0; e.hasMoreElements(); i++) {
	    Debug.note(i + ": " + e.nextElement() + "\n ");
	}
	Debug.noteln("]");
    }

    /**
     * Prints the elements of an Enumeration on separate lines, with
     * an index number and and class name at the start of each line.
     */
    public static synchronized void noteEnumerationClasses(Enumeration e) {
	Debug.note("[");
	for (int i = 0; e.hasMoreElements(); i++) {
	    Object elt = e.nextElement();
	    Debug.note(i + " " + elt.getClass() + ": " + elt + "\n ");
	}
	Debug.noteln("]");
    }

    /**
     * Describes differences to Debug.out, showing element classes
     * when elements are different.
     */
    public static void showListDifferences(List l1, List l2) {
	int s1 = l1.size(), s2 = l2.size();
	if (s1 > s2)
	    Debug.out.println("First list is longer");
	else if (s2 > s1)
	    Debug.out.println("Second list is longer");
	Iterator i1 = l1.iterator();
	Iterator i2 = l2.iterator();
	int i = 0;
	while (i1.hasNext()) {
	    Object e1 = i1.next();
	    Object e2 = i2.next();
	    if (!e1.equals(e2))
		Debug.out.println
		    (i + ": " + e1.getClass() + " " + e1 +
		     " != " + e2.getClass() + " " + e2);
	}
    }

    /**
     * Prints the elements of an enumeration on separate lines with
     * a specified prefix at the start of each line.
     */
    public static synchronized void noteElements(Enumeration e,
						 String prefix) {
	while (e.hasMoreElements())
	    Debug.noteln(prefix, e.nextElement());

    }


    /*
     * Assertions
     */

    /** 
     * expect checks a condition that should always be true and
     * throws an AssertionFailure if it is not.  expect was originally
     * called "assert", but "assert" became a keyword in Java 1.4.<p>
     *
     * An assertion is typically used when all of the following apply:
     * the subsequent code requires a condition to be true; in the absence
     * of bugs, the condition always will be true at that point; and it is
     * not obvious that the condition will be true.<p>
     *
     * AssertionFailure is a RuntimeException and so does not need to be
     * listed in the "throws" clauses of method definitions.  One reason
     * for that is to avoid discouraging the use of assertions.  If
     * AssertionFailure had to be declared, then adding an assertion
     * in a method that had none before would require nonlocal changes
     * in the code.
     *
     * @throws AssertionFailure as notification of the problem
     * @see AssertionFailure
     */
    public static void expect(boolean cond) {
	if (!cond) {
	    noteln("Assertion failed.");
	    throw new AssertionFailure();
	}
    }

    /**
     * A variant that allows a message that describes the assertion.
     * The message will be printed when the assertion fails and will
     * be included in the AssertionFailure exception.
     */
    public static void expect(boolean cond, String message) {
	if (!cond) {
	    noteln("Assertion failed:", message);
	    throw new AssertionFailure(message);
	}
    }

    /**
     * A variant that allows a message that describes the assertion
     * plus an Object that the message is about.
     */
    public static void expect(boolean cond, String message, Object item) {
	if (!cond) {
	    String itemMessage = message + " " + item;
	    noteln("Assertion failed:", itemMessage);
	    throw new AssertionFailure(itemMessage);
	}
    }

    /**
     * Tests whether an object equals the expected value.  By convention,
     * the first argument is the expected value, and the second is the
     * value actually obtained.
     * 
     * @throws AssertionFailure  if the objects aren't equal.
     */
    public static void expectEquals(Object expected, Object found) {
	if (!expected.equals(found)) {
	    String message = "Expected " + expected + ", but found " + found;
	    noteln("Assertion failed:", message);
	    throw new AssertionFailure(message);
	}
    }

    /**
     * Tests whether an object equals the expected value.  By convention,
     * the first argument is the expected value, and the second is the
     * value actually obtained.
     * 
     * @throws AssertionFailure  if the objects aren't equal.
     */
    public static void expectEquals(Object expected, Object found,
				    String messagePrefix) {
	if (!expected.equals(found)) {
	    String message = messagePrefix +
		" Expected " + expected + ", but found " + found;
	    noteln("Assertion failed:", message);
	    throw new AssertionFailure(message);
	}
    }

    /**
     * Tests whether an object is == to the expected value.  By convention,
     * the first argument is the expected value, and the second is the
     * value actually obtained.
     * 
     * @throws AssertionFailure  if the objects aren't ==.
     */
    public static void expectSame(Object expected, Object found) {
	if (expected != found) {
	    String message = "Expected " + expected + ", but found " + found;
	    noteln("Assertion failed:", message);
	    throw new AssertionFailure(message);
	}
    }

    /**
     * Tests whether an object is == to the expected value.  By convention,
     * the first argument is the expected value, and the second is the
     * value actually obtained.
     * 
     * @throws AssertionFailure  if the objects aren't ==.
     */
    public static void expectSame(Object expected, Object found,
				  String messagePrefix) {
	if (expected != found) {
	    String message = messagePrefix +
		" Expected " + expected + ", but found " + found;
	    noteln("Assertion failed:", message);
	    throw new AssertionFailure(message);
	}
    }


}

/* ---------------------------- Change History ----------------------------
 *  (Who)   (When)                   (What)
 *
 */
