/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Wed May 28 11:58:37 2008 by Jeff Dalton
 * Copyright: (c) 2004, 2005, 2007, 2008, AIAI, University of Edinburgh
 */

package ix.test.xml;

import java.util.*;

import ix.icore.plan.Plan;
import ix.icore.domain.Domain;

import ix.icore.process.PNode;
import ix.icore.process.PNodeEnd;

import ix.icore.domain.End;

import ix.iplan.AutoTester;
import ix.iplan.PlanTest;
import ix.iplan.PlanCheckingSimulator;
import ix.iplan.ExecutionStages;
import ix.iplan.Slip;		// for some symbols /\/ and for its MM

import ix.ip2.Ip2ModelManager;

import ix.test.BlockStacker;
import ix.test.HtnBlockStacker;
import ix.test.PrecondBlockStacker;
import ix.test.SimpleBlockSim;

import ix.util.*;
import ix.util.lisp.*;

/**
 * Describes a block-stacking plan test.
 */
public class BlockStackingTest extends PlanTest {

    String problem;

    public BlockStackingTest() {
        super();
    }

    public String getProblem() {
	return problem;
    }

    public void setProblem(String problem) {
	this.problem = problem;
    }

    @Override
    public void setTask(String task) {
	throw new UnsupportedOperationException();
    }

    @Override
    public void setInitialPlan(String initialPlan) {
	throw new UnsupportedOperationException();
    }

    @Override
    public String testDescription() {
	return "Domain " + getDomain() +
	    ", Problem " + getProblem();
    }

    @Override
    public String taskDescription() { // needed by savePlan
	return getProblem();
    }

    @Override
    public AutoTester.TestRunner makeTestRunner(AutoTester auto) {
	return new BlockTestRunner(this, auto);
    }

    static class BlockTestRunner extends AutoTester.TestRunner {

	AutoTester auto;
	BlockStacker stacker;
	BlockStackingTest test;

	BlockTestRunner(BlockStackingTest test, AutoTester auto) {
	    auto.super(test);
	    this.auto = auto;
	    this.test = test;
	}

	// /\/: Relies on readDomain() being called before initialPlan(),
	// because we don't know what kind of BlockStacker to create
	// until we look at the domain.

	@Override
	protected Domain readDomain() {
	    Domain baseDomain = super.readDomain();
	    stacker = makeBlockStacker(baseDomain);
	    Domain domain = stacker.problemDomain();
	    domain.takeFrom(baseDomain);
	    return domain;
	}

	private BlockStacker makeBlockStacker(Domain dom) {
	    BlockStacker stacker =
		dom.getAnnotation(Slip.S_ACHIEVABLE_CONDS) == Slip.K_NONE
		    ? new HtnBlockStacker(test.getProblem())
		    : new PrecondBlockStacker(test.getProblem());
	    stacker.setBaseDomain(dom);
	    return stacker;
	}

	@Override
	protected Plan initialPlan() {
	    return stacker.initialPlan();
	}

	@Override
	protected void checkPlan(Slip slip) {
	    checkBlockStackingMoves(slip);
	    super.checkPlan(slip);
	}

	@Override
	protected Map runSimulation(PlanCheckingSimulator sim) {
	    Map result = super.runSimulation(sim);
	    List fails = stacker.failedGoalPatterns(sim.getWorldStateMap());
	    if (!fails.isEmpty())
		auto.testFailure(test, "False patterns " + fails);
	    else
		auto.traceln("World state is correct");
	    runSimpleBlockSim(sim);
	    return result;
	}

	protected void runSimpleBlockSim(PlanCheckingSimulator sim) {
	    // The super runSimulation method must have already been called
	    // so that the PlanCheckingSimulator will have put the node-ends
	    // into an execution order.
	    SimpleBlockSim blockSim = new SimpleBlockSim(stacker);
	    blockSim.simulatePlan(sim.getExecutionOrder());
	    if (blockSim.isFinished())
		auto.traceln("Block-moving simulation succeeded");
	    else
		auto.testFailure
		    (test, "block-moving simulation failed");
	}

	protected void checkBlockStackingMoves(Slip slip) {
	    Ip2ModelManager mm = slip.getIp2ModelManager();
	    List<PNodeEnd> allNodeEnds = mm.getNodeEnds();
	    List<List<PNodeEnd>> stages =
		new ExecutionStages(allNodeEnds).getStages();
	    List<String> problems = new LinkedList<String>();
	    PNode previousMove = null;
	    PNodeEnd moveBegin = null;
	stageLoop:
	    for (List<PNodeEnd> stage: stages) {
		PNodeEnd moveBeginInThisStage = null;
	    nodeLoop:
		for (PNodeEnd ne: stage) {
		    LList pattern = ne.getNode().getPattern();
		    if (! pattern.get(0).toString().startsWith("move"))
			continue nodeLoop;
		    if (ne.getEnd() == End.BEGIN) {
			if (moveBeginInThisStage == null)
			    moveBeginInThisStage = ne;
			else
			    problems.add
				(twoMovesBeginInSameStage
				     (moveBeginInThisStage, ne));
			if (moveBegin == null)
			    moveBegin = ne;
			else
			    problems.add(overlappingMoves(moveBegin, ne));
			if (previousMove != null) {
			    PNode thisMove = ne.getNode();
			    if (movingSameBlock(previousMove, thisMove))
				problems.add
				    (blockMovedTwice(previousMove, thisMove));
			}
		    }
		    else {
			Debug.expect(ne.getEnd() == End.END);
			Debug.expect(moveBegin != null, "missing move begin");
			if (moveBegin == ne.getOtherNodeEnd()) {
			    previousMove = moveBegin.getNode();
			    moveBegin = null;
			}
			else
			    problems.add(moveEndsMismatch(moveBegin, ne));
		    }
		}
	    }
	    if (! problems.isEmpty()) {
		int n = problems.size();
		auto.trace("block-move check finds ");
		auto.traceln(n == 1 ? "1 problem:" : n + " problems:");
		for (String problem: problems) {
		    auto.traceln("   ", problem);
		}
		auto.testFailure(test, "problems in block moves");
	    }
	}

	private boolean movingSameBlock(PNode move1, PNode move2) {
	    return getBlockMoved(move1) == getBlockMoved(move2);
	}

	private Symbol getBlockMoved(PNode move) {
	    return (Symbol)move.getPattern().get(1);
	}

	private String twoMovesBeginInSameStage(PNodeEnd older,
						PNodeEnd newer) {
	    return "two moves begin in same stage: "
		+  older.getNode() + " and " + newer.getNode();
	}

	private String overlappingMoves(PNodeEnd older, PNodeEnd newer) {
	    return "overlapping moves "
		+  older.getNode() + " and " + newer.getNode();
	}

	private String blockMovedTwice(PNode move1, PNode move2) {
	    return "block moved twice in a row by " + move1 + " and " + move2;
	}

	private String moveEndsMismatch(PNodeEnd begin, PNodeEnd end) {
	    return "mismatched ends " + begin + " and " + end;
	}

    }

}
