/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Tue Nov  6 02:22:38 2007 by Jeff Dalton
 * Copyright: (c) 2007, AIAI, University of Edinburgh
 */

package ix.test;

import java.util.*;

import ix.iface.domain.LTF_Parser;
import ix.iface.domain.LTF_Writer;

import ix.icore.domain.*;

import ix.util.*;
import ix.util.lisp.*;

public class OrderingNotationTest {

    public static void main(String[] argv) {

	LTF_Parser parser = new LTF_Parser();
	LTF_Writer writer = new LTF_Writer();

	for (;;) {
	    try {

		// Ask the user for some orderings in LTF's notation.
		String lines = Util.askLines("\nOrderings:");
		if (lines.equals("bye") || lines.equals(""))
		    return;
		LList notation = Lisp.elementsFromString(lines);

		// Expand the notation into pairs.
		LList expanded = parser.expandAllOrderings(notation);
		System.out.println("Expansion: " + expanded);

		// Convert the pairs to Orderings.
		ListOfOrdering orderings = parser.makeOrderings(expanded);

		// Now go back the other way.
		List unexpanded = 
		    unexpandOrderings((List<Ordering>)orderings);
		System.out.println("\nUnexpanded:\n");
		for (Object seq: unexpanded) {
		    System.out.println("   " + seq);
		}

	    }
	    catch (Throwable t) {
		Debug.noteException(t);
	    }
	}

    }

    /**
     * Turns a list of Orderings into the notation used in LTF.
     */
    static List unexpandOrderings(List<Ordering> orderings) {
	ListIterator<Ordering> i = orderings.listIterator();
	LinkedList<List> result = new LinkedList<List>();
	while (i.hasNext()) {
	    unexpandFront(i, result);
	}
	return result;
    }

    private static void unexpandFront(ListIterator<Ordering> i,
				      LinkedList<List> result) {
	Debug.expect(i.hasNext());
	Ordering ord_1 = i.next();
	if (! i.hasNext()) {
	    addOrdering(result, ord_1);
	    return;
	}
	Ordering ord_2 = i.next();
	if (ord_1.getFrom().equals(ord_2.getFrom())) {
	    fanOut(ord_1, ord_2, i, result);
	}
	else if (ord_1.getTo().equals(ord_2.getTo())) {
	    fanIn(ord_1, ord_2, i, result);
	}
	else {
	    addOrdering(result, ord_1);
	    i.previous();	// next time, look at ord_2 again
	}
    }

    private static void fanOut(Ordering ord_1, Ordering ord_2,
			       ListIterator<Ordering> i,
			       LinkedList<List> result) {
	Debug.expect(ord_1.getFrom().equals(ord_2.getFrom()));
	NodeEndRef from = ord_1.getFrom(); // fans out from this end
	List<NodeEndRef> to = Collect.makeList(ord_1.getTo(), ord_2.getTo());
	while (i.hasNext()) {
	    Ordering next = i.next();
	    if (next.getFrom().equals(from))
		to.add(next.getTo());
	    else {
		i.previous();	// back up
		break;
	    }
	}
	addOrdering(result, from, to);
    }

    private static void fanIn(Ordering ord_1, Ordering ord_2,
			      ListIterator<Ordering> i,
			      LinkedList<List> result) {
	Debug.expect(ord_1.getTo().equals(ord_2.getTo()));
	NodeEndRef to = ord_1.getTo(); // fans in to this end
	List<NodeEndRef> from = 
	    Collect.makeList(ord_1.getFrom(), ord_2.getFrom());
	while (i.hasNext()) {
	    Ordering next = i.next();
	    if (next.getTo().equals(to))
		from.add(next.getFrom());
	    else {
		i.previous();	// back up
		break;
	    }
	}
	addOrdering(result, from, to);
    }

    private static void addOrdering(LinkedList<List> result, Ordering ord) {
	addOrdering(result, ord.getFrom(), ord.getTo());
    }

    private static void addOrdering(LinkedList<List> result,
				    Object from, // NodeEndRef or NER list
				    Object to) { // NodeEndRef or NER list
	from = orderingElementNotation(from, End.END);
	to = orderingElementNotation(to, End.BEGIN);
	Debug.noteln("Adding " + from + " --> " + to);
	if (result.isEmpty())
	    result.add(Collect.makeList(from, to));
	else {
	    List seq = result.getLast();
	    Object finalTo = Collect.lastElement(seq);
	    if (finalTo.equals(from))
		seq.add(to);
	    else
		result.add(Collect.makeList(from, to));
	}
    }

    private static Object orderingElementNotation(Object elt, End expectedEnd) {
	if (elt instanceof NodeEndRef)
	    return nodeEndNotation((NodeEndRef)elt, expectedEnd);
	else if (elt instanceof List) {
	    List result = new LinkedList((List)elt);
	    for (ListIterator i = result.listIterator(); i.hasNext();) {
		NodeEndRef ref = Util.mustBe(NodeEndRef.class, i.next());
		i.set(nodeEndNotation(ref, expectedEnd));
	    }
	    return result;
	}
	else
	    throw new ConsistencyException("Bad ordering element", elt);
    }

    private static Name nodeEndNotation(NodeEndRef ref, End expectedEnd) {
	if (ref.getEnd() == expectedEnd)
	    return ref.getNode();
	else
	    return Name.valueOf	((ref.getEnd() == End.BEGIN ? "b/" : "e/")
				 + ref.getNode());
    }

}
