/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Thu Nov 11 19:51:19 2004 by Jeff Dalton
 * Copyright: (c) 2004, AIAI, University of Edinburgh
 */

package ix.test;

import java.util.*;

import ix.ip2.*;
import ix.util.*;
import ix.util.lisp.*;

/**
 * Produces a distance table from state information in a plan.
 * As a standalone application, this class reads in a plan,
 * looks at type assertions in the state to work out what things
 * are "places", then produces a full distance table for those
 * places, using any existing distance values in the state,
 * with "LONG" as the default (so that it's distinct from a "long"
 * that was already in the table but can easily be turned into "long"
 * in an editor).
 */
public class MakeDistanceTable {

    static final Symbol
	TRANSPORT_DISTANCE = Symbol.intern("transport-distance"),
	TYPE = Symbol.intern("type");

    List placeTypes =
	Lisp.elementsFromString
	       ("location airport hospital aircraft-carrier");

    MakeDistanceTable() { }

    void outputTable(Map state) {
	SortedSet places = findAllPlaces(state);
	println(";;; Distance table for " + places.size() + " places.");
	for (Iterator i = places.iterator(); i.hasNext();) {
	    Object from = i.next();
	    for (Iterator j = places.iterator(); j.hasNext();) {
		Object to = j.next();
		if (from != to)
		    outputDistance(state, from, to);
	    }
	}
    }

    void outputDistance(Map state, Object from, Object to) {
	// Some distances might already be known.
	LList query = Lisp.list(TRANSPORT_DISTANCE, from, to);
	LList rquery = Lisp.list(TRANSPORT_DISTANCE, to, from);
	Object forwardDist = state.get(query);
	Object backDist = state.get(rquery);
	if (forwardDist != null) {
	    if (backDist == forwardDist || backDist == null)
		outDist(from, to, forwardDist);
	    else
		outDist(from, to, forwardDist + " ;; or " + backDist + "?");
	}
	else if (backDist != null) {
	    outDist(from, to, backDist);
	}
	else {
	    outDist(from, to, "LONG"); // "LONG" indicates a default
	}
    }

    void outDist(Object from, Object to, Object value) {
	println("transport-distance " + from + " " + to + " = " + value);
    }

    SortedSet findAllPlaces(Map state) {
	SortedSet result = new TreeSet();
	for (Iterator i = state.entrySet().iterator(); i.hasNext();) {
	    Map.Entry e = (Map.Entry)i.next();
	    LList pat = (LList)e.getKey();
	    Object value = e.getValue();
	    if (pat.get(0) == TYPE && placeTypes.contains(value)) {
		Symbol place = (Symbol)pat.get(1);
		result.add(place);
	    }
	}
	return result;
    }

    void println(String s) {
	System.out.println(s);
    }

    public static void main(String[] argv) {

	Debug.off();

	// Load the plan.
	Ip2 ip2 = new ix.test.PlainIp2();
	ip2.mainStartup(argv);	// should load a plan

	// Get the world state.
	Ip2ModelManager mm = (Ip2ModelManager)ip2.getModelManager();
	Map state = mm.getWorldStateMap();

	new MakeDistanceTable().outputTable(state);

    }

}
