/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Thu Jan 29 17:24:36 2009 by Jeff Dalton
 * Copyright: (c) 2003, AIAI, University of Edinburgh
 */

package ix.test;

import java.util.*;
import java.io.*;

import java.math.BigInteger;

import ix.util.*;
import ix.util.lisp.*;

public class LongToBytes {

    public static void main(String[] argv) {
	for (;;) {
	    String in = Util.askLine("Long:");
	    if (in.equals("bye"))
		return;
	    Long L = (Long)Lisp.readFromString(in);
	    long l = L.longValue();
	    byte[] bytes = longToBytes(l);
	    System.out.println(printName(bytes));
	    System.out.println("= " + bytesToLong(bytes));
	    System.out.println("= " + bytesToLongStrings(bytes));
	    System.out.println("= " + encode(l));
            System.out.println("= " + decode(encode(l)));
	    System.out.println("");
	}
    }

    static String printName(byte[] bytes) {
	StringBuffer buf = new StringBuffer();
	buf.append("byte[")
	   .append(bytes.length)
	   .append("]{");
	for (int i = 0; i < bytes.length - 1; i++) {
	    buf.append(bytes[i]).append(", ");
	}
	buf.append(bytes[bytes.length - 1])
           .append("}");
	return buf.toString();
    }

    /**
     * Converts a long to a (big-endian) array of bytes.
     */
    public static byte[] longToBytes(long l) {
	byte[] bytes = new byte[8];
	int shift = 56, i = 0;
	while (shift >= 0) {
	    long shifted = l >> shift;
	    bytes[i] = (byte)(shifted & 255);
	    shift -= 8;
	    i++;
	}
	return bytes;
    }

    /**
     * Converts a (big-endian) array of bytes to a long
     */
    public static long bytesToLong(byte[] bytes) {
	long l = 0;
	for (int i = 0; i < bytes.length; i++) {
	    l = l << 8;
	    l |= (bytes[i] & 255);
	}
	return l;
    }

    /**
     * Converts an array of bytes to a string containing the
     * "/"-separated string representations of longs taken from
     * the array.  This is done by wrapping a DataInputStream
     * around a ByteArrayInputStream based on the array, to
     * get the longs, and calling {@link #encode(long)}
     * to get the string representations.
     *
     * <p>It is assumed that the length of the byte array
     * is a multiple of 8.
     */
    public static String bytesToLongStrings(byte[] bytes) {
	DataInputStream in =
	    new DataInputStream(new ByteArrayInputStream(bytes));
	StringBuffer result = new StringBuffer();
	try {
	    while (in.available() > 0) {
		if (result.length() > 0)
		    result.append("/");
		result.append(encode(in.readLong()));
	    }
	}
	catch(IOException e) {
	    throw new RethrownException(e);
	}
	return result.toString();
    }

    public static String encode(long l) {
	Debug.expect(DIGITS.length() == 64);
        if (l == 0)
            return "0";
	boolean negate = l < 0;
	StringBuilder result = new StringBuilder();
	BigInteger L = BigInteger.valueOf(l);
	BigInteger Zero = BigInteger.valueOf(0);
	BigInteger SixtyFour = BigInteger.valueOf(64);
	if (negate)
	    L = L.negate();
	while (L.compareTo(Zero) > 0) {
	    BigInteger[] divRem = L.divideAndRemainder(SixtyFour);
	    int rem = divRem[1].intValue();
	    result.insert(0, DIGITS.charAt(rem));
	    L = divRem[0];
	}
	if (negate)
	    result.insert(0, "-");
	return result.toString();
    }

    public static long decode(String s) {
	Debug.expect(DIGITS.length() == 64);
        boolean negate = s.startsWith("-");
        int i = negate ? 1 : 0;
        int len = s.length();
        long result = 0;
        while (i < len) {
            char digit = s.charAt(i++);
            // int value = DIGITS.indexOf(digit); // slow /\/
            int value = DIGIT_VALUE_TABLE.get(digit); // slow? /\/
            if (value < 0)
                throw new ConsistencyException
                    ("Can't decode digit " + digit);
            result = result * 64 + value;
        }
        return negate ? -result : result;
    }

    private static final String DIGITS =
	"0123456789abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ~_";

    private static final Map<Character,Integer> DIGIT_VALUE_TABLE =
        new HashMap<Character,Integer>();
    static {
        int len = DIGITS.length();
        for (int i = 0; i < len; i++)
            DIGIT_VALUE_TABLE.put(DIGITS.charAt(i), i);
    }

}
