/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Tue Aug 22 18:15:03 2006 by Jeff Dalton
 * Copyright: (c) 2001 - 2006, AIAI, University of Edinburgh
 */

package ix.itest;

import java.awt.Color;
// import java.awt.GridLayout;
// import java.awt.FlowLayout;
import java.awt.CardLayout;
import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.event.*;

import javax.swing.*;

import javax.swing.plaf.metal.MetalLookAndFeel;	// for colours

import java.util.*;

import ix.iface.util.RadioButtonBox;
import ix.iface.util.ComboChoice;
import ix.iface.util.GridColumn;
import ix.iface.util.CatchingActionListener;

import ix.util.xml.XML;		// for debugging output

// import ix.ispace.AgentData;
import ix.ispace.GroupSender;
import ix.ispace.ContactManager;
import ix.ispace.event.*;
import ix.ichat.*;

import ix.icore.*;
import ix.icore.domain.Constraint;
import ix.icore.domain.PatternAssignment;

import ix.iface.domain.LTF_Parser; // for constraint parsing
import ix.iface.domain.SyntaxException;

import ix.iface.util.VerticalPanel;

import ix.util.*;
import ix.util.lisp.*;


/**
 * I-Test and Messenger panel for sending messages.
 */
public class SendPanel extends VerticalPanel implements ActionListener {

    private static Symbol MESSAGE_COLOR = Symbol.intern("message-color");

    IXAgent agent;
    ContactManager contactManager;

    ActionListener actionListener
	= CatchingActionListener.listener(this);

    SendTextArea contentText = new SendTextArea(5, 40);

    RadioButtonBox typeSelector = makeTypeControl();
    RadioButtonBox prioritySelector = makePriorityControl();

    SendControlsPanel sendControls; // needs agent

    JButton styleButton = new JButton("Style");

    // Shared models
    JToggleButton.ToggleButtonModel reportCheckModel =
	new JToggleButton.ToggleButtonModel();
    DefaultComboBoxModel destinationChoiceModel =
	new DefaultComboBoxModel(new String[]{"me", "a group ..."});

    public SendPanel(IXAgent agent) {
	this.agent = agent;
	this.contactManager = agent.getContactManager();

	// Add all currently known agent names.
	List names = contactManager.getSortedNameList();
	for (Iterator i = names.iterator(); i.hasNext();) {
	    String name = (String)i.next();
	    destinationChoiceModel.addElement(name);
	}

	sendControls = new SendControlsPanel();
	// agent.getContactManager().addContactListener(this);
//  	setLayout(new BoxLayout(this, BoxLayout.Y_AXIS));
	setBorder(BorderFactory.createTitledBorder("Compose Message"));
	add(new JScrollPane(contentText));
	addFixedHeight(makeControlPanel());

	contactManager.setAgentColor(agent.getAgentSymbolName(), (Color)null);
	styleButton.setForeground(Color.black);
	styleButton.addActionListener(actionListener);

	typeSelector.setSelection("Chat Message");
    }

    public void actionPerformed(ActionEvent event) {
	String command = event.getActionCommand();
	Debug.noteln("SendPanel action:", command);
	if (command.equals("Send")) {
	    sendTo(sendControls.getSelected().getDestination());
	}
	else if (command.equals("Style")) {
	    editMessageColor();
	}
	else
	    throw new ConsistencyException
		("Nothing to do for", command);
    }

    public void editMessageColor() {
	String name = agent.getAgentSymbolName();
	Color color = JColorChooser.showDialog
	    (this, "Color for " + name + " messages", 
	           contactManager.getAgentColor(name, Color.black));
	if (color == null)
	    return;		// user cancelled
	else {
	    contactManager.setAgentColor(name, color);
	    styleButton.setForeground(color);
	}
    }

    class SendTextArea extends JTextArea {

	boolean inSentState = false;

	SendTextArea(int rows, int columns) {
	    super(rows, columns);
	    addKeyListener(new KeyAdapter() {
		public void keyPressed(KeyEvent e) {
		    if (inSentState)
			clearSent();
		}
	    });
	    addMouseListener(new MouseAdapter() {
		public void mousePressed(MouseEvent e) {
		    if (inSentState) {
			clearSent();
			setSelectionStart(0);
			setSelectionEnd(getText().length());
		    }
		}
	    });
	}

	public void setText(String t) {
	    if (inSentState) clearSent();
	    super.setText(t);
	}

	void showSent() {
	    setForeground(MetalLookAndFeel.getControlDisabled());
	    inSentState = true;
	}

	void clearSent() {
	    setForeground(MetalLookAndFeel.getBlack());
	    inSentState = false;
	}

    }

    public void initIssue(Issue issue) {
	typeSelector.setSelection("Issue");
	((IssueControls)sendControls.mustBe("Issue"))
	    .initItem(issue);
    }

    public void initActivity(Activity act) {
	typeSelector.setSelection("Activity");
	((ActivityControls)sendControls.mustBe("Activity"))
	    .initItem(act);
    }

    public void transformToActivity(Issue issue) {
	typeSelector.setSelection("Activity");
	((ActivityControls)sendControls.mustBe("Activity"))
	    .transformToActivity(issue);
	initToMe();
    }

    public void initConstraint(PatternAssignment pv) {
	typeSelector.setSelection("Constraint");
	((ConstraintControls)sendControls.mustBe("Constraint"))
	    .initConstraint(pv);
    }

    public void initConstraintForMe(PatternAssignment pv) {
	initConstraint(pv);
    }

    public void initToMe() {
	sendControls.getSelected().initToMe();
    }

    public void initReport(TaskItem item,
			   Name reportTo,
			   Name ref,
			   ReportType type) {
	if (!contentText.getText().equals("")) {
	    Object[] message = {
		"You have asked to send a report when there",
		"is unsent text in the display.  Do you want",
		"to clear the text and set up to send the report?"
	    };
	    switch(JOptionPane.showConfirmDialog
		      (null, message, "Confirm", JOptionPane.YES_NO_OPTION)) {
	    case JOptionPane.YES_OPTION:
		break;
	    case JOptionPane.NO_OPTION:
		return;
	    }
	}
	typeSelector.setSelection("Report");
	((ReportControls)sendControls.mustBe("Report"))
	    .initReport(item, reportTo, ref, type);
    }

    protected void typeChangedTo(String newType) {
	Debug.noteln("SendPanel type changed to", newType);
    }

    /**
     * Construct an object from the current state of the GUI
     * and send it to the specified destination.
     *
     * @see #send(String, Sendable)
     */
    protected void sendTo(String destination) {
	Sendable contents = (Sendable)objectFromControls();
	send(destination, contents);
	contentText.showSent();
    }

    /**
     * Sends to the indicated destination.  If the destination is
     * "me", the contents are given directly to this agent's
     * handleInput method without going via IPC.
     *
     * <p>All of the SendPanel's sends go via this method.
     * It calls {@link #sending(String, Sendable)} just before
     * the sendable object is sent.
     *
     * <p>N.B. This method may modify the contents object.  Usually,
     * that is no problem for internal use, because the object will
     * have been newly constructed from the panel's GUI.  If modification
     * would be a problem, however, the sendCopy method should be
     * used instead.
     *
     * @see #sendCopy(String, Sendable)
     * @see ix.icore.IXAgent#handleInput(ix.util.IPC.InputMessage)
     */
    protected void send(String destination, Sendable contents) {

	if (contents.getSenderId() == null) {
	    // /\/: Pain that we have to set senderId.
	    contents
		.setSenderId(Name.valueOf(agent.getAgentSymbolName()));
	}

	if (contents instanceof ChatMessage) {
	    Color c = contactManager.getAgentColor(agent.getAgentSymbolName());
	    if (c != null)
		((ChatMessage)contents).setAnnotation(MESSAGE_COLOR, c);
	}

	Debug.noteln("Sending to", destination);
	Debug.noteln("SendPanel contents as XML:",
		     XML.objectToXMLString(contents));

	sending(destination, contents);

	if (destination.equalsIgnoreCase("me")) {
	    // Send to self without going through IPC.
	    agent.handleInput(new IPC.BasicInputMessage(contents));
	}
	else if (destination.equals("a group ...")) {
	    sendToGroup(contents);
	}
	else
	    IPC.sendObject(destination, contents);
    }

    /**
     * A hook to allow subclasses to see messages just before they're sent.
     * For example, a subclass may want to write a description of the
     * message to a transcript area.
     */
    protected void sending(String destination, Sendable contents) {
	// Do nothing.
    }

    /**
     * Used to send things "as if by the Messenger".  The contents
     * are copied because it may be modified by the send method
     * and because if sent to "me" the very same object will be
     * received and might be modified later on.  (Consider, say,
     * an issue from the "Test" menu.  If sent to "me", it might
     * later be modified if its status, priority, etc changed,
     * and we don't want to change the test menu's issue in case
     * we use the same test again.  Sending a copy avoids this
     * problem.
     *
     * @see #send(String, Sendable)
     */
    public void sendCopy(String destination, Sendable contents) {
	// N.B. must make a copy, because it will change
	// its status, etc if sent to self ("me").
	Sendable copy = (Sendable)Util.clone(contents);
	send(destination, copy);
    }

    protected void sendToGroup(Sendable contents) {
	new GroupSender(agent, contents);
    }

    /**
     * Construct an object from the current state of the GUI.
     * The object will be an Issue, Activity, Constraint, Report,
     * or ChatMessage.
     */
    protected Object objectFromControls() {
	String text = contentText.getText().trim();
	if (text.equals(""))
	    throw new IllegalArgumentException("Contents are empty");
	return sendControls.getSelected().objectFromControls();
    }

    /**
     * Obtain a priority from the current state of the GUI.
     */
    protected Priority priorityFromControls() {
	String selected = prioritySelector.getSelection();
	// This is a bit messy.  /\/
	Debug.expect(selected.endsWith(" Priority"));
	String p = Strings.beforeFirst(" ", selected);
	return Priority.valueOf(p.toLowerCase());
    }


    /*
     * Details of GUI construction
     */

    protected JPanel makeControlPanel() {
	JPanel panel = new JPanel();
	panel.setLayout(new BoxLayout(panel, BoxLayout.X_AXIS));

	panel.add(typeSelector);
	panel.add(prioritySelector);

	panel.add(Box.createHorizontalGlue());

	VerticalPanel sendAndStyle = new VerticalPanel();

	JPanel stylePanel = new JPanel();
	stylePanel.setLayout(new BorderLayout());
	stylePanel.add(styleButton, BorderLayout.EAST);

	sendAndStyle.add(sendControls);
        // /\/: Don't show the "Style" button.  Maybe eliminate it later.
	// sendAndStyle.add(stylePanel);

	// panel.add(sendControls);
	panel.add(sendAndStyle);

	return panel;
    }

    protected RadioButtonBox makeTypeControl() {
	RadioButtonBox box = RadioButtonBox.createVerticalBox();

	JRadioButton issueButton = new JRadioButton("Issue");
	JRadioButton activityButton = new JRadioButton("Activity");
	JRadioButton constraintButton = new JRadioButton("Constraint");
	JRadioButton annotationButton = new JRadioButton("Annotation");
	JRadioButton reportButton = new JRadioButton("Report");
	JRadioButton messageButton = new JRadioButton("Chat Message");

	messageButton.setSelected(true);

	annotationButton.setEnabled(false);

	box.add(issueButton);
	box.add(activityButton);
	box.add(constraintButton);
	box.add(annotationButton);
	box.add(reportButton);
	box.add(messageButton);

	box.add(Box.createVerticalGlue());

	box.addActionListener(new ActionListener() {
	    public void actionPerformed(ActionEvent event) {
		String selection = event.getActionCommand();
		sendControls.showCard(selection);
		typeChangedTo(selection);
	    }
	});

	return box;	
    }

    protected RadioButtonBox makePriorityControl() {
	RadioButtonBox box = RadioButtonBox.createVerticalBox();

	box.add(new JRadioButton("Highest Priority"));
	box.add(new JRadioButton("High Priority"));
	box.add(new JRadioButton("Normal Priority", true));
	box.add(new JRadioButton("Low Priority"));
	box.add(new JRadioButton("Lowest Priority"));

	box.add(Box.createVerticalGlue());

	return box;
    }

    class SendControlsPanel extends CardPanel {
	SendControlsPanel() {
	    super();
	    addCard("Issue", new IssueControls());
	    addCard("Activity", new ActivityControls());
	    addCard("Constraint", new ConstraintControls());
	    addCard("Annotation", new AnnotationControls());
	    addCard("Report", new ReportControls());
	    addCard("Chat Message", new ChatMessageControls());
	}
	SendControls getSelected() {
	    return (SendControls)super.getSelectedCard();
	}
	SendControls mustBe(String name) {
	    SendControls result = getSelected();
	    Debug.expectSame(result, getCard(name));
	    return result;
	}
	Component wrapCard(Component card) {
	    VerticalPanel p = new VerticalPanel();
	    p.addFixedHeight(card);
	    p.add(new JPanel()); // fills the rest of the vertical space
	    return p;
	}
    }

    class CardPanel extends JPanel {

	CardLayout cards = new CardLayout();
	Map nameToComp = new HashMap();
	Component selected = null;

	CardPanel() {
	    super();
	    setLayout(cards);
	}

	void addCard(String name, Component c) {
	    nameToComp.put(name, c);
	    add(wrapCard(c), name);
	}

	Component wrapCard(Component card) {
	    return card;
	}

	Component getCard(String name) {
	    Component card = (Component)nameToComp.get(name);
	    Debug.expect(card != null, "No card named", name);
	    return card;
	}

	void showCard(String name) {
	    selected = getCard(name); // not wrapped
	    cards.show(this, name);   // displays the wrapped version
	}

	Component getSelectedCard() {
	    Debug.expect(selected != null, "No selected card");
	    return selected;
	}

    }

    abstract class SendControls extends JPanel {
	GridColumn left = new GridColumn();
	GridColumn right = new GridColumn();

	DestinationChoice destinationChoice =
	    new DestinationChoice(agent, destinationChoiceModel);

	JButton sendButton =
	    new JButton("Send", Util.resourceImageIcon("ip2-send.gif"));

	SendControls() {
	    setLayout(new BoxLayout(this, BoxLayout.X_AXIS));
	    add(left);
	    add(right);

	    sendButton.setHorizontalTextPosition(SwingConstants.LEFT);
	    sendButton.addActionListener(actionListener);
	}

	String getDestination() {
	    return destinationChoice.getSelectedString();
	}

	abstract Object objectFromControls();

	void initToMe() {
	    destinationChoice.setSelectedItem("me");
	}

	void addRecipientAndSendButton() {
	    left.add(makeLeftLabel("Recipient ="));
	    right.add(destinationChoice);
	    left.add(new JLabel(""));
	    right.add(sendButton);
	}

	JLabel makeLeftLabel(String text) {
	    JLabel label = new JLabel(text);
	    label.setForeground(Color.black);
	    label.setHorizontalAlignment(SwingConstants.TRAILING);
	    return label;
	}

    }

    abstract class IssueActivityControls extends SendControls {

	JCheckBox reportCheck = new JCheckBox("Report-back");

	IssueActivityControls() {
//  	    reportCheck.setHorizontalAlignment(SwingConstants.TRAILING);
	    reportCheck.setModel(reportCheckModel);
	    left.add(new JLabel(""));
	    right.add(reportCheck);

	    addRecipientAndSendButton();
	}

	protected void initItem(TaskItem item) {
	    // Go back from Variables to ?-vars.
	    LList pat = (LList)Variable.removeVars(item.getPattern());
	    contentText.setText(PatternParser.unparse(pat));
	    prioritySelector.setSelection
		(Strings.capitalize(item.getPriority().toString())
		 + " Priority");
//  	    refChoice.setSelectedItem(item.getRef() == null ? "" 
//  				      : item.getRef().toString());
	    reportCheck.setSelected(item.getReportBack() == YesNo.YES);
	}

	/**
	 * Fills in the contents of an Issue or Activity.
	 */
	protected TaskItem fillTaskItemFromControls(TaskItem item) {
	    String text = contentText.getText().trim();
//  	    String ref = refChoice.getSelectedString();
	    boolean reportBack = reportCheck.isSelected();

	    item.setPattern(PatternParser.parse(text));
	    item.setSenderId(Name.valueOf((String)agent.getAgentSymbolName()));
	    item.setPriority(priorityFromControls());
//  	    if (!ref.equals(""))
//  		item.setRef(Name.valueOf(ref));
	    if (reportBack)
		item.setReportBack(YesNo.YES);
	    return item;
	}

    }

    class IssueControls extends IssueActivityControls {

	IssueControls() {
	}

	Object objectFromControls() {
	    return issueFromControls();
	}

	/**
	 * Construct an issue from the current state of the GUI.
	 */
	protected Issue issueFromControls() {
	    return (Issue)fillTaskItemFromControls(new Issue());
	}

    }

    class ActivityControls extends IssueActivityControls {

	Issue fromIssue = null;	// for transformToActivity

	ActivityControls() {
	}

	protected void initItem(TaskItem item) {
	    super.initItem(item);
	    fromIssue = null;
	}

	void transformToActivity(Issue issue) {
	    super.initItem(issue);
	    fromIssue = issue;
	}

	Object objectFromControls() {
	    return activityFromControls();
	}

	/**
	 * Construct an activity from the current state of the GUI.
	 */
	protected Activity activityFromControls() {
	    Activity act = (Activity)fillTaskItemFromControls(new Activity());
	    if (fromIssue != null) {
		Name senderId = fromIssue.getSenderId(); // may be null
		Debug.noteln("Transform sender-id", senderId);
		act.setSenderId(senderId);
		Name ref = fromIssue.getRef();
		Debug.noteln("Transform ref", ref);
		act.setRef(ref);
	    }
	    return act;
	}

    }

    class ConstraintControls extends SendControls {

	LTF_Parser constraintParser = new LTF_Parser();

	ConstraintControls() {
	    addRecipientAndSendButton();
	}

	public void initConstraint(PatternAssignment pv) {
	    contentText
		.setText(Lisp.elementsToString(pv.getPattern()) + " = " +
			 Lisp.printToString(pv.getValue()));
	}

	Object objectFromControls() {
	    return constraintFromControls();
	}

	/**
	 * Construct a constraint from the current state of the GUI.
	 */
	protected Constraint constraintFromControls() {
	    String text = contentText.getText().trim();
	    String[] parts = Strings.breakAtFirst("=", text);
	    String pat = parts[0].trim();
	    String val = parts[1].trim();
	    if (val.equals(""))
		val = "true";
	    String source = "(world-state effect (" + pat + ") = " + val + ")";
	    LList spec = (LList)Lisp.readFromString(source);
	    Constraint c = constraintParser.parseConstraint(spec);
	    if (c == null)
		throw new SyntaxException("Invalid constraint: "
					  + Strings.quote(text));
	    c.setSenderId(Name.valueOf((String)agent.getAgentIPCName()));
	    return c;
	}

    }

    class AnnotationControls extends SendControls {

	AnnotationControls() {
	    addRecipientAndSendButton();
	}

	Object objectFromControls() {
	    // return annotationFromControls();
	    throw new UnsupportedOperationException();
	}

    }

    class ReportControls extends SendControls {

	JComboBox reportTypeChoice =
	    new JComboBox(ReportType.values().toArray());

	JTextField refText = new JTextField("");

	ReportControls() {
	    reportTypeChoice.setSelectedItem(ReportType.INFORMATION);
	    refText.setEditable(false);

	    left.add(makeLeftLabel("Report Type ="));
	    right.add(reportTypeChoice);

	    left.add(makeLeftLabel("Ref ="));
	    right.add(refText);

	    addRecipientAndSendButton();
	}

	void initReport(TaskItem item,
			Name reportTo,
			Name ref,
			ReportType type) {
	    contentText.setText("");
	    reportTypeChoice.setSelectedItem(type);
	    if (reportTo != null)
		destinationChoice.setSelectedItem(reportTo.toString());
//  	    refChoice.setSelectedItem(ref == null ? "" : ref.toString());
	    refText.setText(ref == null ? "" : ref.toString());
	}

	Object objectFromControls() {
	    return reportFromControls();
	}

	/**
	 * Construct a report from the current state of the GUI.
	 */
	protected Report reportFromControls() {
	    String text = contentText.getText().trim();
	    String ref = refText.getText();
	    ReportType type = (ReportType)reportTypeChoice.getSelectedItem();

	    Report report = new Report(text);
	    report.setSenderId(Name.valueOf((String)agent.getAgentIPCName()));
	    report.setPriority(priorityFromControls());
	    if (!ref.equals(""))
		report.setRef(Name.valueOf(ref));
	    if (!type.equals(""))
		report.setReportType(type);

	    return report;
	}

    }

    class ChatMessageControls extends SendControls {

	ChatMessageControls() {
	    addRecipientAndSendButton();
	}

	Object objectFromControls() {
	    return chatMessageFromControls();
	}

	/**
	 * Construct a chat message from the current state of the GUI.
	 */
	protected ChatMessage chatMessageFromControls() {
	    String text = contentText.getText();
	    String sender = (String)agent.getAgentIPCName();
	    return new ChatMessage(text, sender);
	}

    }

}

// Issues:
// * We use getAgentSymbolName instead of getAgentIPCName just
//   because it's declared to return a String.  /\/
