/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Thu May  8 13:50:27 2003 by Jeff Dalton
 * Copyright: (c) 2001, 2003, AIAI, University of Edinburgh
 */

package ix.ispace.event;

import java.util.EventObject;
import ix.ispace.*;

/**
 * A change in what we know about another agent.
 *
 * <p>Exactly one of the following should return true:
 * <ul>
 *  <li>{@link #isNewContact()}
 *  <li>{@link #isDeletion()}
 *  <li>{@link #isRelationshipChange()}
 *  <li>{@link #isCapabilityChange()}
 * </ul>
 * </p>
 */
public class ContactEvent extends EventObject {

    protected AgentData oldData;
    protected AgentData newData;

    public ContactEvent(ContactManager source, 
			AgentData oldData,
			AgentData newData) {
	super(source);
	this.oldData = oldData;
	this.newData = newData;
    }

    public AgentData getOldData() {
	return oldData;
    }

    public AgentData getNewData() {
	return newData;
    }

    public boolean isNewContact() {
	return oldData == null;
    }

    public boolean isDeletion() {
	return newData == null;
    }

    public boolean isRelationshipChange() {
	return oldData != null && newData != null
	    && oldData.getRelationship() != newData.getRelationship();
    }

    public boolean isCapabilityChange() {
	return oldData != null && newData != null
	    && !oldData.getCapabilities().equals(newData.getCapabilities());
    }

    public boolean isNewCapability(Capability c) {
	// For a new contact, we count all known capabilities as new.
	return newData != null && newData.hasCapability(c)
	    && (oldData == null || !oldData.hasCapability(c));
    }

    public String toString() {
	return "ContactEvent[" + oldData + " --> " + newData + "]";
    }

}
