package ix.isim.util;

import java.util.*;

/**
 * <p>This class represents a priority queue, that is a queue in which elements
 * can be held and from which they can be retrieved by priority. For this
 * purpose, each object has a priority associated with it that determines when
 * in relation to the other elements in the queue this element should be
 * retrieved. Priorities are <code>long</code>s here. </p>
 *
 * <p>If there are several elements in the queue that have the same priority
 * their order can be managed by adding them to the front or the end of the
 * list of elements at that priority. Similarly, retrival can be from the front
 * or the end of the list of elements with the same priority. </p>
 *
 * <p>The implementation of this priority queue is achieved by maintaining the
 * objects in a sorted tree. The order is determined by the priority and each
 * tree node contains a linked list of all the nodes with the same priority. In
 * this way a reasonably efficient execution of all the operations provided can
 * be guaranteed even for very long queues. Note, however, that the removal of
 * elements from this queue and the contains-test with a given priority is not
 * supported. </p>
 *
 * @author Gerhard Wickler
 * @version 8.Apr.2005
 **/

public class TreeOfListsLongPriorityQueue
        implements LongPriorityQueue {

    ////////////////////////
    // instance variables //
    ////////////////////////

    // protected //

    /** the data structure that contains a tree of lists of nodes **/
    protected LongKeyTreeMap pTree = new LongKeyTreeMap();

    // private //

    /** the number of elements currently in this PriorityQueue **/
    private int size = 0;

    /////////////////////////
    // public constructors //
    /////////////////////////

    /**
     * <p>This constructor creates an empty PriorityQueue. </p>
     **/
    public TreeOfListsLongPriorityQueue() {
    }

    /**
     * <p>This constructor creates a priority queue with exactly one element,
     * the given object queued at the given priority. </p>
     *
     * @param object the <code>Object</code> to be the initial member of this
     * <code>PriorityQueue</code>
     * @param priority the priority with which the element is to be queued
     **/
    public TreeOfListsLongPriorityQueue(Object object, long priority) {
        this.addElementAtEnd(object, priority);
    }

    /////////////
    // cloning //
    /////////////

    /**
     * <p>This class does not support cloning and an Exception will be thrown if
     * this method is called. </p>
     *
     * @return nothing
     * @exception CloneNotSupportedException will be thrown
     **/
    protected Object clone() throws CloneNotSupportedException {
        throw new CloneNotSupportedException();
    }

    ///////////////////////////////
    // public instance functions //
    ///////////////////////////////

    // inherited //

    /**
     * <p>This function adds the given object at the given priority to this
     * priority queue. If there are already elements queued at this priority,
     * the new element will be queued at the front. </p>
     *
     * @param object the <code>Object</code> to be added to this
     * <code>LongPriorityQueue</code>
     * @param priority the priority with which the element is to be queued
     **/
    synchronized public void addElementAtFront(Object object, long priority) {
        addTreeList(priority).addFirst(object);
        size++;
    }

    /**
     * <p>This function adds the given object at the given priority to this
     * priority queue. If there are already elements queued at this priority,
     * the new element will be queued at the end. </p>
     *
     * @param object the <code>Object</code> to be added to this
     * <code>LongPriorityQueue</code>
     * @param priority the priority with which the element is to be queued
     **/
    synchronized public void addElementAtEnd(Object object, long priority) {
        addTreeList(priority).addLast(object);
        size++;
    }

    /**
     * <p>This function tests whether this priority queue is empty. </p>
     *
     * @return <code>true</code> if and only if this <code>PriorityQueue</code>
     * contains no elements
     **/
    public boolean isEmpty() {
        return size == 0;
    }

    /**
     * <p>This function returns the size of this priority queue. </p>
     *
     * @return the number of elements contained in this
     * <code>PriorityQueue</code>
     **/
    public int length() {
        return size;
    }

    /**
     * <p>This function throws an <code>UnsupportedOperationException</code> as
     * there is no efficient implementation of this operation for this class.
     * </p>
     *
     * @param object Object the element to be tested for
     * @param priority int the priority at which it should be queued
     * @return boolean whether it is indeed in the queue
     * @exception UnsupportedOperationException will be thrown
     */
    public boolean containsElementAt(Object object, long priority) {
        throw new java.lang.UnsupportedOperationException();
    }

    /**
     * <p>This function returns the element in this priority queue that has the
     * lowest priority. If there are several elements queued at that priority,
     * the element at the front of the list is retrieved. An exception is thrown
     * if there is no such element, i.e. if this priority queue is empty. </p>
     *
     * @return the <code>Object</code> that has the lowest priority in this
     * <code>LongPriorityQueue</code>
     * @exception NoSuchElementException if this <code>LongPriorityQueue</code> is
     * empty
     **/
    public Object getLowestFront() throws NoSuchElementException {
        if (size == 0) {
            throw new NoSuchElementException("Priority queue is empty!");
        }
        return ( (LinkedList) pTree.get(pTree.firstKey())).getFirst();
    }

    /**
     * <p>This function returns the element in this priority queue that has the
     * lowest priority. If there are several elements queued at that priority,
     * the element at the end of the list is retrieved. An exception is thrown
     * if there is no such element, i.e. if this priority queue is empty. </p>
     *
     * @return the <code>Object</code> that has the lowest priority in this
     * <code>LongPriorityQueue</code>
     * @exception NoSuchElementException if this <code>LongPriorityQueue</code> is
     * empty
     **/
    public Object getLowestEnd() throws NoSuchElementException {
        if (size == 0) {
            throw new NoSuchElementException("Priority queue is empty!");
        }
        return ( (LinkedList) pTree.get(pTree.firstKey())).getLast();
    }

    /**
     * <p>This function returns the element in this priority queue that has the
     * highest priority. If there are several elements queued at that priority,
     * the element at the front of the list is retrieved. An exception is thrown
     * if there is no such element, i.e. if this priority queue is empty. </p>
     *
     * @return the <code>Object</code> that has the highest priority in this
     * <code>LongPriorityQueue</code>
     * @exception NoSuchElementException if this <code>LongPriorityQueue</code> is
     * empty
     **/
    public Object getHighestFront() throws NoSuchElementException {
        if (size == 0) {
            throw new NoSuchElementException("Priority queue is empty!");
        }
        return ( (LinkedList) pTree.get(pTree.lastKey())).getFirst();
    }

    /**
     * <p>This function returns the element in this priority queue that has the
     * highest priority. If there are several elements queued at that priority,
     * the element at the end of the list is retrieved. An exception is thrown
     * if there is no such element, i.e. if this priority queue is empty. </p>
     *
     * @return the <code>Object</code> that has the highest priority in this
     * <code>LongPriorityQueue</code>
     * @exception NoSuchElementException if this <code>LongPriorityQueue</code> is
     * empty
     **/
    public Object getHighestEnd() throws NoSuchElementException {
        if (size == 0) {
            throw new NoSuchElementException("Priority queue is empty!");
        }
        return ( (LinkedList) pTree.get(pTree.lastKey())).getLast();
    }

    /**
     * <p>This function removes and returns the element in this priority queue
     * that has the lowest priority. If there are several elements queued at
     * that priority, the element at the front of the list is retrieved. An
     * exception is thrown if there is no such element, i.e. if this priority
     * queue is empty. </p>
     *
     * @return the <code>Object</code> that has the lowest priority in this
     * <code>LongPriorityQueue</code>
     * @exception NoSuchElementException if this <code>LongPriorityQueue</code> is
     * empty
     **/
    synchronized public Object removeLowestFront() throws NoSuchElementException {
        if (size == 0) {
            throw new NoSuchElementException("Priority queue is empty!");
        }
        size--;
        return removeLowestTreeList().removeFirst();
    }

    /**
     * <p>This function removes and returns the element in this priority queue
     * that has the lowest priority. If there are several elements queued at
     * that priority, the element at the end of the list is retrieved. An
     * exception is thrown if there is no such element, i.e. if this priority
     * queue is empty. </p>
     *
     * @return the <code>Object</code> that has the lowest priority in this
     * <code>LongPriorityQueue</code>
     * @exception NoSuchElementException if this <code>LongPriorityQueue</code> is
     * empty
     **/
    synchronized public Object removeLowestEnd() throws NoSuchElementException {
        if (size == 0) {
            throw new NoSuchElementException("Priority queue is empty!");
        }
        size--;
        return removeLowestTreeList().removeLast();
    }

    /**
     * <p>This function removes and returns the element in this priority queue
     * that has the highest priority. If there are several elements queued at
     * that priority, the element at the front of the list is retrieved. An
     * exception is thrown if there is no such element, i.e. if this priority
     * queue is empty. </p>
     *
     * @return the <code>Object</code> that has the highest priority in this
     * <code>LongPriorityQueue</code>
     * @exception NoSuchElementException if this <code>LongPriorityQueue</code> is
     * empty
     **/
    synchronized public Object removeHighestFront() throws NoSuchElementException {
        if (size == 0) {
            throw new NoSuchElementException("Priority queue is empty!");
        }
        size--;
        return removeHighestTreeList().removeFirst();
    }

    /**
     * <p>This function removes and returns the element in this priority queue
     * that has the highest priority. If there are several elements queued at
     * that priority, the element at the end of the list is retrieved. An
     * exception is thrown if there is no such element, i.e. if this priority
     * queue is empty. </p>
     *
     * @return the <code>Object</code> that has the highest priority in this
     * <code>LongPriorityQueue</code>
     * @exception NoSuchElementException if this <code>LongPriorityQueue</code> is
     * empty
     **/
    synchronized public Object removeHighestEnd() throws NoSuchElementException {
        if (size == 0) {
            throw new NoSuchElementException("Priority queue is empty!");
        }
        size--;
        return removeHighestTreeList().removeLast();
    }

    /**
     * <p>This function throws an <code>UnsupportedOperationException</code> as
     * there is no efficient implementation of this operation for this class.
     * </p>
     *
     * @param obj the <code>Object</code> to be removed from this
     * <code>PriorityQueue</code>
     * @param p the priority at which the element is queued
     * @return nothing
     * @exception UnsupportedOperationException will be thrown
     **/
    public boolean removeElementAt(Object obj, long p) {
        throw new java.lang.UnsupportedOperationException();
    }

    /**
     * <p>This function returns an Iterator for this priority queue. </p>
     *
     * @return an <code>Iterator</code> for this <code>PriorityQueue</code>
     **/
    public Iterator elements() {
        return new TreeOfListsLongPriorityQueueIterator();
    }

    //////////////////////
    // Object functions //
    //////////////////////

    /**
     * <p>This function creates the printable string representation of this
     * priority queue. </p>
     *
     * @return the <code>String</code> representing this
     * <code>PriorityQueue</code>
     **/
    public String toString() {
        String result = "[";
        Iterator pqe = elements();
        if (pqe.hasNext()) {
            result += pqe.next().toString();
        }
        while (pqe.hasNext()) {
            result += " " + pqe.next().toString();
        }
        return result + "]";
    }

    /**
     * <p>This class does not support equality testing and an exception will be
     * thrown if this method is called. </p>
     *
     * @param obj the <code>Object</code>object this should be compared to
     * @return nothing
     * @exception UnsupportedOperationException will be thrown
     **/
    public boolean equals(Object obj) throws UnsupportedOperationException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>This class does not support hashing and an exception will be thrown if
     * this method is called. </p>
     *
     * @return nothing
     * @exception UnsupportedOperationException will be thrown
     **/
    public int hashCode() throws UnsupportedOperationException {
        throw new UnsupportedOperationException();
    }

    ////////////////////////////////
    // private instance functions //
    ////////////////////////////////

    private LinkedList addTreeList(long priority) {
        LinkedList pList = (LinkedList) pTree.get(priority);
        if (pList == null) {
            pList = new LinkedList();
            pTree.put(priority, pList);
        }
        return pList;
    }

    private LinkedList removeLowestTreeList() {
        long minPri = pTree.firstKey();
        LinkedList pList = (LinkedList) pTree.get(minPri);
        if (pList.size() == 1) {
            pTree.remove(minPri);
        }
        return pList;
    }

    private LinkedList removeHighestTreeList() {
        long maxPri = pTree.lastKey();
        LinkedList pList = (LinkedList) pTree.get(maxPri);
        if (pList.size() == 1) {
            pTree.remove(maxPri);
        }
        return pList;
    }

    ///////////////////
    // inner classes //
    ///////////////////

    /**
     * <p>This class represents an enumerator for the elements in a priority
     * queue. Elements will be enumerated in order! </p>
     **/
    private class TreeOfListsLongPriorityQueueIterator
            implements Iterator {
        /** the iterator for the elements in the current list **/
        Iterator listElts;
        /** the iterator for the elements in the tree **/
        Iterator treeElts;

        /**
         * <p>This constructor initializes an Iterator of the elements in the
         * given priority queue. </p>
         **/
        TreeOfListsLongPriorityQueueIterator() {
            this.treeElts = pTree.values();
            if (treeElts.hasNext()) {
                this.listElts = ( (LinkedList) treeElts.next()).iterator();
            }
            else {
                this.listElts = null;
            }
        }

        /**
         * <p>This function tests whether there are more elements to be
         * enumerated. </p>
         *
         * @return <code>true</code> if and only if there are more elements
         **/
        public boolean hasNext() {
            if (listElts == null) {
                return false;
            }
            return listElts.hasNext();
        }

        /**
         * <p>This function retrieves the next element in this Iterator. </p>
         *
         * @return the next element
         * @exception NoSuchElementException is thrown iff there are no more
         * elements
         **/
        public Object next() throws NoSuchElementException {
            if (listElts == null) {
                throw new NoSuchElementException();
            }
            Object result = listElts.next();
            if (listElts.hasNext()) {
                return result;
            }
            if (treeElts.hasNext()) {
                listElts = ( (LinkedList) treeElts.next()).iterator();
            }
            return result;
        }

        public void remove() {
            throw new UnsupportedOperationException();
        }
    }

}
