/* Author: Gerhard Wickler <g.wickler@ed.ac.uk>
 * Copyright: (c) 2006, AIAI, University of Edinburgh
 */

package ix.isim;

import ix.icore.IXAgent;
import ix.icore.Activity;

import ix.util.lisp.Symbol;
import ix.util.Debug;
import ix.iface.util.ToolController;

/**
 * <p>An ISimTimer is a device for keeping track of simualted time. There are
 * two sub-classes here that can be used: </p>
 *
 * <ul>
 * <li>an ISimTimerServer belongs to an ISim agent and controls simulated time
 * for other ISimTimers (that have regitered with the ISimTimerServer); there
 * should only be one such timer for any I-X application</li>
 * <li>an ISimTimerClient registers with an ISimTimerServer and obtains the
 * current simulated time from the server for displaying it, not for changing
 * it</li>
 * </ul>
 *
 * <p>The sole constructor of this class is protected and the static method
 * getISimTimer(...) should be used to obtain the right kind of ISimTimer for
 * the given agent type. </p>
 */
abstract public class ISimTimer {

    /** some command strings used for the synchronization of multiple
     * ISimTimers */
    protected final Symbol SYNCHRONIZE_LABEL =
            Symbol.intern("synchronize-simulation-time");
    protected final String START = "start";
    protected final String PAUSE = "pause";
    protected final String RESUME = "resume";
    protected final String ACCELERATE = "accelerate";
    protected final String STOP = "stop";
    protected final String REGISTER = "register";

    /** the IXAgent to which this timer belongs **/
    protected IXAgent owner;
    /** indicates whether simulation time is currently passing **/
    protected boolean simulationActive = false;
    /** the frame displaying the simuation time **/
    protected ISimTimeFrame simTimeDisplay;
    /** the Thread used for updating the display **/
    protected UpdateTimeDisplayThread tUpdater;

    /** gives the factor at which simulation time is progressing (compared to
     ** real time) **/
    private double accelerationFactor = 1.0d;
    /** system time of last user change of simulation time **/
    private long lastSimTimeChangeInReality = 0L;
    /** ISim time of of last user change of simulation time **/
    private long lastSimTimeChangeInSimTime = 0L;

    /**
     * Protected constructor for use by sub-classes. It registers this timer as
     * a tool with the given IXAgent so that the user can display it.
     * @param agent IXAgent owner of this timer
     */
    protected ISimTimer(IXAgent agent) {
        this.owner = agent;
        this.simTimeDisplay = new ISimTimeFrame();
        this.tUpdater = new UpdateTimeDisplayThread();
        final ToolController tc = new ToolController("I-Sim Clock") {
            public Object createTool() {
                return simTimeDisplay;
            }
        };
        owner.addTool(tc);
    }

    /**
     * This function needs to be implemented by the inheriting class. This is
     * because the two types of ISimTimer, ISimTimerClient and ISimTimerServer,
     * behave differently with respect to synchronization. Having it abstract
     * ensures that no ISimTimer can ever be constructed.
     * @param synchActivity a synchronization Activity (see final Strings in
     * this class for possible activities)
     */
    abstract public void handleSynchronizeActivity(Activity synchActivity);

    /**
     * Returns whether the start function has been called.
     * @return boolean whether simulation time has been started
     */
    public boolean simulationStarted() {
        return lastSimTimeChangeInReality > 0L;
    }

    /**
     * This function returns the current time in the simulation. This function
     * should only be called after the simulation has been started. Then it will
     * return the correct simulated time even if the simulation is paused.
     * @return long
     */
    public long getSimTimeInMillis() {
        if (simulationActive) {
            long realElapsedTime = System.currentTimeMillis() -
                                   lastSimTimeChangeInReality;
            long simElapsedTime = (long) (realElapsedTime * accelerationFactor);
            return lastSimTimeChangeInSimTime + simElapsedTime;
        } else {
            return lastSimTimeChangeInSimTime;
        }
    }

    /**
     * Returns the currently used time acceleration factor
     * @return double current acceleration factor
     */
    public double getAccelerationFactor() {
        return accelerationFactor;
    }

    /**
     * This function starts the simulation clock. The simulation time will be
     * set to the given time and simulated time will progress with the given
     * acceleration factor. It also shows the frame displaying current sim-time
     * if it isn't already showing.
     * @param simTime initial value of the simulated time
     * @param factor acceleration factor for simulated time
     * @throws ISimTimerException if this function has been called before
     */
    public void start(long simTime, double factor) throws ISimTimerException {
        if (simulationStarted()) {
            throw new ISimTimerException(
                    "ISimTimer already started. Cannot start again.");
        }
        // initialize the timer:
        this.simulationActive = true;
        this.accelerationFactor = factor;
        this.lastSimTimeChangeInReality = System.currentTimeMillis();
        this.lastSimTimeChangeInSimTime = simTime;
        // start the Thread that updates the display:
        tUpdater.start();
        // show the frame displaying the time:
        simTimeDisplay.setRunning();
        if (this.owner instanceof ix.isim.ISim) {
            simTimeDisplay.setVisible(true);
        }
    }

    /**
     * This function pauses this simulation timer, be it a Server or Client. As
     * a result simulationActive will be false and the display will show it is
     * paused. However, the time update thread will continue to run.
     * @return long the simulation time when the simulation was paused
     */
    public long pause() {
        // must get this first for correct computation of sim-time:
        this.lastSimTimeChangeInSimTime = getSimTimeInMillis();
        this.lastSimTimeChangeInReality = System.currentTimeMillis();
        this.simulationActive = false;
        simTimeDisplay.setPaused();
        tUpdater.updateNow();
        return lastSimTimeChangeInSimTime;
    }

    /**
     * This function re-starts the simulation timer after it had been paused. As
     * a result simulationActive will be true and the display will show that it
     * is running.
     * @return long the simulation time when the simulation was resumed
     */
    public long resume() {
        this.simulationActive = true;
        this.lastSimTimeChangeInReality = System.currentTimeMillis();
        simTimeDisplay.setRunning();
        tUpdater.updateNow();
        return lastSimTimeChangeInSimTime;
    }

    /**
     * This function changes the time acceleration factor to the given value.
     * @param acceleration double the new acceleration factor
     * @return long the simulation time when the acceleration factor was changed
     */
    public long changeAccelerationFactor(double acceleration) {
        this.accelerationFactor = acceleration;
        this.lastSimTimeChangeInReality = System.currentTimeMillis();
        this.lastSimTimeChangeInSimTime = getSimTimeInMillis();
        tUpdater.updateNow();
        return lastSimTimeChangeInSimTime;
    }

    /**
     * This function should be called terminate this Timer. It will gracefully
     * terminate the update thread for the display.
     */
    public void terminateTimer() {
        tUpdater.terminate();
    }

    protected void setCurrentSimTime(long time) {
        this.lastSimTimeChangeInReality = System.currentTimeMillis();
        this.lastSimTimeChangeInSimTime = time;
    }

    /**
     * Factory for getting the right kind of timer for this agent. ISim agents
     * get an ISimTimerServer and others get an ISimTimerClient.
     * @param agent IXAgent owner of this timer
     * @return ISimTimer the new timer
     */
    static public ISimTimer getISimTimer(IXAgent agent) {
        Debug.noteln("Creating ISimTimer ...");
        if (agent instanceof ix.isim.ISim) {
            return new ISimTimerServer(agent);
        } else {
            return new ISimTimerClient(agent);
        }
    }

    /**
     * This Thread is used to update the displayed time on a regular basis.
     * Normally the display is updated once a minute, but this depends on the
     * current acceleration factor. Updates occur only when the simulation is
     * active.
     */
    class UpdateTimeDisplayThread extends Thread {

        private boolean terminate = false;

        synchronized public void updateNow() {
            this.interrupt();
        }

        synchronized public void terminate() {
            terminate = true;
            simTimeDisplay.setPaused();
            this.interrupt();
        }

        public void run() {
            while (true) {
                synchronized (this) {
                    simTimeDisplay.setTime(getSimTimeInMillis());
                }
                try {
                    this.sleep((long) (6000 / accelerationFactor));
                } catch (InterruptedException ie) {
                    // simply continue when interrupted
                }
                if (terminate) {
                    return;
                }
            }
        }
    }

}
