/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Modifications by: Gerhard Wickler <G.Wickler@ed.ac.uk>
 * Copyright: (c) 2005, 2006, AIAI, University of Edinburgh
 */

package ix.isim;

import java.util.LinkedList;

import ix.ip2.Ip2;
import ix.util.Util;

/**
 * <p>This class is the entry point for starting an ISim agent. It essentially
 * an Ip2 with the following modifications: </p>
 *
 * <ul>
 * <li>it always has an ISimTimer associated with it that can be used to control
 * other ISimTimers</li>
 * <li>it has an ISimEngine associated with it that performs all the simulating
 * of events; the interface to this simulator is provided as a "I-Sim Control"
 * tool</li>
 * </ul>
 */
public class ISim extends Ip2 {

    /** the simulator used by this agent to modify the world state **/
    protected ISimEngine simulator;

    /**
     * The sole constructor for a stand-alone ISim I-X agent. The constructor is
     * private and used by main(...) to create the agent that must be unique in
     * this JVM.
     */
    private ISim() {
        super(true); // true = stand-alone
        agentTypeName = "I-X World Simulator";
        logoLine1 = "I-X World Simulator";
        displayName = "I-X World Simulator";
        showOptions = false;
    }

    /**
     * This function creates the simulator that belongs to this ISim agent. It
     * is called automatically during the initialization of an Ip2. Use the
     * getISimEngine() method to get the simulation engine.
     */
    public void completeStartup() {
        super.completeStartup();
        simulator = new ISimEngine(this);
    }

    /**
     * returns the simulation engine used by this ISim agent
     * @return ISimEngine the simulator
     */
    public ISimEngine getISimEngine() {
        return simulator;
    }

    /**
     * This function creates an ISimTimer for this ISim agent. Whereas other
     * IXAgents have an ISimTimer only if the "isim-agent-name" parameter has
     * been given to them, an ISim agent always has an ISimTimer. This function
     * is called automatically during the initialization of an IXAgent. Use this
     * IXAgent's getISimTimer() method to retrieve the ISimTimer.
     */
    protected void setupISimTimer() {
        iSimTimer = ISimTimer.getISimTimer(this);
    }

    /**
     * This function creates the ISim I-X agent. It starts up everything around
     * it incl. communication services and user interface by creating an ISim
     * and calling its mainStartup(...) function.
     * @param argv String[] parameters for this ISim agent
     */
    public static void main(String[] argv) {
        Util.printGreeting("I-Sim");
        new ISim().mainStartup(argv);
    }


    /* A "pretend" class for {@link DynamicObjectsFiles}. */
    public static class AnimatedObjectList extends LinkedList {
        public AnimatedObjectList() {}
    }

}
