/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Wed Jun 11 17:06:41 2008 by Jeff Dalton
 * Copyright: (c) 2004 - 2008, AIAI, University of Edinburgh
 */

package ix.iplan;

import java.util.*;

import ix.ip2.*;
import ix.icore.*;
import ix.icore.process.*;
import ix.icore.domain.*;
import ix.util.*;
import ix.util.lisp.*;
import ix.util.match.*;

// /\/: Still requires effects to be only at END ends.
// /\/: Conditions only at BEGIN ends.
// /\/: Assumes all conds and effects come from refinements that put them
//      at BEGIN or END of self during planning.  So if the planner doesn't
//      expand a node, it assumes the node has no conditions.

/**
 * Finds an executable node-end and either simulates its execution
 * or begins its expansion.
 */
class SlipFindExecutable implements Slip.Step {

    Slip slip;

    SlipFindExecutable(Slip slip) {
	this.slip = slip;
    }

    public void run() {
	LList nodeEnds = getNodeEnds();
	findNext(nodeEnds, endIterator(nodeEnds));
    }

    void findNext(LList allNodeEnds, LListIterator nodeEndIter) {
	while (true) {
	    // First, find a node-end with status POSSIBLE, skipping
	    // node-ends that are waiting for another node-end to execute.
	    // If none exists, an exception is thrown to indicate
	    // either that the plan is complete or that we must
	    // try an alternative.
	    PNodeEnd ne = nextExecutable(nodeEndIter);
	    Debug.noteln("SlipFindExecutable considering", ne);
	    // END ends don't expand, but they may execute.
	    if (ne.getEnd() == End.END) {
		// Could the node-end execute?
		if (slip.canExecEnd(ne)) {
		    Debug.noteln("Can execute", ne);
		    // Yes -- we'll execute this node-end after posting
		    // an alt that will try others instead.
		    postFindNextAlt(ne, allNodeEnds, nodeEndIter, "exec end");
		    // Process its effects and make it COMPLETE.
		    slip.execEnd(ne);
		    // More node-ends might now have status POSSIBLE.
		    nodeEndIter = endIterator(allNodeEnds);
		}
		continue;
	    }
	    // (From here on, we know we have a BEGIN end.)
	    PNode node = ne.getNode();
	    // It might be a goal node
	    if (slip.isGoalNode(ne)) {
		Debug.noteln("Examining goal node", ne);
		PatternAssignment goalCond = slip.getGoalCond(ne);
		ListOfConstraint goalList = new LinkedListOfConstraint();
		goalList.add(makeCondition(goalCond));
		Debug.noteln("Goal cond", goalCond);
		List envs = slip.MM().evalFilters(goalList, new MatchEnv());
		// If the cond can be satisfied, we delete the goal node.
		if (!envs.isEmpty()) {
		    Debug.noteln("Envs for goal cond", envs);
		    // We might have tried a different node-end first.
		    postFindNextAlt(ne, allNodeEnds, nodeEndIter, "delete");
		    // /\/: Expanding the goal node can still be considered
		    // a valid possibility.
		    if (slip.liberalAchieveSemantics)
			maybePostExpandGoalAlt(ne);
		    // Tell the variable manager about the ways the
		    // cond might be satisfied.
		    // /\/: We ought to protect over the range from
		    // the possible contributors to the node-end that
		    // wants the goal cond achieved.  That would need
		    // a multiple-contributor GOST entry.
		    List branches = slip.MM().reevaluateFilters(goalList);
		    Debug.expect(!branches.isEmpty(),
				 "Cannot match goal", goalCond);
		    // Also, the node that had the condition might
		    // now be executable, since it no longer needs
		    // to wait for the goal node.  /\/: Since we don't
		    // have a handy ref to that node-end, we'll make
		    // the end of the goal complete before deleting,
		    // which should do the trick.
		    slip.setStatus(node.getEnd(), Status.COMPLETE);
		    slip.releaseWaitingEnds(node.getBegin());
		    slip.releaseWaitingEnds(node.getEnd());
		    // Delete the goal node.
		    ActivityItem item = (ActivityItem)ne.getNode();
		    Agenda actAgenda = slip.getController()
			.getActivityAgenda();
		    Debug.noteln("Deleting goal node", item);
		    actAgenda.removeItem(item);
		    // Since we've changed the list of nodes, we
		    // need to get it again.
		    allNodeEnds = getNodeEnds();		    
		    // Find the next node-end to execute.
		    // We (probably?) don't make anything executable except
		    // the node-end that wanted the condition satisfied,
                    // but we may be in an alt in which we skipped an
		    // executable earlier in the list of node-ends.
		    nodeEndIter = endIterator(allNodeEnds);
		    continue;
		}
		// Otherwise, treat as if it were a normal node.
		Debug.expect(!node.isExpanded());
	    }
	    // The node may already be expanded.
	    if (isAlreadyExpanded(ne)) {
		Debug.noteln("Is already expanded");
		ListOfConstraint conds = slip.MM().getNodeConditions(node);
		if (conds == null || conds.isEmpty()) {
		    // No conditions, so treat as primitive
		    Debug.noteln("No conditions");
		    postFindNextAlt(ne, allNodeEnds, nodeEndIter,
				    "exec expanded primitive");
		    slip.execBegin(ne);
		    nodeEndIter = endIterator(allNodeEnds);
		    continue;
		}
		List envs = slip.MM().evalFilters(conds, new MatchEnv());
		Debug.noteln("Conditions", conds);
		Debug.noteln("Envs", envs);
		// The envs tell us all the ways we could satisfy
		// all the filters by binding.  Each env is a way to
		// satisy all the filters.
		if (envs.isEmpty()) {
		    // Cannot satisfy the conds, so not really executable
		    // at this point.
		    Debug.noteln("Cannot satisfy conds");
		    // If it looks like we might be able to satisfy the
		    // conds by introducing some nodes, post an alt that
		    // does that.  /\/: Perhaps we should do this even
		    // if all conds can be satisfied now.
		    if (!slip.isWaitingForAchieve(node)
			  && mightAchieveConds(conds)) {
			postSatisfyByAchieveAlt(node, conds);
		    }
		    // Try a different node-end now, because we can't
		    // execute this one.
		    continue;
		}
		// We can satisfy the conditions.
		// We're going to do something with this node-end,
		// so post an alt to try another instead.
		postFindNextAlt(ne, allNodeEnds, nodeEndIter,
				"satisfy conds for expanded");
		// Satisfy the conds, posting an alt for any other
		// ways to satisfy them.
		satisfyNodeEndConds(ne, conds, envs);
		// See what node-ends are now POSSIBLE.
		nodeEndIter = endIterator(allNodeEnds);
		continue;
	    }
	    // Try refinements
	    Debug.noteln("Trying refinements");

	    LinkedList useNow = new LinkedList();
	    LinkedList useLater = new LinkedList();
	    if (!nodeExpanders(ne, useNow, useLater)) {
		// No refinements apply, so treat as primitive.
		Debug.noteln("No refinements apply");
		postFindNextAlt(ne, allNodeEnds, nodeEndIter,
				"exec primitive");
		slip.execBegin(ne);
		nodeEndIter = endIterator(allNodeEnds);
		continue;
	    }
	    Debug.noteln("useNow", useNow);
	    Debug.noteln("useLater", useLater);
	    if (!useNow.isEmpty()) {
		// The node can be expanded now.
		Debug.noteln("Can expand");
		// Though we might have picked another node instead.
		postFindNextAlt(ne, allNodeEnds, nodeEndIter, "expand");
		if (!useLater.isEmpty()) {
		    // We can also expand it later
		    postExpandLaterAlt(node, useLater);
		}
		// Do the expansion.
		slip.setNextStep(new SlipExpandNode(slip, node, useNow));
		return;
	    }
	    if (!useLater.isEmpty()) {
		// If it looks like we might be able to expand if we
		// introduced some nodes to satisfy conditions,
		// post an alt that does that.  /\/: We might want
		// to allow node-introduction even in the useNow case
		// above.  [Done]
		postExpandLaterAlt(node, useLater);
		// But continue looking for a node that can be
		// executed now.
	    }
	    // At least one refinement applied, but none could be used now.
	    // So try to exec something else.
	    Debug.noteln("No refinement can expand now");
	    continue;
	}
    }

    void maybePostExpandGoalAlt(PNodeEnd selected) {
	// The goal node's condition can be satisfied, so we can
	// delete the goal node.  However, expanding the goal node
	// can also be considered a valid possibility, and it might
	// provide different ways to satisfy the condition --
	// different values for some variable(s).
	// N.B. We have already posted a findNext alt.
	Debug.expect(slip.liberalAchieveSemantics);
	Debug.expect(slip.isGoalNode(selected));
	Debug.expectSame(End.BEGIN, selected.getEnd());
	PNode node = selected.getNode();
	// See if the condition contains any unbound variables.
	// /\/: I'm not sure that's the right test at this point.
	// Probably what we want is whether the cond originally
	// contained any variables whose values weren't
	// absolutely forced.
	PatternAssignment goalCond = slip.getGoalCond(selected);
	if (Variable.isFullyBoundEverywhere(goalCond))
	    // Yes, so don't bother expanding the goal, even though,
	    // strictly speaking, having the expanded node can be
	    // considered a valid possibility anyway and might
	    // provide some effects useful elsewhere.
	    // return;
	    ;
	// See if it is possible to expand the goal.
	LinkedList useNow = new LinkedList();
	LinkedList useLater = new LinkedList();
	if (!nodeExpanders(selected, useNow, useLater))
	    // Can't expand, so don't bother with the alternative.
	    return;
	// /\/: We also get here if an unusableRefinementApplies
	if (!useLater.isEmpty()) {
	    Debug.noteln("Can expand later");
	    Debug.noteln("useLater", useLater);
	    postExpandLaterAlt(node, useLater);
	}
	if (!useNow.isEmpty()) {
	    // The node can be expanded now.
	    Debug.noteln("Can expand goal now");
	    Debug.noteln("useNow", useNow);
	    // Post an alt to do the expansion.
	    slip.postAlternative(new ExpandGoalAlt(node, useNow));
	}
    }

    class ExpandGoalAlt extends Alternative { // cf ExpandLaterAlt

	PNode node;
	LinkedList useNow;

	ExpandGoalAlt(PNode node, LinkedList useNow) {
	    this.node = node;
	    this.useNow = useNow;
	}

	public void run() {
	    slip.setNextStep(new SlipExpandNode(slip, node, useNow));
	}

	protected String extraToStringContents() {
	    return " to expand " + Variable.removeVars(node.getPattern());
	}

    }

    void postFindNextAlt(PNodeEnd selected,
			 LList allNodeEnds,
			 LListIterator nodeEndIter,
			 String doing) {
	// If the selected node-end doesn't work when used now, we don't
	// want to try it again (in a different position relative to other
	// node-ends that could have been selected now) unless the move
	// would make a difference.
	List<PNodeEnd> interacting = null;
	if (!slip.domainSays("iplan-try-pointless-permutations")
	      && slip.useDomainAnalysis) {
	    interacting = findInteractingEnds(selected);
	    if (interacting.isEmpty()) {
		Debug.noteln("No interactions with", selected);
		return;
	    }
	    Debug.noteln(selected + " interacts with", interacting);
	}
	// If we're about to expand, and the node won't have any constraints
	// of its own, then it shouldn't matter too much when it starts
	// "executing".
	if (doing.equals("expand") 
	      && slip.noPossibleTopLevelConstraints(selected)) {
	    Debug.noteln("No possible top-level constraints when expanding.");
	    return;
	}
	// An alt is posted only if there is another executable node-end.
	// Otherwise, the alt, if chosen, would just poison without
	// doing anything interesting.
	if (nodeEndIter.hasNext()) {
	    // Make a new iterator to avoid modifying the existing one.
	    // This is a precaution -- it's not clear it's necessary, /\/
	    // but we need a ListIterator in any case, so that we can
	    // go back 1.
	    LListListIterator i =
		new LListListIterator(nodeEndIter.getTail());
	    // See if there's another executable node-end.
	    PNodeEnd ne = slip.findExecutable(i);
	    if (ne != null) {
		// There is another executable.  Unfortunately, our
		// iterator has now moved one step too far.
		i.previous();
		slip.postAlternative
		    (new Resume(allNodeEnds, i, selected, interacting, doing));
	    }
	}
    }

    class Resume extends Alternative {
	LList allNodeEnds;
	LListIterator nodeEndIter;
	String doing;
	PNodeEnd selected;
	List<PNodeEnd> interacting;

	Resume(LList allNodeEnds, LListIterator nodeEndIter,
	       PNodeEnd selected, List<PNodeEnd> interacting,
	       String doing) {
	    this.allNodeEnds = allNodeEnds;
	    this.nodeEndIter = nodeEndIter;
	    this.selected = selected;
	    this.interacting = interacting;
	    this.doing = doing;
	    this.isLocalChoice = true;
	}

	public void run() {
	    findNext(allNodeEnds, nodeEndIter);
	}

	@Override
	public void whenPicked() {
	    Debug.expect(selected != null);
	    if (interacting != null) {
		Debug.noteln(selected +" must wait for one of "+ interacting);
		slip.isWaiting.put(selected, Boolean.TRUE);
		for (PNodeEnd i: interacting)
		    slip.unwaitTable.addValue(i, selected);
	    }
	}

	protected String extraToStringContents() {
	    return " instead of " + doing + " " + selected;
	}

    }

    /*
     * Interactions between node-ends
     */

    List<PNodeEnd> findInteractingEnds(PNodeEnd selected) {
	// /\/: If any of the nodes we look at has already been expanded,
	// that could reduce their possible conditions or effects.
	DomainAnalyser ann = slip.analysis;
	boolean selectedIsGoal = slip.isGoalNode(selected);
	PNodeEnd selectedAtEnd = selectedIsGoal
	    ? slip.getAtEndForGoal(selected) : null;
	List<PNodeEnd> result = new LinkedList<PNodeEnd>();
	Set conds = slip.getPossibleConditions(selected);
	Set effects = slip.getPossibleEffects(selected);
	Set constraints = slip.getPossibleOtherConstraints(selected);
	long mark = selected.markShadowedEnds(0);
	for (PNodeEnd ne: slip.MM().getNodeEnds()) {
	    if (ne == selected)
		continue;
	    // Look only at node-ends that look like they could execute now
	    // or if helped by something that could execute now.
	    if (slip.getStatus(ne) != Status.POSSIBLE
		  && slip.getStatus(ne) != Status.BLANK)
		continue;
	    // Skip ends that could not possibly go before the selected one.
	    if (ne.isMarked(0, mark))
		continue;
	    if (selectedIsGoal && slip.isGoalNode(ne)
		  && selectedAtEnd == slip.getAtEndForGoal(ne)) {
		// Since executing one goal node can let another one vanish,
		// goal nodes can interact even when their conds and effects
		// don't show it.  See the achieve-introduction-test domain
		// for an example.
		result.add(ne);
		continue;
	    }
	    if (interactingCondsAndEffects(selected, conds, effects, ne)) {
		result.add(ne);
		continue;
	    }
	    if (slip.possiblyInteractingConstraints(selected, constraints, ne))
		result.add(ne);
	}
	return result;
    }

    boolean interactingCondsAndEffects(PNodeEnd selected,
				       Set conds, Set effects,
				       PNodeEnd ne) {
	DomainAnalyser ann = slip.analysis;
	// Reasons to try the selected end after the item end:
	// * Some conds might be satisfied by the item's effects.
	// * Some effects might be in the way of something else
	//   that could satisfy the item's conds.  (There needs
	//   to be something else.)
	// * Some effects might satisfy some other end's conds
	//   in a different way than the item's effects do.
	//   (There needs to be such conds.)
	Set itemEffects = slip.getPossibleEffects(ne);
	if (ann.haveCommonPatterns(conds, itemEffects)) {
	    return true;
	}
	Set itemConds = slip.getPossibleConditions(ne);
	if (ann.haveCommonPatterns(effects, itemConds)) {
	    // See if the conds could still be satisfied if
	    // 'selected' were after 'ne'.
	    if (couldStillSatisfyConds(ne, selected))
		return true;
	}
	if (ann.haveCommonPatterns(effects, itemEffects)) {
	    // See of there's a node-end that could be after 'ne'
	    // and that has conds that might be satisfied by 'selected's
	    // effects.
	    // Mark ends that can't be after 'ne'.
	    long mark1 = ne.markEndsBefore(1); // 0's in use
	    for (PNodeEnd e: slip.MM().getNodeEnds()) {
		// Skip completed ends and ones that can't be after 'ne'.
		if (slip.getStatus(e) == Status.COMPLETE
		    || e.isMarked(1, mark1)
		    || e == selected)
		    continue;
		// Get 'e's possible conditions.
		Set endConds = slip.getPossibleConditions(e);
		if (ann.haveCommonPatterns(effects, endConds))
		    return true;
	    }
	}
	return false;
    }

    private boolean couldStillSatisfyConds(PNodeEnd ne, PNodeEnd selected) {
	// We want to know if 'ne' can satisfy its conds without any
	// help from 'selected'.  We're going to look only at conds
	// on 'ne' itself, not on any children, since working it out
	// for children is much harder.  If 'ne' has already been
	// expanded, then all of its conds are already in the MM;
	// otherwise, we have to look at possible refinements, and
	// then 'ne's conds can be satisfied if any of the refinement's
	// can be.
	//   We have to be right if we say the conds cannot still be
	// satisfied, but not if we say they can.
	//   This routine is always called from within findInteractingEnds,
	// and the 0 mark is already in use.  At present, we know that
	// END ends do not have any conditions and that therefore we
	// should not be called for them.  /\/
	Debug.expectSame(End.BEGIN, ne.getEnd());
	PNode node = ne.getNode();
	// We know 'selected' was marked when marking all node-ends  /\/
	// linker after it, and we'll need to identify those node-ends,
	// because they can't be used to satisfy the conds.
	long mark0 = selected.getMark(0);
	// Mark 'ne' and everything after it, because those node-ends
	// also cannot be used to satisfy the conds.
	long mark1 = ne.markShadowedEnds(1);
	if (isAlreadyExpanded(ne)) {
	    // If the node's been expanded, all of its conds are
	    // already in the MM.
	    ListOfConstraint conds = slip.MM().getNodeConditions(node);
	    return canSatisfyAll(conds, mark0, mark1);
	}
	// The node hasn't been expanded, so we have to look at
	// refinements.  It's enough if any of the refinement's
	// conds all looks satisfiable.
	else if (slip.isGoalNode(ne)) {
	    List achievers = slip.getGoalAchievers(node);
	    for (Iterator i = achievers.iterator(); i.hasNext();) {
		SlipAchiever achiever = (SlipAchiever)i.next();
		Refinement r = achiever.refinement;
		Debug.expect(!Collect.isEmpty(r.getConstraints()));
		// /\/: We should use the env from when the cond was
		// matched to the refinement.   See #goalExpanders.
		ListOfConstraint filters = r.getFilterConstraints();
		if (canSatisfyAll(filters, mark0, mark1))
		    return true;
	    }
	    return false;
	}
	else {
	    AdviceManager advice = slip.MM().getAdviceManager();
	    LList nodePat = node.getPattern();
	    for (Refinement r: slip.getDomain().getRefinements()) {
		MatchEnv e = Matcher.match(r.getPattern(), nodePat);
		if (e == null || !advice.isUsableRefinement(r))
		    continue;
		// /\/: We should use the env.  See #nodeExpanders.
		ListOfConstraint filters = r.getFilterConstraints();
		if (canSatisfyAll(filters, mark0, mark1))
		    return true;
	    }
	    return false;
	}
    }

    private boolean canSatisfyAll(ListOfConstraint conds,
				  long mark0, long mark1) {
	// We assume that COMPUTE conds can be satisfied and
	// look only at WORLD-STATE CONDITIONs.
	DomainAnalyser ann = slip.analysis;
	condLoop:
	for (Constraint c: conds) {
	    if (c.getType() != Refinement.S_WORLD_STATE)
		continue condLoop;
	    // If the cond can be satisfied from the current world-state,
	    // that's enough.
	    if (worldStateSatisfiesCond(c))
		continue condLoop;
	    // If there are achievers for the cond, that's sufficient too.
	    if (slip.existsAchieversForCond(c))
		continue condLoop;
	    // Now there must be a parallel node that might satisfy it.
	    PatternAssignment pv = c.getPatternAssignment();
	    for (PNodeEnd ne: slip.MM().getNodeEnds()) {
		// Skip completed and shadowed node-ends.
		if (slip.getStatus(ne) == Status.COMPLETE
		    || ne.isMarked(0, mark0)
		    || ne.isMarked(1, mark1))
		    continue;
		Set possibleEffects = slip.getPossibleEffects(ne);
		if (ann.mightSatisfy(pv, possibleEffects))
		    continue condLoop;
	    }
	    return false;
	}
	return true;
    }

    /*
     * Condition satisfaction
     */

    Constraint makeCondition(PatternAssignment pv) {
	return new Constraint
	    (Refinement.S_WORLD_STATE, Refinement.S_CONDITION,
	     Lisp.list(pv));
    }

    void satisfyNodeEndConds(PNodeEnd ne, ListOfConstraint conds, List envs) {
	AgendaItem item = (AgendaItem)ne.getNode();
	if (envs.size() == 1) {
	    // Only one way to satisfy the conds, so we can just
	    // do it.
	    slip.MM().satisfyConds(item, conds, (MatchEnv)envs.get(0));
	    slip.execBegin(ne);
	    return;
	}
	// Multiple envs.
	// /\/: Here we rely on the envs being a LinkedList so that
	// we can use them as a queue in alternatives.
	LinkedList condEnvs = (LinkedList)envs;
	MatchEnv e1 = (MatchEnv)condEnvs.removeFirst();
	slip.postAlternative
	    (new SlipSatisfyConds(slip)
	     .new ResumeSatisfaction
	            (item, conds, condEnvs));
	slip.MM().satisfyConds(item, conds, e1);
	slip.execBegin(ne);
    }

    void postSatisfyByAchieveAlt(PNode node, ListOfConstraint conds) {
	slip.postAlternative(new SatisfyByAchieveAlt(node, conds));
    }

    class SatisfyByAchieveAlt extends Alternative {

	PNode node;
	ListOfConstraint conds;

	SatisfyByAchieveAlt(PNode node, ListOfConstraint conds) {
	    this.node = node;
	    this.conds = conds;
	}

	public void run() {
	    slip.setNextStep(new SlipAchieveConds(slip, node, conds));
	}

    }

    /**
     * Determines whether all the conditions can be satisfied
     * individually either by the current world-state or by
     * introducing nodes.  It does not determine whether all
     * the conds can be satisfied together; nor does it determine
     * that introducing nodes will definitely work, because it
     * might not be possible to get the right variable bindings.
     * At least one of the conditions must look like it can be
     * solved by introducing a node; for now at least /\/ this method
     * is called only when it's already known that the world state
     * cannot satisfy all of the conds together.
     */
    boolean mightAchieveConds(ListOfConstraint conds) {
	boolean haveAnAchievable = false;
	for (Iterator ci = conds.iterator(); ci.hasNext();) {
	    Constraint c = (Constraint)ci.next();
	    if (c.getType() != Refinement.S_WORLD_STATE)
		continue;
	    if (condMightBeAchieved(c))
		haveAnAchievable = true;
	    else if (!worldStateSatisfiesCond(c))
		return false;
	}
	return haveAnAchievable;
    }

    boolean worldStateSatisfiesCond(Constraint cond) {
	//\/ Inefficient but simple.
	ListOfConstraint conds = new LinkedListOfConstraint();
	conds.add(cond);
	List envs = slip.MM().evalFilters(conds, new MatchEnv());
	return !envs.isEmpty();
    }

    private boolean condMightBeAchieved(Constraint cond) {
	return slip.existsAchieversForCond(cond);
    }

    /*
     * Node-end utilities
     */

    LList getNodeEnds() {
	// /\/: We use an LList because we know we can copy iterators.
	return LList.newLList(slip.MM().getNodeEnds());
    }

    LListIterator endIterator(LList l) {
	// This is called to get the initial nodeEndIter
	// when we want to find an executable.  We can pre-process
	// the list to execute any "trivial" node-ends -- ones
	// that have neither conditions nor effects -- because
	// it doesn't matter when they're executed relative to
	// other node-ends.  This reduces the number of permutations
	// that might be considered, because the trivial ends
	// aren't permuted.
	if (!slip.domainSays("iplan-try-pointless-permutations"))
	    executeTrivialEnds(l);
	return (LListIterator)l.iterator();
    }

    void executeTrivialEnds(LList nodeEnds) {
	boolean changed = true;
	while (changed) {
	    changed = false;
	    for (Iterator i = nodeEnds.iterator(); i.hasNext();) {
		PNodeEnd ne = (PNodeEnd)i.next();
		if (slip.getStatus(ne) == Status.POSSIBLE && isTrivial(ne)) {
		    Debug.noteln("Pre-executing", ne);
		    slip.setStatus(ne, Status.COMPLETE);
		    changed = true;
		}
	    }
	}
    }

    boolean isTrivial(PNodeEnd ne) {
	PNode node = ne.getNode();
	if (ne.getEnd() == End.BEGIN) {
	    if (slip.isGoalNode(ne))
		return false;
	    if (!isAlreadyExpanded(ne))
		return !someRefinementApplies(node);
	    ListOfConstraint conditions = slip.MM().getNodeConditions(node);
	    if (conditions != null && !conditions.isEmpty())
		return false;
	    List constraints = slip.MM().getOtherNodeConstraints(node);
	    return constraints == null || constraints.isEmpty();
	}
	else {
	    List effects = slip.MM().getNodeEffects(node);
	    if (effects != null && !effects.isEmpty())
		return false;
	    List constraints = slip.MM().getOtherNodeConstraints(node);
	    return constraints == null || constraints.isEmpty();
	}
    }

    boolean someRefinementApplies(PNode node) {
	LList pattern = node.getPattern();
	for (Iterator i = slip.getDomain().getRefinements().iterator()
		 ; i.hasNext();) {
	    Refinement r = (Refinement)i.next();
	    if (refinementApplies(r, pattern))
		return true;
	}
	return false;
    }

    boolean refinementApplies(Refinement r, LList pattern) {
	// Determines whether the refinement is intended for
	// expanding activities with patterns like the one given.
	// This is not completely well-defined.  /\/
	LList refpat = r.getPattern();
	if (!refpat.car().equals(pattern.car()))
	    return false;
	int rlen = refpat.length(), plen = pattern.length();
	return rlen == plen
	    || (rlen-2 <= plen && refpat.contains(Matcher.REST));
    }

    PNodeEnd nextExecutable(Iterator i) {
	PNodeEnd ne = slip.findExecutable(i);
	if (ne == null) {
	    if (slip.planIsComplete())
		throw new Slip.HavePlanException();
	    else
		throw slip.poison("no executable node-end");
	}
	return ne;
    }

    boolean isAlreadyExpanded(PNodeEnd ne) {
	Debug.expectSame(End.BEGIN, ne.getEnd());
	PNode node = ne.getNode();
	if (node.isExpanded()) {
	    Debug.expect(!slip.isGoalNode(ne));
	    return true;
	}
	else {
	    // If not expanded, there should (for now /\/) be
	    // no childred or conditions.
	    Debug.expect(node.getChildren().isEmpty(),
			 "Has children when not expanded", node);
	    Debug.expect(Collect.isEmpty(slip.MM().getNodeConditions(node)),
			 "Has conds when not expanded", node);
	    return false;
	}
    }

    /*
     * Expansion
     */

    boolean nodeExpanders(PNodeEnd ne, LinkedList useNow,
			               LinkedList useLater) {
	// Uses LinkedLists because they can be used like queues
	// in SlipExpandNode.
	// Returns false when no refinements apply, which means the
	// node needn't be expanded and can be treated as a primitive.
	if (slip.isGoalNode(ne))
	    return goalExpanders(ne, useNow, useLater);
	AgendaItem item = (AgendaItem)ne.getNode();
	boolean unusableRefinementApplies = false;
	AdviceManager advice = slip.MM().getAdviceManager();
	for (Iterator i = slip.getDomain().getRefinements().iterator()
		 ; i.hasNext();) {
	    Refinement r = (Refinement)i.next();
	    LList itemPat = item.getPattern();
	    // See if the refinement's pattern matches the node-end's.
	    // Advice may say a refinement cannot be used even though
	    // it matches.
	    MatchEnv e = Matcher.match(r.getPattern(), itemPat);
	    if (e == null || !advice.isUsableRefinement(r)) {
		// Refinement doesn't match, but perhaps it "applies".
		unusableRefinementApplies = unusableRefinementApplies
		    || refinementApplies(r, itemPat);
		continue;
	    }
	    if (Collect.isEmpty(r.getNodes())
		     && Collect.isEmpty(r.getConstraints())) {
		// When there are no subnodes or constraints, the
		// refinement name still specifies a way of doing
		// it manually.
		useNow.add(makeManualExpander(item, r, e));
	    }
	    else {
// 		// See if the refinement can be used now.
// 		SlipExpander expander = makeExpander(item, r, e);
// 		if (expander != null) {
// 		    // Add action to expand using the refinement.
// 		    useNow.add(expander);
// 		    continue;
// 		}
// 		// See if the refinement might be used if some nodes
// 		// were added to satisfy some conditions.
// 		expander = makeLaterExpander(item, r, e);
// 		if (expander != null) {
// 		    useLater.add(expander);
// 		    continue;
// 		}
// 		// The refinement matched the node's pattern, and
// 		// therefore applies, but cannot be used.
// 		unusableRefinementApplies = true;
		// See if the refinement can be used now.
		SlipExpander now = makeExpander(item, r, e);
		SlipExpander later = null;
		if (now == null || slip.liberalAchieveSemantics)
		    // See if the refinement might be used if some nodes
		    // were added to satisfy some conditions.
		    later = makeLaterExpander(item, r, e);
		if (now != null)
		    useNow.add(now);
		if (later != null)
		    useLater.add(later);
		if (now == null && later == null)
		    // The refinement matched the node's pattern, and
		    // therefore applies, but cannot be used.
		    unusableRefinementApplies = true;
	    }
	}
	return unusableRefinementApplies ||
	       !useNow.isEmpty() || !useLater.isEmpty();
    }

    boolean goalExpanders(PNodeEnd ne, LinkedList useNow,
			               LinkedList useLater) {
	// Always returns true (meaning some refinements apply),
        // because we wouldn't have created the goal node if
	// none applied.  ["Applies" here means that the
	// refinement has a matching effect, so it's stricter
	// than for expansion.]  We also know the refinement
	// must have at least one effect, so we don't have to
	// consider the "manual expander" case.
	AgendaItem item = (AgendaItem)ne.getNode();
	List achievers = slip.getGoalAchievers(item);
	for (Iterator i = achievers.iterator(); i.hasNext();) {
	    SlipAchiever achiever = (SlipAchiever)i.next();
	    Refinement r = achiever.refinement;
	    Debug.expect(!Collect.isEmpty(r.getConstraints()));

	    // Some Variables may have been bound since we
	    // matched the condition and effect.
	    MatchEnv e = achiever.rematchCond();
	    if (e == null) {
		Debug.noteln("Can't use " + achiever +
			     " for goal " + item +
			     " because " + achiever.condMatchEnv +
			     " no longer applies");
		continue;
	    }

	    // See if the refinement can be used now.
	    SlipExpander now = makeExpander(item, r, e);
	    if (now != null) {
		// Add action to expand using the refinement.
		useNow.add(now);
		if (! slip.liberalAchieveSemantics)
		    continue;
	    }
	    // See if the refinement might be used if some nodes
	    // were added to satisfy some conditions.
	    SlipExpander later = makeLaterExpander(item, r, e);
	    if (later != null) {
		useLater.add(later);
		continue;
	    }
	    if (now != null) continue;
	    // The refinement cannot be used.
	    Debug.noteln("Can't use " + achiever + " for goal " + item);
	}
	if (useNow.isEmpty() && useLater.isEmpty())
	    Debug.noteln("Cannot expand goal", item);
	return true;
    }

    SlipExpander makeManualExpander(AgendaItem item, Refinement r,
				    MatchEnv env) {
	// We're using filterEnvs == null to say the filters could not
	// be satisfied, so when they're satisfied trivially, because
	// there weren't any, we need to list the env from matching
	// the expands pattern.
	List filterEnvs = Lisp.list(env);
	return new SlipExpander(slip, item, r, env, filterEnvs);
    }

    SlipExpander makeExpander(AgendaItem item, Refinement r, MatchEnv env) {
	MatchEnv refinementMatchEnv = env;
	ListOfConstraint filters = r.getFilterConstraints();
	List filterEnvs = filters.isEmpty()
	    ? Collections.EMPTY_LIST
	    : slip.MM().evalFilters(filters, refinementMatchEnv);
	boolean isReady = filters.isEmpty() || !filterEnvs.isEmpty();
	if (isReady)
	    return new SlipExpander(slip, item, r,
				    refinementMatchEnv, filterEnvs);
	else
	    return null;
    }

    SlipExpander makeLaterExpander(AgendaItem item, Refinement r,
				   MatchEnv env) {
	// This is called when there's no way to satisfy all of the
	// refinement's filter conds from the current world-state.
	// However, it still might be possible to satisfy them
	// individually or by adding nodes.
	if (!slip.isWaitingForAchieve(item) // ensure not already achieving
	      && mightAchieveConds(r.getFilterConstraints()))
	    return new SlipExpander(slip, item, r, env, null);
	else
	    return null;
    }

    void postExpandLaterAlt(PNode node, LinkedList useLater) {
	slip.postAlternative(new ExpandLaterAlt(node, useLater));
    }

    class ExpandLaterAlt extends Alternative {

	PNode node;
	LinkedList useLater;

	ExpandLaterAlt(PNode node, LinkedList useLater) {
	    this.node = node;
	    this.useLater = useLater;
	}

	public void run() {
	    // /\/: It's not clear that either call to releaseWaitingEnds
	    // should happen now, but for some reaon we have to do the
	    // one for the BEGIN end to make the achieve-limitation-test-1
	    // domain work.  Still, perhaps it could be done somewhere
	    // else.
	    slip.releaseWaitingEnds(node.getBegin());
	    // slip.releaseWaitingEnds(node.getEnd()); // now? /\/
	    slip.setNextStep(new SlipExpandNode(slip, node, useLater));
	}

	protected String extraToStringContents() {
	    return " to expand " + Variable.removeVars(node.getPattern());
	}

    }

}
