/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Mon Jun  9 22:02:13 2008 by Jeff Dalton
 * Copyright: (c) 2004, 2005, 2007, 2008, AIAI, University of Edinburgh
 */

package ix.iplan;

import java.util.*;

import ix.ip2.*;
import ix.icore.*;
import ix.icore.process.*;
import ix.icore.domain.*;
import ix.util.*;
import ix.util.lisp.*;
import ix.util.match.*;

/**
 * Introduces nodes intended to satisfy preconditions of a node
 * already in the plan.
 */
class SlipAchieveConds implements Slip.Step {

    // Our situation:

    // 1. The node has been expanded.

    // 2. The node has some conditions, and at least one of them matches
    //    at least one refinement effect.

    // 3. It isn't possible to satisfy all of the conditions together
    //    using the current world-state.  It may, however, be possible
    //    to satisfy each cond individually.  (We might later decide
    //    we should come here even if all conds can be satisfied together,
    //    but for now we don't.)

    // For each cond that matches any refinement effects, we want
    // to either introduce a node, and expand it using one of those
    // refinments, or else do nothing (here) about the cond and hope
    // that it will be satisfiable from the world-state by the time
    // the cond's node tries to execute.

    // It may seem that we have to introduce a node for at least one cond,
    // because the case where it's clear that no nodes are needed
    // is handled elsewhere.  (See 3. above.)  However, there may
    // be existing, currently "in parallel", nodes that could
    // satisfy some of the conditions if they executed soon enough.

    // We have to remember that we're "achieving" for the node (in this
    // context) so that when we reconsider the node for execution we
    // don't start the achieving process all over again.

    // /\/: When an introduced node is expanded, we need to tie the effect
    // to the cond it is supposed to satisfy so that we can protect
    // their relationship from potential deleters.  The relationship
    // may also force (or at least constrain) some variable values.

    Slip slip;
    AgendaItem item;
    ListOfConstraint conds;

    SlipAchieveConds(Slip slip, PNode node, ListOfConstraint conds) {
	this.slip = slip;
	this.item = (AgendaItem)node; // should use node instead /\/
	this.conds = conds;
    }

    public void run() {
	// Make sure we don't restart the achieving process for a node
	// that's already doing it.
	Debug.expect(!slip.isWaitingForAchieve(item));
	slip.recordAsWaitingForAchieve(item);

	// Put in goal nodes
	// List goalNodes = makeGoalNodes();
	List goalNodes = new LinkedList();
	ListOfConstraint evalNow = new LinkedListOfConstraint();
	ListOfConstraint impossible = new LinkedListOfConstraint();
	if (!findSatisfactionMethods(goalNodes, evalNow, impossible)) {
	    // Debug.noteln("World state:", slip.MM().getWorldStateMap());
	    throw slip.poison
		("Cannot satisfy conditions at " + item + ": " +
		 impossible);
	}
	// Here's where we actually add the goal nodes to the model.
	Debug.expect(!Collect.isEmpty(goalNodes), "no goal nodes");
	Agenda actAgenda = slip.getController().getActivityAgenda();
	actAgenda.addItemsBefore(item, goalNodes);
	// The goal nodes don't yet have anything ordered before
	// them, so they should get status POSSIBLE.  They should
	// also be ordered before the node that needs the conditions
	// satisfied.
	for (Iterator i = goalNodes.iterator(); i.hasNext();) {
	    PNode n = (PNode)i.next();
	    slip.setStatus(n.getBegin(), Status.POSSIBLE);
	    n.getEnd().linkBefore(item.getBegin());
	}

	// Change the status of the node that needs conditions
	// achieved back to BLANK.
	Debug.expectSame(Status.POSSIBLE, slip.getStatus(item.getBegin()));
	Debug.expectSame(Status.BLANK, slip.getStatus(item.getEnd()));
	slip.setStatus(item.getBegin(), Status.BLANK);

	// Check that the goal nodes are there.
	// slip.describeNodeEnds(slip.MM().getNodeEnds());

	// Decide what to do next.
	if (evalNow.isEmpty())
	    // Go back to finding node-ends to execute.
	    slip.setNextStep(new SlipFindExecutable(slip));
	else {
	    // First satisfy the conds that have to be
	    // satisfiable now.
	    Debug.noteln("evalNow:", evalNow);
	    slip.setNextStep(new SatisfyEvalNowConds(slip, item, evalNow));
	}

    }
    
    static class SatisfyEvalNowConds extends SlipSatisfyConds {

	SatisfyEvalNowConds(Slip slip, PNode node, ListOfConstraint conds) {
	    super(slip, node, conds);
	}

	void adjustStatus(AgendaItem item) {
	    // Normally, this method makes the begin node-end's
	    // status COMPLETE, but we're not ready to execute it.
	    Debug.noteln("Satisfied eval-now conds for", item);
	    Debug.noteln("Eval-now conds:", conds);
	}

	void checkCondsSatisfied(List envs) {
	    if (envs.isEmpty()) {
		throw slip.poison
		    ("cannot satisfy eval-now conditions at " + item + ": " +
		     conds);
	    }
	}

    }

    /*
    List makeGoalNodes() {
	List result = new LinkedList();
	for (Iterator i = conds.iterator(); i.hasNext();) {
	    Constraint cond = (Constraint)i.next();
	    if (cond.getType() != Refinement.S_WORLD_STATE)
		continue;
	    PatternAssignment pv = cond.getPatternAssignment();
	    List achievers = slip.getAchieversForCond(pv);
	    if (!achievers.isEmpty()) {
		PNode goalNode = slip.makeGoalNode(pv, achievers);
		// Link goal node before the node that has the condition.
		goalNode.getEnd().linkBefore(item.getBegin());
		result.add(goalNode);
	    }
	}
	return result;
    }
    */

    boolean findSatisfactionMethods
        // Returns 4 results: its boolean value and:
	    (List goalNodes,	// new nodes that might satisfy conds
	     List evalNow,	// conds that must be eval'd now
	     List impossible)	// conds that cannot be satisfied now
    {
	// /\/: Doesn't check slip.useDomainAnalysis.
	Debug.expect(goalNodes.isEmpty());
	Debug.expect(evalNow.isEmpty());
	Debug.expect(impossible.isEmpty());
	PNodeEnd atEnd = item.getBegin(); // /\/ assume at begin_of
	long shadowedMark = -1;
	ListOfConstraint toTest = new LinkedListOfConstraint();
	for (Iterator i = conds.iterator(); i.hasNext();) {
	    Constraint cond = (Constraint)i.next();
	    // We skip non-WORLD-STATE constraints apart from COMPUTEs.
	    if (cond.getType() != Refinement.S_WORLD_STATE) {
		// Check that the vars needed by the COMPUTE will be bound
		// buy the "eval now"s that were before it, and add it to
		// the "eval now"s if so.
		if (cond.getType() == Refinement.S_COMPUTE
		      && willBindVars(evalNow, cond))
		    evalNow.add(cond);
		continue;
	    }
	    // See if there are refinements that look like they could
	    // satisfy the cond if they were used to expand a node.
	    PatternAssignment pv = cond.getPatternAssignment();
	    List achievers = slip.getAchieversForCond(cond);
	    if (!achievers.isEmpty()) {
		// It looks like we can satisfy the cond by adding a node,
		// so create one.
		PNode goalNode = slip.makeGoalNode(pv, achievers, atEnd);
		goalNodes.add(goalNode);
		continue;
	    }
	    // At this point, we have a world-state condition
	    // that can't be achieved.  See if it might be
	    // satisfied by a parallel node.
	    if (shadowedMark == -1)
		shadowedMark = item.getBegin().markShadowedEnds(0);
	    if (parallelMightSatisfy(pv, shadowedMark))
		continue;
	    // Now we have a world-state condition that would
	    // have to be satisfied by the current world state.
	    evalNow.add(cond);
	    // Check against the world state here, and (evantually /\/)
	    // return false if it won't satisfy the cond.
	    toTest.clear();
	    toTest.add(cond);
	    List satisfied = slip.MM().testFilters(toTest, Matcher.emptyEnv);
	    if (satisfied.isEmpty())
		impossible.add(cond);
	}
	return impossible.isEmpty();
    }

    boolean willBindVars(List evalNow, Constraint cond) {
	// This checks that those "evaluate now" constraints
	// will bind all the variables needed by the COMPUTE cond.
	Debug.expectSame(Refinement.S_COMPUTE, cond.getType());
	Debug.noteln("willBindVars in " + cond + " from " + evalNow);
	Set need = Variable.unboundVarsAnywhereIn(cond.getPattern());
	for (Iterator i = evalNow.iterator(); i.hasNext();) {
	    Debug.noteln("Need", need);
	    if (need.isEmpty())
		return true;
	    Constraint c = (Constraint)i.next();
	    need.removeAll(Variable.varsAnywhereIn(c.getPatternAssignment()));
	}
	Debug.noteln("Finally need", need);
	return need.isEmpty();
    }

    boolean parallelMightSatisfy(PatternAssignment pv, long shadowedMark) {
	Debug.noteln("Parallel might satisfy:", pv);
	DomainAnalyser ann = slip.analysis;
	for (Iterator i = slip.MM().getNodeEnds().iterator(); i.hasNext();) {
	    PNodeEnd ne = (PNodeEnd)i.next();
	    // Skip completed and shadowed node-ends.
	    if (slip.getStatus(ne) == Status.COMPLETE
		  || ne.isMarked(0, shadowedMark))
		continue;
	    Set possibleEffects = slip.getPossibleEffects(ne);
	    // Debug.noteln("Possible effects of " + ne, possibleEffects);
	    if (ann.mightSatisfy(pv, possibleEffects))
		return true;
	}
	return false;
    }

}
