/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Tue Oct  7 17:37:48 2008 by Jeff Dalton
 * Copyright: (c) 2008, AIAI, University of Edinburgh
 */

package ix.iplan;

import java.util.*;

import ix.ip2.*;
import ix.icore.*;
import ix.icore.process.*;
import ix.icore.domain.End;

import ix.util.*;
import ix.util.lisp.*;

/**
 * Compares plans in-place: in their model-managers.  Some differences
 * are allowed on the assumption that the 'original' plan was produced
 * directly by the automatic planner while the 'copy' was converted
 * to a Plan object and then loaded into a new model.  This class is
 * intended primarily as a bug-finding aid.
 *
 * @see #okMissingSuccessor(PNodeEnd, PNodeEnd)
 */
public class PlanModelComparer {

    Ip2ModelManager originalMM;
    Ip2ModelManager copyMM;

    Map<ActivityItem,ActivityItem> nodeMap;

    public PlanModelComparer(Ip2ModelManager original, Ip2ModelManager copy) {
	this.originalMM = original;
	this.copyMM = copy;
    }

    public void compare() throws PlanDifferenceException {
	nodeMap = makeNodeMap();
	compareOrderings();
    }

    public static class PlanDifferenceException extends RuntimeException {
	PlanDifferenceException(String message) {
	    super(message);
	}
    }

    private PlanDifferenceException problem(String message) {
	return new PlanDifferenceException(message);
    }

    private Map<ActivityItem,ActivityItem> makeNodeMap() {

	Map<ActivityItem,ActivityItem> map =
	    new LinkedHashMap<ActivityItem,ActivityItem>();

	Iterator<ActivityItem> copy_i = getActivities(copyMM).iterator();

	for (ActivityItem item: getActivities(originalMM)) {
	    if (!copy_i.hasNext())
		throw problem("Too few nodes in copy");
	    ActivityItem copy = copy_i.next();
	    LList pat = (LList)Variable.removeVars(item.getPattern());
	    LList copyPat = (LList)Variable.removeVars(copy.getPattern());
	    if (pat.equals(copyPat))
		map.put(item, copy);
	    else
		throw problem("Mismatch: " + pat + " vs " + copyPat);
	    
	}
	if (copy_i.hasNext())
	    throw problem("Too many nodes in copy");

	return map;

    }

    /**
     * Return activites in their natural execution order.
     */
    private List<ActivityItem> getActivities(final Ip2ModelManager mm) {
        // The Ref<Proc> stops the compiler from complaining that
        // "variable walker might not have been initialized" while
        // still letting 'walker' be final. /\/
	final List<ActivityItem> items = new LinkedList<ActivityItem>();
	final Ref<Proc> walker = new Ref<Proc>();
	walker.set(new Proc() {
	    public void call(Object a) {
		ActivityItem item = (ActivityItem)a;
		items.add(item);
		mm.walkNodeChildren(item, walker.get());
	    }
	});
	mm.walkTopNodes(walker.get());
	return items;
    }

    private void compareOrderings() {
	for (ActivityItem item: nodeMap.keySet()) {
	    ActivityItem copy = nodeMap.get(item);
	    compareOrderings(item.getBegin(), copy.getBegin());
	    compareOrderings(item.getEnd(), copy.getEnd());
	}
    }

    private void compareOrderings(PNodeEnd ne, PNodeEnd copy) {
	Set uniqueSuccessors = new LinkedHashSet(ne.getSuccessors());
	List mapped = mapEndsToCopy(uniqueSuccessors);
	List copySuccessors = copy.getSuccessors();
	// mapped and copySuccessors should be equal, at least as sets.
	List copyExtras = (List)Collect.difference(copySuccessors, mapped);
	List copyMissing = (List)Collect.difference(mapped, copySuccessors);
	dropOkMissingSuccessors(copy, copyMissing);
	if (! (copyExtras.isEmpty() && copyMissing.isEmpty()))
	    throw problem
		(ne + " has different successors.\n" +
		 "   copy extras: " + copyExtras + "\n" +
		 "   copy missing: " + copyMissing);
    }

    private void dropOkMissingSuccessors(PNodeEnd copy, List copyMissing) {
	for (Iterator i = copyMissing.iterator(); i.hasNext();) {
	    if (okMissingSuccessor(copy, (PNodeEnd)i.next()))
		i.remove();
	}
    }

    /**
     * Decides whether it's ok for a node-end to be missing a successor
     * in the copy.  The idea here is (1) that in the original, a node was
     * expanded, then ensureChildrenLinkedToParent was called, and then
     * some planning was done, which could add more links which, if they'd
     * beed there originally -- as they will be in the copy -- would have
     * let ensureChildrenLinkedToParent add fewer links.  So when
     * ensureChildrenLinkedToParent is called by the PlanInstaller,
     * it does in fact add fewer links.  And (2) that begin_of parent
     * --> end_of child links can be added because a precondition
     * of the parent was affected by an effect of the child, or a
     * begin_of child --> end_of parent might have been added because
     * a precondition of the child was affected by one of the parent's
     * effects, but those links are implied by the ensureChildrenLinkedToParent
     * ones.
     */
    protected boolean okMissingSuccessor(PNodeEnd copy, PNodeEnd missing) {
	// /\/: It would be easier just to say that any missing links
	// that involve parent and child ends are ok, because they're
	// implied by the ensureChildrenLinkedToParent ones.
	// See the comment in the PlanMaker buildChildSiblingOrderings.
	return okMissingSuccessor1(copy, missing)
	    || okMissingSuccessor2(copy, missing);
    }

    private boolean okMissingSuccessor1(PNodeEnd copy, PNodeEnd missing) {
	// /\/: getShadowedEnds is arguably too expensive
	return (copy.getNode().isParentOf(missing.getNode())
		&& copy.getEnd() == End.BEGIN
		&& missing.getEnd() == End.BEGIN
		&& copy.getShadowedEnds().contains(missing))
	    || (missing.getNode().isParentOf(copy.getNode())
		&& copy.getEnd() == End.END
		&& missing.getEnd() == End.END
		&& copy.getShadowedEnds().contains(missing));
    }

    private boolean okMissingSuccessor2(PNodeEnd copy, PNodeEnd missing) {
	return (copy.getNode().isParentOf(missing.getNode())
		&& copy.getEnd() == End.BEGIN
		&& missing.getEnd() == End.END)
	    || (missing.getNode().isParentOf(copy.getNode())
		&& missing.getEnd() == End.END
		&& copy.getEnd() == End.BEGIN);
    }

    private List mapEndsToCopy(Collection nodeEnds) {
	List result = new LinkedList();
	for (Object ne_i: nodeEnds) {
	    PNodeEnd ne = (PNodeEnd)ne_i;
	    PNodeEnd copyEnd = getCopyEnd(ne);
	    Debug.expect(copyEnd != null);
	    result.add(copyEnd);
	}
	return result;
    }

    private PNodeEnd getCopyEnd(PNodeEnd ne) {
	PNode copyNode = nodeMap.get(ne.getNode());
	return ne.getEnd() == End.BEGIN
	    ? copyNode.getBegin()
	    : copyNode.getEnd();
    }

}


