/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Mon Jun  2 16:58:58 2008 by Jeff Dalton
 * Copyright: (c) 2004 - 2008, AIAI, University of Edinburgh
 */

package ix.iplan;

import java.io.*;
import java.util.*;

import ix.ip2.*;
import ix.icore.*;
import ix.icore.process.*;
import ix.icore.plan.Plan;
import ix.icore.domain.*;

import ix.iface.util.KeyValueTable; // for a comparator

import ix.util.*;
import ix.util.lisp.*;

/**
 * A simple plan-execution simulator that complains if any condition
 * is not satisfied when a node-end is executed after obeying all
 * ordering constraints.  To ensure that it's an independent check,
 * it doesn't share any interesting code with the planner.  It
 * doesn't even use a status-change propagation algorithm to
 * determine the execution order; it just does a topological sort
 * of the node-ends.  However, it independently maintains a set
 * of ready-to-execute node-ends in order to check that they wouldn't
 * delete any of the executing node-end's conditions if they'd been
 * chosen to execute instead.
 *
 * <p><i>This class is partly based on parts of O-Plan.</i></p>
 *
 * @see AutoTester
 * @see PlanTest
 */
public class PlanCheckingSimulator {

    protected static final Symbol TRUE = Symbol.intern("true");
    protected static final Symbol FALSE = Symbol.intern("false");

    protected Ip2ModelManager modelManager;

    protected ComputeInterpreter computeInterpreter;

    protected SanityChecker sanityChecker;

    protected TwoKeyHashMap<Symbol,Symbol,ConstraintChecker> checkerTable =
	new TwoKeyHashMap<Symbol,Symbol,ConstraintChecker>();

    protected Map worldState;
    protected List<String> problems;
    protected List<PNodeEnd> executionOrder;

    protected Random random;

    protected boolean shuffle = false;
    protected boolean trace = true;

    protected long randomSeed = -4775043618027098404L;
    protected boolean randomized = false;

    protected PrintStream traceOut = Debug.out;

    /**
     * Creates a simulator for the specified agent's current plan.
     */
    public PlanCheckingSimulator(Ip2 ip2) {
	modelManager = (Ip2ModelManager)ip2.getModelManager();
	computeInterpreter = new CheckingInterpreter(ip2);
	sanityChecker = new SanityChecker(this);
	registerConstraintCheckers();
    }

    /**
     * Creates a simulator for the specified plan and domain.
     */
    public PlanCheckingSimulator(Plan plan, Domain domain) {
	this(makeModelHolder(plan, domain));
    }

    private static Ip2 makeModelHolder(Plan plan, Domain domain) {
	// /\/: Static so it can be called in this(...) in a constructor.
	Debug.expect(plan != null, "No plan was supplied");
	Debug.expect(domain != null, "No domain was supplied");
	Ip2 ip2 = IPlanOptionManager.ModelHolder.newInstance();
	ip2.loadDomain(domain);
	ip2.loadPlan(plan);
	return ip2;
    }

    /**
     * Returns the model-manager that contains the model that
     * this simulator is checking.
     */
    public Ip2ModelManager getModelManagerBeingChecked() {
	return modelManager;
    }

    /**
     * Controls whether the node-ends are permuted when putting them
     * in order for execution.  If the argument is true, a random number
     * generator is created using this simulator's current random seed,
     * and the simulator will use that generator when making lists of
     * node-ends while determining the execution order.  All of the
     * ordering constraints will still be obeyed, but this can change
     * the relative order of node-ends that aren't constrained against
     * each other.
     *
     * <p>If the argument is false, any current random number generator
     * is discarded, and the simulator will not permute the node-ends.</p>
     *
     * <p>There is no way to explicitly set the random seed (though
     * that might change in a subclass).  There is a fixed default seed
     * to allow repeatable simulations; or else a seed based on the
     * current time can be set by calling {@link #randomize}.</p>
     */
    public void setShuffle(boolean shuffle) {
	this.shuffle = shuffle;
	if (shuffle)
	    random = new Random(randomSeed);
	else
	    random = null;
    }

    /**
     * Changes this simulator's random number generator to one
     * set up with a seed based on the current time.  It is not
     * necessary to separately call {@link #setShuffle(boolean)},
     * because this method calls <code>setShuffle(true)</code>.
     */
    public void randomize() {
	randomSeed = System.currentTimeMillis();
	setShuffle(true);
	randomized = true;
    }

    /**
     * Tells this simulator whether or not it should produce trace
     * output.  Trace output shows which node-end is being executed
     * and when conditions and effects are eveluated.
     *
     * @see #setTraceOutput(PrintStream)
     */
    public void setTrace(boolean trace) {
	this.trace = trace;
    }

    /**
     * Sets the stream on which trace output appears.  If this method
     * is not called, the stream {@link Debug#out} will be used.
     *
     * @see #setTrace(boolean)
     */
    public void setTraceOutput(PrintStream ps) {
	this.traceOut = ps;
    }

    // N.B. The simulator works with the ModelManager's representation
    // of a plan, not with a Plan, and not always with MM structures
    // produced from a Plan.  Therefore, patterns, constraints, etc
    // can contain Variables.  For some purposes, we leave them in.
    // For example, trace output may be more useful if it leaves
    // the variables in.  For other purposes, in particular for
    // maintaining the world state, we have to remember to take
    // them out.

    // /\/: Removing Variables will replace any that still lack
    // values with ItemVars.  We don't yet check for that.

    /**
     * Simulates execution.
     */
    public void run() {
	// We start with the MM's current world-state and discard
	// all node-ends that are already COMPLETE.
	worldState = new HashMap(getInitialWorldStateMap());
	problems = new LinkedList<String>();
	resetConstraintCheckers();
	// Run the sanity-checker.
	sanityChecker.checkPlan();
	// if (shuffle) random = new Random(randomSeed);
	checkNodeStatusValues();
	checkNodeEndStatusValues();
	List<PNodeEnd> allNodeEnds = modelManager.getNodeEnds();
	new ExecutionStages(allNodeEnds).getStages();
	List<PNodeEnd> nodeEnds =
	    maybeShuffle(removeIf(Status.COMPLETE, allNodeEnds));
	List<PNodeEnd> sorted =
	    (List<PNodeEnd>)new ExecOrderSorter().sort(nodeEnds);
	Set<PNodeEnd> waiting = new HashSet<PNodeEnd>(nodeEnds);
	Set<PNodeEnd> ready = new HashSet<PNodeEnd>();
	Set<PNodeEnd> complete =
	    new HashSet<PNodeEnd>(keepIf(Status.COMPLETE, allNodeEnds));
	executionOrder = sorted;
	for (PNodeEnd ne: sorted) {
	    traceln("Executing", ne);
	    nextExecFringe(ready, waiting, complete);
	    Debug.expect(ready.contains(ne), "Not ready", ne);
	    checkConditions(ne, ready);
	    checkOtherConstraints(ne);
	    doEffects(ne);
	    ready.remove(ne);
	    complete.add(ne);
	}
    }

    public List<String> getProblems() {
	return problems;
    }

    public Map getInitialWorldStateMap() {
	return modelManager.getWorldStateMap();
    }

    public Map getWorldStateMap() {
	return worldState;
    }

    public Map getChangedWorldStateMap() {
	Map state = new TreeMap(new KeyValueTable.PatternObjectComparator());
	state.putAll(getWorldStateMap());
	Collect.removeAll(state, getInitialWorldStateMap());
	return state;
    }

    public List<PNodeEnd> getExecutionOrder() {
	return executionOrder;
    }

    public void report() {
	report(traceOut);
    }

    public void report(PrintStream out) {
	if (shuffle && randomized)
	    out.println("Shuffle with seed " + randomSeed);
	if (problems.isEmpty()) {
	    out.println("No problems found.");
	    return;
	}
	int n = problems.size();
	out.println(n == 1 ? "1 problem:" : n + " problems:");
	for (String prob: problems) {
	    out.println(prob);
	}
    }

    public void describeFinalWorldState() {
	describeFinalWorldState(traceOut);
    }

    public void describeFinalWorldState(PrintStream out) {
	Map state = new TreeMap(new KeyValueTable.PatternObjectComparator());
	state.putAll(getWorldStateMap());
	describeState(out, "Final world state", state);
    }

    public void describeChangedWorldState() {
	describeChangedWorldState(traceOut);
    }

    public void describeChangedWorldState(PrintStream out) {
	describeState(out, "Changed world state", getChangedWorldStateMap());
    }

    protected void describeState(PrintStream out, String label, Map state) {
	if (state.isEmpty())
	    out.println(label + " is empty");
	else {
	    out.println(label + ":");
	    for (Iterator i = state.entrySet().iterator(); i.hasNext();) {
		Map.Entry e = (Map.Entry)i.next();
		out.println("   " + e.getKey() + " = " + e.getValue());
	    }
	}
    }

    /**
     * Topological sort for node-ends to provide a total order
     * that can be used to simulate execution.
     */
    protected class ExecOrderSorter extends TopologicalSorter {
	protected Collection getChildren(Object nodeEnd) {
	    // Return node-ends linked directly after this one.
	    return maybeShuffle(((PNodeEnd)nodeEnd).getSuccessors());
	}
    }

    protected List<PNodeEnd> maybeShuffle(List<PNodeEnd> list) {
	if (shuffle) {
	    List result = new ArrayList<PNodeEnd>(list);
	    Collections.shuffle(result, random);
	    return result;
	}
	else
	    return list;
    }

    protected List<PNodeEnd> removeIf(final Status status,
				      List<PNodeEnd> nodeEnds) {
	return (List<PNodeEnd>)Collect.filter(nodeEnds, new Predicate1() {
	    public boolean trueOf(Object ne) {
		return ((PNodeEnd)ne).getStatus() != status;
	    }
	});
    }

    protected List<PNodeEnd> keepIf(final Status status,
				    List<PNodeEnd> nodeEnds) {
	return (List<PNodeEnd>)Collect.filter(nodeEnds, new Predicate1() {
	    public boolean trueOf(Object ne) {
		return ((PNodeEnd)ne).getStatus() == status;
	    }
	});
    }

    protected void checkNodeStatusValues() {
	for (PNode n: modelManager.getNodes()) {
	    Status status = n.getStatus();
	    Status correct = n.statusFromNodeEnds();
	    if (status != correct)
		wrongStatusProblem(n, correct);
	}
    }

    protected void checkNodeEndStatusValues() {
	// Although we don't use the node-ends' own status values --
	// except in removeIfComplete above -- we can check that they
	// all look correct.
	for (PNodeEnd end: modelManager.getNodeEnds()) {
	    checkNodeEndStatusValue(end);
	}
    }

    protected void checkNodeEndStatusValue(PNodeEnd end) {
	boolean isReady =
	    PNode.allHaveStatus(end.getPredecessors(), 
				Status.COMPLETE);
	Status status = end.getStatus();
	if (status == Status.COMPLETE) {
	    // All predecessors must be COMPLETE.
	    if (!isReady)
		wrongStatusProblem(end, Status.BLANK);
	}
	else if (status == Status.BLANK || status == Status.POSSIBLE) {
	    Status correct = isReady ? Status.POSSIBLE : Status.BLANK;
	    if (status != correct)
		wrongStatusProblem(end, correct);
	}
	else if (status == Status.EXECUTING) {
	    // Node-ends should never be EXECUTING?
	    wrongStatusProblem(end, Status.COMPLETE);
	}
	else {
	    problem(end + " has status " + status);
	}
    }

    protected void wrongStatusProblem(HasStatus h, Status correct) {
	problem(h + " has status " + h.getStatus() +
		" when it should be " + correct);
    }

    /**
     * Moves node-ends from waiting to ready.
     */
    protected void nextExecFringe(Set ready, Set waiting, Set complete) {
	// /\/: Once the simulation is under way, it should be only
	// successors of the just-executed node-end that might move
	// from waiting to ready, so we could make this more efficient
	// if we wanted to.
	for (Iterator wi = waiting.iterator(); wi.hasNext();) {
	    PNodeEnd ne = (PNodeEnd)wi.next();
	    if (complete.containsAll(ne.getPredecessors())) {
		// The waiting node-end is now ready
		wi.remove();
		ready.add(ne);
	    }
	}
    }

    public void traceln(String message) {
	if (trace) traceOut.println(message);
    }

    public void traceln(String message, Object obj) {
	if (trace) traceOut.println(message + " " + obj);
    }

    protected void problem(String description) {
	if (trace) traceln("Problem:", description);
	problems.add(description);
    }

    protected void checkConditions(PNodeEnd ne, Set ready) {
	for (Iterator i = getConditions(ne).iterator(); i.hasNext();) {
	    Constraint c = (Constraint)i.next();
	    if (c.getType() == Refinement.S_COMPUTE) {
		checkComputeCondition(c);
		continue;
	    }
	    if (c.getType() != Refinement.S_WORLD_STATE) {
		problem("Cannot evaluate " + c);
		continue;
	    }
	    PatternAssignment pv = c.getPatternAssignment();
	    traceln("  Condition:", pv);
	    LList p = (LList)Variable.removeVars(pv.getPattern());
	    Object v = Variable.removeVars(pv.getValue());
	    Object vInWorld = worldState.get(p);
	    if (vInWorld == null)
		problem("pattern " + p + " has no value " +
			" at " + ne + ", expected " + v);
	    else if (!vInWorld.equals(v))
		problem("pattern " + p + " has value " + vInWorld +
			" at " + ne + ", expected " + v);
	    checkForDeleters(ne, p, v, ready);
	}
    }

    protected void checkForDeleters(PNodeEnd ne, LList p, Object v,
				    Set ready) {
	// p and v already have Variables removed.
	for (Iterator ri = ready.iterator(); ri.hasNext();) {
	    PNodeEnd r = (PNodeEnd)ri.next();
	    if (r == ne) continue;
	    for (Iterator ei = getEffects(r).iterator(); ei.hasNext();) {
		PatternAssignment e = (PatternAssignment)ei.next();
		LList ep = (LList)Variable.removeVars(e.getPattern());
		Object ev = Variable.removeVars(e.getValue());
		if (ep.equals(p) && !ev.equals(v))
		    problem("Condition " + p + " = " + v + " at " + ne +
			    " is deleted by value " + ev + " at " + r);
	    }
	}
    }

    protected void checkComputeCondition(Constraint c) {
	PatternAssignment pv = c.getPatternAssignment();
	traceln(c.getRelation() == Refinement.S_MULTIPLE_ANSWER
		? "  Multiple-answer compute condition:"
		: "  Compute condition:",
		pv);
	LList p = (LList)Variable.removeVars(pv.getPattern());
	Object v = Variable.removeVars(pv.getValue());
	Object value = computeInterpreter.compute(p);
	if (c.getRelation() == Refinement.S_MULTIPLE_ANSWER) {
	    if (!(value instanceof Collection))
		problem("The multiple-answer value of " + p +
			" was " + value + ", not a collection.");
	    else if (!((Collection)value).contains(v))
		problem("The multiple-answer value of " + p +
			", " + value + ", did not contain " + v);
	}
	else {
	    checkComputeResult(p, v, value);
	}
    }

    protected void checkComputeResult(LList p, Object expected,
				      Object actual) {
	if (!matchComputeResult(expected, actual))
	    problem("The value of " + p + " was " + actual +
		    " instead of " + expected);
    }

    protected boolean matchComputeResult(Object expected, Object actual) {
	if (expected == TRUE)
	    return computeInterpreter.isTrue(actual) == true;
	else if (expected == FALSE)
	    return computeInterpreter.isTrue(actual) == false;
	else
	    return expected.equals(actual);
    }

    protected void doEffects(PNodeEnd ne) {
	for (Iterator i = getEffects(ne).iterator(); i.hasNext();) {
	    PatternAssignment pv = (PatternAssignment)i.next();
	    traceln("  Effect:", pv);
	    LList p = (LList)Variable.removeVars(pv.getPattern());
	    Object v = Variable.removeVars(pv.getValue());
	    worldState.put(p, v);
	}
    }

    // /\/: Conditions are all at begin, and effects all at end, of node.

    protected List getConditions(PNodeEnd ne) {
	return ne.getEnd() == End.BEGIN
	    ? Collect.ensureList(modelManager.getNodeConditions(ne.getNode()))
	    : Lisp.NIL;
    }

    protected List getEffects(PNodeEnd ne) {
	return ne.getEnd() == End.END
	    ? Collect.ensureList(modelManager.getNodeEffects(ne.getNode()))
	    : Lisp.NIL;
    }

    protected class CheckingInterpreter extends LispComputeInterpreter {
	public CheckingInterpreter(Ip2 ip2) {
	    super(null);	// don't pass along the modelManager.
	    // /\/: May need more than the newest support code?
	    loadSupportCode
		(ip2.getDomain()
		 .getAnnotation(Symbol.intern("compute-support-code")));
	}
	protected Object getWorldStateValue(LList pattern) {
	    return PlanCheckingSimulator.this.worldState.get(pattern);
	}
    }

    /*
     * Constraint Checkers
     */

    protected void checkOtherConstraints(PNodeEnd ne) {
	PNode node = ne.getNode();
	List<Constraint> constraints =
	    modelManager.getOtherNodeConstraints(node);
	if (constraints != null) {
	    for (Constraint c: constraints) {
		checkConstraint(c, ne);
	    }
	}
    }

    protected void checkConstraint(Constraint c, PNodeEnd at) {
	ConstraintChecker checker = 
	    checkerTable.get(c.getType(), c.getRelation());
	if (checker != null)
	    checker.checkConstraint(c, at);
    }

    protected void register(String type, String relation, 
			    ConstraintChecker c) {
	checkerTable.put(Symbol.intern(type), Symbol.intern(relation), c);
    }

    protected void registerConstraintCheckers() {
	register("resource", "use", new UseChecker());
    }

    protected void resetConstraintCheckers() {
	for (ConstraintChecker checker: checkerTable.values())
	    checker.reset();
    }

    protected abstract class ConstraintChecker {
	protected abstract void checkConstraint(Constraint c, PNodeEnd at);
	protected abstract void reset();
    }

    protected class UseChecker extends ConstraintChecker {

	protected Map<LList,PNodeEnd> useTable =
	    new HashMap<LList,PNodeEnd>();

	protected void checkConstraint(Constraint c, PNodeEnd at) {
	    c = (Constraint)Variable.removeAllVars(c);
	    LList pattern = c.getPattern();
	    if (at.getEnd() == End.BEGIN)
		checkUseBegin(pattern, at);
	    else
		checkUseEnd(pattern, at);
	}

	protected void checkUseBegin(LList pattern, PNodeEnd at) {
	    PNodeEnd user = useTable.get(pattern);
	    if (user != null) {
		problem(at + " can't use " + pattern +
			" because it's already used by " + user);
	    }
	    useTable.put(pattern, at);
	}

	protected void checkUseEnd(LList pattern, PNodeEnd at) {
	    PNodeEnd user = useTable.get(pattern);
	    if (user == null) {
		problem(at + " tries to end use of unused " + pattern);
	    }
	    else if (user != at.getOtherNodeEnd()) {
		problem(at + " tries to end use of " + pattern +
			" that began at " + user);
	    }
	    else {
		useTable.remove(pattern);
	    }
	}

	protected void reset() {
	    useTable.clear();
	}

    }

    /**
     * Standalone main program for testing.  The most useful
     * command-line arguments are:
     * <pre>
     *   -plan=<i>resource name</i>
     *   -trace=true <i>or</i> false<i>, default</i> true
     *   -shuffle=true <i>or</i> false<i>, default</i> false
     *   -randomize=true <i>or</i> false<i>, default</i> false
     * </pre>
     *
     * @see #setTrace(boolean)
     * @see #setShuffle(boolean)
     * @see #randomize()
     */
    public static void main(String[] argv) {

	Debug.off();

	Ip2 ip2 = new ix.test.PlainIp2();
	ip2.mainStartup(argv);	// should load a plan

	Debug.on = true;

	PlanCheckingSimulator sim = new PlanCheckingSimulator(ip2);
	sim.processCommandLineArguments();

	sim.traceln("");
	sim.traceln("- - - - - Beginning simulation - - - - -");
	sim.traceln("");

	sim.run();

	sim.traceln("");
	sim.report();

	sim.traceln("");
	if (sim.trace) { sim.describeFinalWorldState(); }
	if (sim.trace) { sim.traceln("");
	                 sim.describeChangedWorldState(); }
    }

    public void processCommandLineArguments() {
	// N.B. be careful if more parameters are processed,
	// because things that creat a PlanCheckingSimulator
	// may call this method.
	setTrace(Parameters.getBoolean("trace", true));
	setShuffle(Parameters.getBoolean("shuffle", false));
	if (Parameters.getBoolean("randomize", false))
	    randomize();
    }

}

// Issues:
//  * Need to complete conversion to Java generics, which chiefly
//    means the Maps and the ExecOrderSorter.
