/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Mon Oct 24 02:49:25 2005 by Jeff Dalton
 * Copyright: (c) 2004, 2005, AIAI, University of Edinburgh
 */

package ix.iplan;

import javax.swing.*;

import java.awt.event.*;

import java.io.*;
import java.util.*;

import ix.icore.IXAgent;
import ix.icore.plan.Plan;
import ix.ip2.*;
import ix.iplan.event.*;
import ix.iface.util.IFUtil;
import ix.iface.util.CheckTableDialog;
import ix.util.*;
import ix.util.xml.*;

public class OptionUI implements ActionListener, OptionListener {

    IPlanOptionManager optMan;
    // Ip2Frame frame;
    JFrame frame;

    JMenu optionMenu;
    JMenu selectOptionMenu;
    JMenu clearOptionMenu;
    JMenu deleteOptionMenu;
    JMenuItem copyOptionItem;
    JMenuItem splitOptionItem;
    JMenuItem renameOptionItem;
    JMenuItem deleteOptionItem;

    JFileChooser directoryChooser;

    public OptionUI(IPlanOptionManager m, JFrame frame) {
	this.optMan = m;
	this.frame = frame;
	constructOptionMenu();
	optMan.addOptionListener(this);
    }

    public JMenu getOptionMenu() {
	return optionMenu;
    }

    void constructOptionMenu() {
	// Don't need to populateSelectOptionMenu();
	// Indeed, it won't yet work, because the map is null.
	optionMenu = new JMenu("Options");
	optionMenu.add(selectOptionMenu = new JMenu("Select"));
	optionMenu.add(makeItem("New Option"));
	optionMenu.add(splitOptionItem = makeItem("Split X"));
	optionMenu.add(copyOptionItem = makeItem("Copy X"));
	optionMenu.add(renameOptionItem = makeItem("Rename X"));
	optionMenu.add(clearOptionMenu = new JMenu("Clear X"));
	clearOptionMenu.add(makeItem("Clear All"));
	clearOptionMenu.add(makeItem("Clear All But State"));
	optionMenu.add(deleteOptionMenu = new JMenu("Delete"));
	deleteOptionMenu.add(deleteOptionItem = makeItem("Delete X"));
	deleteOptionMenu.add(makeItem("Delete Selected Options"));
	optionMenu.addSeparator();
	optionMenu.add(makeItem("Sync State"));
	optionMenu.add(makeItem("Preferences"));
	optionMenu.addSeparator();
	optionMenu.add(makeItem("Load Options From ..."));
	optionMenu.add(makeItem("Save Options In ..."));
	// Fix the action commands in menu-items that include the
	// option name in their text.
	splitOptionItem.setActionCommand("Split Option");
	copyOptionItem.setActionCommand("Copy Option");
	renameOptionItem.setActionCommand("Rename Option");
	deleteOptionItem.setActionCommand("Delete Option");
    }

    private JMenuItem makeItem(String text) {
	return IFUtil.makeMenuItem(text, this);
    }

    public void actionPerformed(ActionEvent e) {
        String command = e.getActionCommand();
        Debug.noteln("Option action:", command);
	if (command.equals("Select Option")) {
	    JMenuItem item = (JMenuItem)e.getSource();
	    String optionName = item.getText();
	    Debug.noteln("Select option", optionName);
	    optMan.setOption(optionName);
	}
	else if (command.equals("New Option")) {
	    String name = JOptionPane.showInputDialog
		(frame, "Enter a name for the new option");
	    if (name != null)
		optMan.newOption(name);
	}
	else if (command.equals("Split Option")) {
	    // checkOptionName(e);
	    String minusDirection = 
		Strings.beforeLast(" (", splitOptionItem.getText());
	    checkOptionName(minusDirection);
	    optMan.splitOption();
	}
	else if (command.equals("Copy Option")) {
	    checkOptionName(e);
	    String newName = JOptionPane.showInputDialog
		(frame, "Enter a name for the copy");
	    if (newName != null)
		optMan.copyOption(newName);
	}
	else if (command.equals("Rename Option")) {
	    checkOptionName(e);
	    String newName = JOptionPane.showInputDialog
		(frame, "Enter a new name for the option");
	    if (newName != null)
		optMan.renameOption(newName);
	}
	else if (command.equals("Clear All")) {
	    checkOptionName(clearOptionMenu.getText());
	    if (Util.dialogConfirms
		   (frame, "Are you sure you want to clear the option?"))
		optMan.clearOption();
	}
	else if (command.equals("Clear All But State")) {
	    checkOptionName(clearOptionMenu.getText());
	    if (Util.dialogConfirms
		   (frame, "Are you sure you want to clear " +
		           "all but the option's state?"))
		optMan.clearOptionAllButState();
	}
	else if (command.equals("Delete Option")) {
	    checkOptionName(e);
	    if (Util.dialogConfirms
		   (frame, "Are you sure you want to delete the option?"))
		optMan.deleteOption();
	}
	else if (command.equals("Delete Selected Options")) {
	    deleteSelectedOptions();
	}
	else if (command.equals("Sync State")) {
	    if (optMan.getNameToOptionMap().size() < 2)
		Util.displayAndWait(frame, new String[] {
		    "There is only one option, and so none to",
		    "take state from the current option"});
	    else
		syncSelectedOptions();
	}
	else if (command.equals("Preferences")) {
	    editPreferences();
	}
	else if (command.equals("Load Options From ...")) {
	    loadOptions();
	}
	else if (command.equals("Save Options In ...")) {
	    saveOptions();
	}
	else
	    throw new ConsistencyException
		("Unknown option command", command);
    }

    private void checkOptionName(ActionEvent e) {
	JMenuItem item = (JMenuItem)e.getSource();
	checkOptionName(item.getText());
    }

    private void checkOptionName(String menuText) {
	String optionName = Strings.afterFirst(" ", menuText);
	Debug.expectEquals
	    (optionName, optMan.getOption().getName(),
	     "The current option does not have the expected name.");
    }

    /*
     * OptionListener methods
     */

    public void optionAdded(OptionEvent e) {
	populateSelectOptionMenu();
    }

    public void optionSet(OptionEvent e) {
	noticeCurrentOptionName(e);
    }

    public void optionRenamed(OptionEvent e, String oldName) {
	if (e.getOption() == optMan.getOption())
	    noticeCurrentOptionName(e);
	populateSelectOptionMenu();
    }

    public void optionContentsChanged(OptionEvent e, EventObject how) {
	Debug.expectSame(optMan.getOption(), e.getOption());
	menuShowSplitDirection(e);
    }

    public void optionDeleted(OptionEvent e) {
	populateSelectOptionMenu();
    }

    /* End of OptionListener methods */

    void populateSelectOptionMenu() {
	JMenu menu = selectOptionMenu;
	menu.setEnabled(false);
	menu.removeAll();
	for (Iterator i = optMan.getNameToOptionMap().keySet().iterator()
		 ; i.hasNext();) {
	    String name = (String)i.next();
	    JMenuItem item = IFUtil.makeMenuItem(name, this);
	    item.setActionCommand("Select Option");
	    menu.add(item);
	}
	menu.setEnabled(true);
    }

    void noticeCurrentOptionName(OptionEvent e) {
	Debug.expectSame(optMan.getOption(), e.getOption());
	String name = e.getOptionName();
	menuShowSplitDirection(e);
	copyOptionItem.setText("Copy " + name);
	renameOptionItem.setText("Rename " + name);
	clearOptionMenu.setText("Clear " + name);
	deleteOptionItem.setText("Delete " + name);
	frameShowOptionName(name);
    }

    void menuShowSplitDirection(OptionEvent e) {
	String direction = e.getOption().hasChanged()
	    ? " (down)" : " (across)";
	splitOptionItem.setText("Split " + e.getOptionName() + direction);
    }

    void frameShowOptionName(String name) {
	// /\/: This should really be in the frame, and it
	// should be or have an option listener.  Also, the
	// NewIp2Frame optionLabel field should stop being public.
	if (frame instanceof NewIp2Frame) {
	    NewIp2Frame f = (NewIp2Frame)frame;
	    f.optionLabel.setText(name);
	}
    }

    void deleteSelectedOptions() {
	String safe = optMan.getOptionForInput().getName();
	Set optionNames = optMan.getNameToOptionMap().keySet();
	List choices = new LinkedList(optionNames);
	choices.remove(safe);
	if (choices.isEmpty()) {
	    String[] message = {
		"There are no deletable options.  Remember that the",
		"option selected for input, " + safe + ",",
		"cannot be deleted."};
	    Util.displayAndWait(frame, message);
	    return;
	}
	Object[] selected =
	    CheckTableDialog.showDialog(frame, "Options to delete",
					choices.toArray());
	optMan.deleteOptions(Arrays.asList(selected));
    }

    List savedSyncSelection = Collections.EMPTY_LIST;

    void syncSelectedOptions() {
	String safe = optMan.getOption().getName();
	Set optionNames = optMan.getNameToOptionMap().keySet();
	List choices = new LinkedList(optionNames);
	choices.remove(safe);
	savedSyncSelection = 
	    (List)Collect.intersection(choices, savedSyncSelection);
	Debug.expect(!choices.isEmpty(), "no syncable options");
	Object[] selected =
	    CheckTableDialog.showDialog
	        (frame,
		 "Options to sync state from " + safe,
		 choices.toArray(),
		 savedSyncSelection.toArray());
	savedSyncSelection = Arrays.asList(selected);
	optMan.syncState(savedSyncSelection);
    }


    /*
     * Preferences
     */

    void editPreferences() {
	// For now, handles only the option for input.  /\/
	editOptionForInput();
    }

    void editOptionForInput() {
	String current = optMan.getOptionForInput().getName();
	String selected =
	    askUserToSelectOption
	        ("Select the option to receive input", current);
	if (selected != null)
	    optMan.setOptionForInput(selected);
    }

    public String askUserToSelectOption(String message,
					String initialSelection) {
	Object[] choices = optMan.getNameToOptionMap().keySet().toArray();
	Object selected =
	    JOptionPane.showInputDialog
  	        (frame,
		 message,
		 // /\/: We used to always have an Ip2Frame
		 // frame.getAgent().getAgentDisplayName() + " Select Option",
		 IXAgent.getAgent().getAgentDisplayName() + " Select Option",
		 JOptionPane.QUESTION_MESSAGE,
		 null,		// icon
		 choices,
		 initialSelection);
	return selected == null ? null : (String)selected;
    }

    /*
     * Load Options From ...
     */

    void loadOptions() {
	if (directoryChooser == null)
	    directoryChooser = makeDirectoryChooser();
	while (true) {
	    int option = directoryChooser.showOpenDialog(frame);
	    if (option != JFileChooser.APPROVE_OPTION)
		return; // user quit
	    try {
		if (loadOptions(directoryChooser.getSelectedFile()))
		    return;
	    }
	    catch (Throwable t) {
		Debug.displayException(t);
	    }
	}
    }

    private boolean loadOptions(File directory) {
	Debug.noteln("Loading options from", directory);
	String directoryName = directory.toString();
	SortedMap plans = optMan.readPlans(directoryName);
	List names = new ArrayList(plans.keySet());
	if (Util.dialogConfirms(frame, new String[] {
	        Strings.foldLongLine
		    ("Found plans " + Strings.conjunction(names) + "."),
		"",
		"Treat them as options?"})) {
	    optMan.makeTopLevelOptions(directoryName, plans);
	    return true;
	}
	return false;
    }

    JFileChooser makeDirectoryChooser() {
	String dir = Parameters.getParameter("option-directory",
					     System.getProperty("user.dir"));
	JFileChooser chooser = new JFileChooser(dir);
	chooser.setFileSelectionMode(JFileChooser.DIRECTORIES_ONLY);
//  	JFileChooser chooser =
//  	    new XMLLoader(null, Plan.class).makeFileChooser();
//    	chooser.setFileSelectionMode(JFileChooser.FILES_AND_DIRECTORIES);
	return chooser;
    }

    /*
     * Save Options In ...
     */

    void saveOptions() {
	if (directoryChooser == null)
	    directoryChooser = makeDirectoryChooser();
	int option = directoryChooser.showSaveDialog(frame);
	if (option == JFileChooser.APPROVE_OPTION) {
	    File directory = directoryChooser.getSelectedFile();
	    saveOptions(directory);
	}
    }

    void saveOptions(File directory) {
	Debug.noteln("Saving options in", directory);
	// Find any ".xml" files already in the directory.
	File[] files = directory.listFiles(new FileFilter() {
	    public boolean accept(File f) {
		return f.getName().endsWith(".xml");
	    }
	});
	if (files == null)
	    throw new IllegalArgumentException
		(directory + " does not name a directory");
	// See if any of the existing files will be overwritten
	// and whether the user minds.
	Set optionNames = optMan.getNameToOptionMap().keySet();
	Set conflicts = (Set)Collect.intersection
	                        (fileNameSet(files), optionNames);
	Debug.noteln("Will overwrite", conflicts);
	if (!conflicts.isEmpty()
	    && !Util.dialogConfirms
		 (frame, 
		  Strings.foldLongLine
		   ("Saving options to " + directory +
		    " will overwite " +
		    Strings.conjunction(new ArrayList(conflicts)) + ". " +
		    " Continue?")))
	    return;
	// Get all the options as Plans.
	SortedMap plans = optMan.getOptionsAsPlans();
	// Save the plans.
	for (Iterator i = plans.entrySet().iterator(); i.hasNext();) {
	    Map.Entry e = (Map.Entry)i.next();
	    String name = (String)e.getKey();
	    Plan plan = (Plan)e.getValue();
	    File planFile = new File(directory, name + ".xml");
	    if (planFile.exists())
		Util.renameToBackup(planFile);
	    Debug.noteln("Writing", planFile);
	    XML.writeObject(plan, planFile.getPath());
	}
    }

    Set fileNameSet(File[] files) {
	Set result = new TreeSet();
	for (int i = 0; i < files.length; i++) {
	    result.add(Strings.beforeLast(".", files[i].getName()));
	}
	return result;
    }

}
