/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Thu Aug 31 23:40:05 2006 by Jeff Dalton
 * Copyright: (c) 2004, 2005, AIAI, University of Edinburgh
 */

package ix.iplan;

import javax.swing.*;
import javax.swing.border.TitledBorder;

import java.awt.BorderLayout;

import java.io.*;
import java.util.*;

import ix.ip2.*;
import ix.iplan.event.*;
import ix.icore.plan.Plan;
import ix.util.*;
// import ix.util.context.Context; // /\/ should do the context stuff in IPlan

/**
 * I-Plan's automatic planner as a tool.
 */
public class IPlanTool {

    Ip2 ip2;
    // Slip slip;                  // s.b. generic /\/
    PlanStats stats;
    // Context homeContext;
    // Context initialPlanContext;

    ToolFrame frame;

    public IPlanTool(Ip2 ip2) {
	this.ip2 = ip2;
	this.frame = new ToolFrame
	    (ip2.getAgentDisplayName() + " I-Plan Tool");
	ip2.addResetHook(new ResetHook());
	ip2.getOptionManager().addOptionListener(new OptListener());
	frame.noteCurrentOption(ip2.getOptionManager().getOption());
    }

    public void setVisible(boolean v) {
	frame.setVisible(v);
    }

    public ToolFrame getFrame() {
	return frame;
    }

    protected class ResetHook implements Runnable {
	// Contexts have already been cleared by the time this is is called.
	public void run() {
	    Debug.noteln("Resetting", IPlanTool.this);
	    frame.reset();
// 	    initialPlanContext = null;
// 	    slip = null;
	}
    }

    class OptListener extends AbstractOptionListener {
	public void optionSet(OptionEvent e) {
	    frame.noteCurrentOption(e.getOption());
	}
	public void optionRenamed(OptionEvent e, String oldName) {
	    if (e.getOption() == ip2.getOptionManager().getOption())
		frame.noteCurrentOption(e.getOption());
	}
	public void optionContentsChanged(OptionEvent e, EventObject how) {
	    if (e.getOption() == ip2.getOptionManager().getOption())
		frame.noteCurrentOption(e.getOption());
	}
    }

    /* ------------------------------------------------------------

    void getPlanFromPlanner() {
	// We need to move plans between our MM which has listenters
	// and the Slip MM which doesn't.  Hence various context
	// manipulations.
	// /\/: We also have to do this:
	Parameters.setParameter("plan-state-to-save", "*"); //\/
	if (slip != null) {
	    discardPlanner();
	}
	Plan plan = ip2.getModelManager().getPlan();
	homeContext = Context.getContext();
	createPlanner(plan);
	try {
	    slip.plan();
	}
	catch (Slip.NoPlanException npe) {
	    handleNoPlan();
	    Util.displayAndWait(frame, "No plan was found");
	    return;
	}
	catch (Throwable t) {
	    handleNoPlan();
	    Debug.displayException(t);
	    return;
	}
	// Get the plan in new context and display it.
	handleNewPlan();
	// Allow replanning.
	frame.replanButton.setEnabled(true);
    }

    void getReplanFromPlanner() {
	homeContext = Context.getContext();
	try {
	    slip.replan();
	}
	catch (Slip.NoPlanException npe) {
	    handleNoPlan();
	    Util.displayAndWait(frame, "No plan was found");
	    return;
	}
	catch (Throwable t) {
	    handleNoPlan();
	    Debug.displayException(t);
	    return;
	}
	handleNewPlan();
    }

    void createPlanner(Plan initialPlan) {
	Context.setContext(Context.rootContext);
	initialPlanContext = Context.pushContext();
	slip = new Slip(false);
	slip.mainStartup(new String[]{});
	slip.setDomain(ip2.getDomain());
	slip.loadPlan(initialPlan);
    }

    void discardPlanner() {
	initialPlanContext.discard();
	initialPlanContext = null;
	// slip.reset();
	slip = null;
    }

    // /\/: Handling the stats object is tricky, becuase calling
    // slip.replan() makes Slip create a new one, and we set
    // slip = null when there's no plan, so we can't always
    // ask Slip for the stats object.  So we get the stats
    // object in the "handle" methods.

    void handleNewPlan() {
	// Get the plan in new context and display it.
	Plan newPlan = slip.getModelManager().getPlan();
	stats = slip.getStatistics();
	Context.setContext(Context.rootContext);
	Context.pushContext();
	ip2.resetViewers();
	ip2.loadPlan(newPlan);
    }

    void handleNoPlan() {
	stats = slip.getStatistics();
	// Go back to the context we were in before planning.
	Context.setContext(homeContext);
	// Eliminate the planner and the contexts it used.
	discardPlanner();
	// Disable replanning.
	frame.replanButton.setEnabled(false);
    }

    ------------------------------------------------------------ */

    void checkPlan() {
	checkPlan(new PlanCheckingSimulator(ip2));
    }

    void checkPlan(PlanCheckingSimulator sim) {
	ensureSimFrame();
	simFrame.setText("");
	simFrame.setVisible(true);

	TraceStream trace = new TraceStream(simFrame);

	// Simulate plan execution.
	// PlanCheckingSimulator sim = new PlanCheckingSimulator(ip2);
	sim.setTraceOutput(trace);
	sim.run();

	// Print a report that describes any problems found.
	trace.println("");
	sim.report();

	// Describe the world state changes
	trace.println("");
	sim.describeChangedWorldState();
    }

    private SimFrame simFrame = null;

    private void ensureSimFrame() {
	if (simFrame == null) {
	    String title = ip2.getAgentDisplayName() +" Plan Check";
	    simFrame = new SimFrame(title);
	}
    }

    SimFrame getSimFrame() {
	return simFrame;
    }

    static class SimFrame extends TextAreaFrame {
	SimFrame(String title) {
	    super(0, 0, title, new String[]{"Cancel"});
	    setEditable(false);
	}
	protected void finishFrame() {
	    // Doesn't make visible, unlike super method.
	    frame.pack();
	    frame.setSize(400, 300);
	    frame.validate();
	}
    }

    private class TraceStream extends PrintStream {
	TextAreaFrame textFrame;
	TraceStream(TextAreaFrame f) {
	    super(new RefusingOutputStream());
	    this.textFrame = f;
	}
	public void println(String text) {
	    textFrame.appendLine(text);
	}
    }

    private class RefusingOutputStream extends OutputStream {
	public void write(int b) throws IOException {
	    throw new UnsupportedOperationException
		("Attempt to write to a RefusingOutputStream");
	}
    }

    class ToolFrame extends TextAreaFrame {

	JButton planButton;
	JButton replanButton;
	TitledBorder border;

	ToolFrame(String title) {
	    super(0, 0, title, new String[]{});
	    setEditable(false);
	    addListener(new ToolTListener());
	    border = BorderFactory.createTitledBorder("Plan");
	    setBorder(border);
	}

	protected void finishFrame() {
	    // Doesn't make visible, unlike super method.
	    frame.pack();
	    frame.setSize(400, 300);
	    frame.validate();
	}

	void setBorderTitle(String title) {
	    // This doesn't show the change: border.setTitle(title);
	    border = BorderFactory.createTitledBorder(title);
	    setBorder(border);
	    // not needed: frame.validate();
	}

	void reset() {
	    setText("");
	    planButton.setEnabled(true);
	    replanButton.setEnabled(false);
	    border.setTitle("Plan");
	}

	void noteCurrentOption(IPlanOptionManager.Opt opt) {
	    // Called when the agent has switched to a different option
	    // or when certain properties of the current option change.
	    Debug.noteln("IPlanTool notices", opt);
	    setBorderTitle(opt.getName());
	    setText("");
	    replanButton.setEnabled(opt.allowsReplan());
	}

//  	void planStats() {
//  	    stats.report(new TraceStream(this));
//  	}

	void planStats(PlanStats stats) {
	    stats.report(new TraceStream(this));
	}

	protected TFrame makeTFrame(String title, String[] buttons) {
	    Debug.expect(buttons.length == 0);
	    return new PlanFrame(title, buttons);
	}

	class PlanFrame extends TFrame {
	    PlanFrame(String title, String[] buttons) {
		super(title, buttons);
		contentPane.add(makeButtonPanel(), BorderLayout.SOUTH);
	    }
	    protected JPanel makeButtonPanel() {
		JPanel panel = new JPanel();
		panel.add(planButton = makeButton("Plan"));
		panel.add(replanButton = makeButton("Replan"));
		panel.add(makeButton("Check Plan"));
		replanButton.setEnabled(false);
		return panel;
	    }
	}

//  	class ToolTListener implements TListener {
//  	    ToolTListener() {}
//  	    public void buttonPressed(String command) {
//  		if (command.equals("Plan")) {
//  		    setText("");
//  		    getPlanFromPlanner();
//  		    planStats();
//  		}
//  		else if (command.equals("Replan")) {
//  		    setText("");
//  		    getReplanFromPlanner();
//  		    planStats();
//  		}
//  		else if (command.equals("Check Plan")) {
//  		    checkPlan();
//  		}
//  		// Else maybe another listener will handle it.
//  	    }

//  	}

	class ToolTListener implements TListener {
	    ToolTListener() {}
	    public void buttonPressed(String command) {
		IPlanOptionManager optMan = ip2.getOptionManager();
		if (command.equals("Plan")) {
		    // getPlanFromPlanner();
		    // planStats();
		    if (!replanButton.isEnabled() || planIsIntended()) {
			setText("");
			optMan.plan();
			planStats(optMan.getStats());
		    }
		}
		else if (command.equals("Replan")) {
		    setText("");
		    // getReplanFromPlanner();
		    // planStats();
		    optMan.replan();
		    planStats(optMan.getStats());
		}
		else if (command.equals("Check Plan")) {
		    checkPlan();
		}
		// Else maybe another listener will handle it.
	    }
	    private boolean planIsIntended() {
		return Util.dialogConfirms(frame,
		    "Are you sure you want to plan instead of replan?");
	    }
	}

    }

}
