/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Fri Aug  8 19:42:25 2008 by Jeff Dalton
 * Copyright: (c) 2004, 2007, 2008, AIAI, University of Edinburgh
 */

package ix.iplan;

import java.io.*;		// for reading initial options
import java.util.*;

import ix.iface.util.ToolManager;     // /\/
import ix.iface.util.ToolController;  // /\/
import ix.icore.IXAgent;	      // /\/

import ix.icore.domain.Domain;
import ix.icore.process.ProcessModelManager;
import ix.icore.plan.Plan;	// /\/
import ix.icore.Activity;	// /\/
import ix.ip2.*;
import ix.util.*;
import ix.util.lisp.*;		// /\/
import ix.util.xml.*;
import ix.util.context.*;

/**
 * An interactive planner.  I-Plan is a version of Ip2 that
 * has different handlers, and other changes, to make it more
 * suitable for plan-construction and less suitable for execution.
 */

public class IPlan extends Ip2 {

    public IPlan() {
	this(true);
    }

    protected IPlan(boolean setAgent) {
	super(setAgent);
	agentTypeName = "I-X Planner";
	logoLine1 = "I-X Planner";
	displayName = agentTypeName;
	showOptions = true;	// default value
	Parameters.setParameter("plan-state-to-save", "*"); //\/
	Parameters.getParameter("plan-state-to-save"); // mark as used /\/
    }

    public static void main(String[] argv) {
	Util.printGreeting("I-Plan");
	new IPlan().mainStartup(argv);
    }

    public void setDomain(Domain dom) {
	dom.checkConsistency();
	this.domain = dom;
    }

    /**
     * Returns a Planner suitable for use within Java.  This method
     * should normally be used in non-I-X code, rather than instantiating
     * a planner class directly.  This method should be called when the
     * current {@link ContextHolder} is the right one for the planner to use.
     * The planner will ensure that the current context in that holder
     * is saved and restored around any calls to methods in the
     * {@link Planner} interface.  This method also ensures that the
     * current context is a new child of the root context when the
     * planner is created.
     *
     * <p>An implicit assumption is that only one thread will be using
     * the planner at any one time.  In a multi-thread environment, it's
     * also necessary to manage contexts carefully, and typically to
     * use a {@link ThreadLocalHoldingStrategy}.</p>
     *
     * @see Context
     */
    public static Planner makeDefaultPlanner() {
	
	ContextHolder holder = Context.getContextHolder();
	Context savedContext = holder.getContext();
	Slip slip;
	try {
	    Context c = new Context();
	    holder.setContext(c);
	    Debug.noteln("Making default planner in " + c +
			 " with ancestors " + c.getAncestors());
	    slip = new Slip(false);
	    slip.mainStartup(new String[]{});
	}
	finally {
	    holder.setContext(savedContext);
	}
	Planner p = ContextLockingProxy.newProxyInstance(Planner.class, slip);
	Predicate1 filter = new ix.test.DigestSet().new IsNewPredciate();
	return new FilteredPlanner(p, filter);
    }

    protected ProcessModelManager makeModelManager() {
	return new IPlanModelManager(this);
    }

    protected Ip2Frame makeIp2Frame() {
	return new IPlanFrame(this);
    }

    protected void initOptions() {
	optionManager.initOptions();
    }

    static java.awt.Component displayFrame() {
	// /\/: We shouldn't have any GUI code in this class.
	IXAgent agent = IXAgent.getAgent();
	if (!(agent instanceof Ip2))
	    return null;
	Ip2 ip2 = (Ip2)agent;
	Ip2Frame frame = ip2.getFrame();
	if (frame == null)
	    return null;
	ToolManager toolMan = frame.getToolManager();
	if (toolMan == null)	// can this happen?? /\/
	    return frame;
	ToolController tc = toolMan.findTool("I-Plan");
	if (tc == null)
	    return frame;
	IPlanTool ipt = (IPlanTool)tc.getTool();
	if (ipt == null)
	    return frame;
	return ipt.getFrame();	// may be null?  /\/
    }


    /**
     * Install any built-in issue and activity handlers.
     */ 
    protected void addHandlers() {

	// Handler that lets the user do nothing or make items complete.
	controller
	    .addItemHandler(new IPlanCompletionHandler(this));

	// Expansion
	controller
	    .addActivityHandler(new IPlanExpandHandler(this));

	// Condition satisfaction
	// controller
	//     .addActivityHandler(new IPlanConditionHandler(this));

    }

}

// Issues:
// * Instead of e.g. IPlanExpandHandler, names based on the algorithm,
//   because IPlan ought to be able to work with > 1 algorithm.
