/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Fri May 23 15:20:40 2008 by Jeff Dalton
 * Copyright: (c) 2007, 2008, AIAI, University of Edinburgh
 */

package ix.iplan;

import java.util.*;

import ix.util.*;
import ix.util.xml.XML;

/**
 * Specifies a group of PlanTests.  The tests can either be taken from
 * a {@link PlanTestGroup} that was defined earlier or can be given directly
 * in this instance.  If an ExpandPlanTestGroup appears in a list of
 * PlanTests given to an {@link AutoTester}, it will get the AutoTester
 * to run those tests.  That happens via the TestRunner returned by the 
 * {@link #makeTestRunner(AutoTester)} method.
 *
 * <p>If an ExpandPlanTestGroup specifies both a named PlanTestGroup
 * to "expand" and a list of tests of its own, the ones from the
 * named group are run first.
 *
 * <p>Since an ExpandPlanTestGroup is a PlanTest, it can be used
 * to provide default values to other PlanTests, typically to the
 * PlanTests in the expansion.  Indeed, that happens when the expansion
 * is processed by an AutoTester.
 */
public class ExpandPlanTestGroup extends PlanTest {

    /** This group's name. */
    protected String name;

    /** 
     * Tests directly in this expansion rather than taken from a named group.
     */
    protected List tests;

    public ExpandPlanTestGroup() {
        super();
    }

    public String getName() {
	return name;
    }

    public void setName(String name) {
	this.name = name;
    }

    public List getTests() {
	return tests;
    }

    public void setTests(List tests) {
	this.tests = tests;
    }

    /**
     * Returns a TestRunner that will run the tests in the group.
     */
    public AutoTester.TestRunner makeTestRunner(final AutoTester auto) {
	return auto.new TestRunner(ExpandPlanTestGroup.this) {
	    public void runTest() {
		runGroupTests(auto, auto.getPlanTestGroup(name));
	    }
	};
    }

    protected void runGroupTests(AutoTester auto, PlanTestGroup group) {
	// If a group was specified, get a copy of the group's tests
	// so that we don't modify the original tests.
	List allTests = new LinkedList();
	if (group != null) {
	    // /\/: We use serializing to preserve test.isExhaustiveWasSet
	    // values.
	    allTests.addAll((List)Util.cloneBySerializing(group.getTests()));
	    // Use this expander, then the group, to supply defaults
	    // to each test.
	    for (Iterator i = allTests.iterator(); i.hasNext();) {
		PlanTest test = (PlanTest)i.next();
		test.takeDefaults(this);
		test.takeDefaults(group);
	    }
	}
	// Add any tests of our own
	if (getTests() != null) {
	    // After supplying defaults from this expansion
	    for (Iterator i = getTests().iterator(); i.hasNext();) {
		PlanTest test = (PlanTest)i.next();
		test.takeDefaults(this);
	    }
	    allTests.addAll(getTests());
	}
	// Run the tests.
	auto.testLoop(allTests);
    }

    public String testDescription() {
	return "Expand group named " + name; // /\/
    }

}
