/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Tue Apr 25 20:39:09 2006 by Jeff Dalton
 * Copyright: (c) 2002, 2006, AIAI, University of Edinburgh
 */

package ix.ip2.test;

import java.awt.event.*;
import javax.swing.*;

import java.util.*;

import ix.icore.Sendable;
import ix.ip2.PanelFrame;
import ix.util.*;

/**
 * A specification of a test that can be selected from a panel's
 * "Test" menu.
 */
public abstract class PanelTest implements TestElement {

    protected String menuText;
    protected PanelFrame frame;

    public PanelTest() { }

    public String getMenuText() { return menuText; }

    public void setMenuText(String menuText) {
	this.menuText = menuText;
    }

    public String getFullMenuText() {
	// Subclasses might fill placeholders in the text.
	return getMenuText();
    }

    // Makes every PanelTest look a bit like a TestSequence.
    public abstract List getTestItems();

    /**
     * Adds a test menu item based on a PanelTest such as a TestItem
     * and sets this PanelTests's frame.
     */
    public void addToMenu(PanelFrame frame) {
	this.frame = frame;

	// We get the sendable items now so that any exceptions will
	// be thrown now rather than only if and when the user selects
	// the test.
	final List items = getTestItems();
	String fullMenuText = getFullMenuText();

	if (items.size() == 1) {
	    TestItem item = (TestItem)items.get(0);
	    if (item.getDelayBefore() <= 0) {
		frame.addTest(fullMenuText,
			      new TestItemListener(item));
		return;
	    }
	}
	frame.addTest(fullMenuText,
		      new TestSequenceListener(fullMenuText, items));
    }

    /**
     * ActionListener for AbstractButtons that sends a message.
     */
    class TestItemListener implements ActionListener {
	TestItem item;
	TestItemListener(TestItem item) {
	    this.item = item;
	}
	public void actionPerformed(ActionEvent e) {
	    sendTest(item.getToName(), item.getContents());
	}
    }

    void sendTest(String toName, Sendable contents) {
	// N.B. must make a copy, because it will change
	// its status, etc if sent to self ("me").
	frame.ensureSendPanelExists().sendCopy(toName, contents);
    }

    /**
     * ActionListener for AbstractButtons that send a sequence of messages
     * or a single message after a delay.
     */
    class TestSequenceListener implements ActionListener {
	String menuText;
	List testItems;
	SendingThread sender;
	AbstractButton button;

	public TestSequenceListener(String menuText, List testItems) {
	    this.menuText = menuText;
	    this.testItems = testItems;
	}

	public void actionPerformed(ActionEvent event) {
	    button = (AbstractButton)event.getSource();
	    String command = event.getActionCommand();
	    if (command.equals(menuText)) {
		sender = new SendingThread(this, testItems);
		button.setText("Stop: " + menuText);
		sender.start();
	    }
	    else if (command.startsWith("Stop: ")) {
		sender.exit = true;
		Debug.noteln("Stopped messages for test", menuText);
	    }
	    else
		Debug.noteln("Strange test button action", command);
	}

	void finished() {
	    button.setText(menuText);
	}

    }

    /**
     * Stoppable thread for sending a series of messages.
     * Stop by doing <code>thread.exit = true</code>.
     */
    class SendingThread extends CatchingThread {

	TestSequenceListener listener;
	AbstractButton button;
	List testItems;

	int minDelay = 0 * 1000;	// milliseconds // /\/

	volatile boolean exit = false;  // gets the thread to exit.

	public SendingThread(TestSequenceListener listener,
			     List testItems) {
	    this.listener = listener;
	    this.testItems = testItems;
	}

	public void innerRun() {

	messageLoop:
	    for (Iterator i = testItems.iterator();
		 !exit && i.hasNext();) {

		TestItem item = (TestItem)i.next();
		final String toName = item.getToName();
		final Sendable contents = item.getContents();
		int delay = item.getDelayBefore();

		delay = Math.max(minDelay, delay);

		if (delay > 0) {
		    Debug.noteln("Sleeping", delay);
		    try { Thread.sleep(delay); }
		    catch (InterruptedException e) {}
		}
		if (!exit) {
		    Util.swingAndWait(new Runnable() {
			public void run() {
			    send(toName, contents);
			}
		    });
		}
	    }
	    SwingUtilities.invokeLater(new Runnable() {
		public void run() {
		    listener.finished();
		}
	    });

	}

	void send(final String toName, final Sendable contents) {
	    try {
		sendTest(toName, contents);
	    }
	    catch (Throwable t) {
		Debug.displayException(t);
	    }
	}

    }

}
