/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Mon Jun  9 01:10:45 2008 by Jeff Dalton
 * Copyright: (c) 2007, 2008, AIAI, University of Edinburgh
 */

package ix.ip2;

import java.util.*;

import ix.icore.*;
import ix.icore.process.*;
import ix.icore.domain.Constraint;
import ix.icore.domain.End;

import ix.iplan.DomainAnalyser;

import ix.util.*;
import ix.util.lisp.*;
import ix.util.match.*;
import ix.util.context.*;

/**
 * Manager for "use" constraints.
 */
public class UseCM implements ConstraintManager {

    private static final Symbol	USE = Symbol.intern("use");

    // /\/: Might be better to use node-ends in the tables.

    // Maps a resource pattern to the PNode, if any, that
    // is currently using the resource.
    ContextHashMap useTable = new ContextHashMap();

    // Maps a resource pattern to the PNode, if any, that
    // most recently freed the resource.
    ContextHashMap freeTable = new ContextHashMap();

    Ip2ModelManager mm;

    public UseCM(Ip2ModelManager mm) {
	this.mm = mm;
    }

    public void registerWith(ConstraintManagerRegistry r) {
	r.addConstraintManager
	    (this, Symbol.intern("resource"),
	           new Symbol[]{USE});
    }

    public void reset() {
	useTable.clearCompletely();
	freeTable.clearCompletely();
    }

    public void clear() {
	useTable.clear();
	freeTable.clear();
    }

    /** @throws UnsupportedOperationException if called. */
    public void addConstraint(Constraint c) {
      throw new UnsupportedOperationException
	("Method addConstraint(Constraint) is not supported by this CM.");
    }

    public void addConstraint(PNode node, Constraint c) {
	Debug.noteln("Use CM adding", c);
	Debug.noteln("At", node);
	// evalConstraint(node, c);
    }

    public void evalAtBegin(PNodeEnd ne, Constraint c) {
	Debug.noteln("Use CM eval " + c + " at " + ne);
	Debug.expectSame(End.BEGIN, ne.getEnd());
	c = (Constraint)Variable.removeAllVars(c);
	LList pattern = c.getPattern();
	PNode user = (PNode)useTable.get(pattern);
	if (user != null)
	    throw new FailureException
		(pattern + " already in use at " + user);
	// Cool.  So we can use it.
	useTable.put(pattern, ne.getNode());
	// If someone was using it before, we must be linked after them.
	PNode releaser = (PNode)freeTable.get(pattern);
	if (releaser != null)
	    mm.linkBefore(releaser.getEnd(), ne);
    }

    public void evalAtEnd(PNodeEnd ne, Constraint c) {
	Debug.noteln("Use CM eval " + c + " at " + ne);
	Debug.expectSame(End.END, ne.getEnd());
	c = (Constraint)Variable.removeAllVars(c);
	LList pattern = c.getPattern();
	// We should be the one using the resource.
	Debug.expectSame(ne.getNode(), useTable.get(pattern));
	// Now we can free it.
	useTable.put(pattern, null);
	freeTable.put(pattern, ne.getNode());
    }

    public boolean mightInteract(PNodeEnd ne1, Constraint c1,
				 PNodeEnd ne2, Constraint c2) {
	if (ne1.getEnd() == End.END && ne2.getEnd() == End.END)
	    return false;
	else
	    return DomainAnalyser.match(c1.getPattern(), c2.getPattern());
    }

}
