/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Sat Apr  2 04:11:03 2005 by Jeff Dalton
 * Copyright: (c) 2004, AIAI, University of Edinburgh
 */

package ix.ip2;

import java.util.*;

import ix.icore.domain.Constraint;
import ix.util.*;
import ix.util.lisp.Lisp;
import ix.util.match.MatchEnv;

/**
 * Explains that a {@link HandlerAction} is not ready to be used
 * because conditions cannot be satisfied.
 *
 * @see HandlerAction#isReady()
 * @see HandlerAction#getUnreadyReason()
 */
public class UnsatisfiedConditionsUnreadyReason extends ActionUnreadyReason {

    protected List satisfiedConditions;
    protected List unsatisfiedConditions;
    protected MatchEnv condEnv;

    protected String[] explanation = null;

    /**
     * Creates an unready reason for the specified action and conditions.
     */
    public UnsatisfiedConditionsUnreadyReason
	       (HandlerAction action,
		List allConditions,
		List satisfiedConditions, // N.B. *sat*isfied conditions
		MatchEnv condEnv) {
	super(action);
	this.satisfiedConditions = satisfiedConditions;
	this.unsatisfiedConditions =
	    (List)Collect.difference(allConditions, satisfiedConditions);
	this.condEnv = condEnv;
    }

    public List getSatisfiedConditions() {
	return satisfiedConditions;
    }

    public List getUnsatisfiedConditions() {
	return unsatisfiedConditions;
    }

    public MatchEnv getConditionEnv() {
	return condEnv;
    }

    public String[] getExplanation() {
	if (explanation == null)
	    explanation = makeExplanation();
	return explanation;
    }

    protected String[] makeExplanation() {
	List lines = new LinkedList();
	if (unsatisfiedConditions.isEmpty()) {
	    lines.add("The conditions could all be satisfied individually,");
	    lines.add("but not together.");
	}
	lines.addAll(condDescrs("Can be satisfied individually:",
				satisfiedConditions));
	lines.addAll(condDescrs("Cannot be satisfied individually:",
				unsatisfiedConditions));
	lines.add(Strings.foldLongLine
		   ("Environment: " + envDescription(condEnv), 75, "  "));
	return Strings.toArray(lines);
    }

    protected List condDescrs(String label, List conds) {
	if (conds.isEmpty())
	    return Lisp.list(label + " none");
	List lines = new LinkedList();
	lines.add(label);
	for (Iterator i = conds.iterator(); i.hasNext();) {
	    Constraint c = (Constraint)i.next();
	    // PatternAssignment pv = (PatternAssignment)i.next();
	    // lines.add("   " + pv);
	    lines.add("   " + c);
	}
	return lines;
    }

    protected String envDescription(MatchEnv env) {
	if (env.isEmpty())
	    return "empty";
	StringBuffer descr = new StringBuffer();
	for (Iterator i = env.entrySet().iterator(); i.hasNext();) {
	    Map.Entry e = (Map.Entry)i.next();
	    descr.append(e.getKey());
	    descr.append("=");
	    descr.append(e.getValue());
	    if (i.hasNext())
		descr.append(", ");
	}
	return descr.toString();
    }

}
