/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Wed Sep 27 16:32:22 2006 by Jeff Dalton
 * Copyright: (c) 2006, AIAI, University of Edinburgh
 */

package ix.ip2;

import java.util.*;

import ix.icore.*;
import ix.icore.domain.*;
import ix.icore.process.event.ProcessStatusEvent;

import ix.util.*;

public class SyncStateServer extends SyncState implements IXAgentExtension {

    // We use an sorted set for the clients because the behaviour
    // is more easily reproduced.
    protected Set clients = new TreeSet();

    public SyncStateServer(Ip2 ip2) {
        super(ip2);
    }

    public void installExtension()  {
        ip2.getController()
            .addActivityHandler(new SyncStateServerHandler(ip2));
        ip2.addStartupHook(new StartupHook());
        ip2.addResetHook(new ResetHook());
    }

    protected class StartupHook implements Runnable {
        public void run() {
            // Shouldn't install this earlier because IPC isn't set up
            // (though it probably does't matter since we'll have
            // no clients).
            ip2.getModelManager()
                .addProcessStatusListener(new ProcessListener());
        }
    }

    protected class ResetHook implements Runnable {
        public void run() {
        }
    }

    protected void sendToClients(ClientOperation op) {
        for (Iterator i = clients.iterator(); i.hasNext();) {
            String clientName = (String)i.next();
            try {
                send(clientName, op);
            }
            catch (Exception e) {
                Debug.displayException
                    ("Will drop " + clientName + " as a sync-state client",
                     e);
                i.remove();
                Debug.noteln("Sync-state clients", clients);
            }
        }
    }

    protected class ProcessListener extends NullStateViewer {

        public void stateChange(ProcessStatusEvent event, Map delta) {
            if (!ip2.isReloadingViewers())
                sendToClients(new ServerStateChange(delta));
        }

        public void stateDeletion(ProcessStatusEvent event, Map delta) {
            if (!ip2.isReloadingViewers())
                sendToClients(new ServerStateDeletion(delta));
        }

    }

    /*
     * Server operations.
     */

    public class SyncStateServerHandler extends SyncStateHandler {
        public SyncStateServerHandler(Ip2 ip2) {
            super(ip2, "Synchronized-state server");
        }
        public void handle(AgendaItem item) {
            handleOperation((ServerOperation)getOperation(item));
        }
    }

    protected void handleOperation(ServerOperation op) {
        Debug.noteln("Sync-state server handing " + op +
                     " for " + op.getClientName());
        op.run(this);
    }

    /* *** Register client *** */

    protected void handle(RegisterClient reg) {
        // Get the new client's name, but don't register it yet.
        String clientName = reg.getClientName();
        // Check whether it's really new.  If not, assume the
        // agent exited and is now re-registering.
        if (clients.contains(clientName)) {
            Debug.noteln("Sync-state server will re-register", clientName);
            clients.remove(clientName);
        }
        // We also get the new client's current state.
        // /\/: Is the client's state new information or old?
        // We assume it's logically newer than what we already
        // have, but that might not really  be so.
        ListOfPatternAssignment delta = reg.getDelta();
        // Update our state model.  This will Send the new information
        // to all existing (old) clients - but not to the client that
        // is registering.
        ip2.getIp2ModelManager()
            // /\/: Assumes only one option.
            .handleEffects(delta);
        registerClient(clientName);
    }

    protected void registerClient(String clientName) {
        Debug.noteln("Sync-state server registering", clientName);
        Debug.expect(!clients.contains(clientName), "Registering twice?");

        // Register the new client;
        clients.add(clientName);

        // Now send the new client the entire state.
        Map state = ip2.getIp2ModelManager().getWorldStateMap();
        // /\/: Must copy state because we get a context-layered one.
        state = new StableHashMap(state);
        send(clientName, new ServerFullState(state));
    }

    protected void registerIfNew(String clientName) {
        // /\/: If the server exited and restarted, clients will (probably?)
        // not re-register.  So if we get a message from a client that
        // hasn't registered, we probably ought to add it to the set
        // of clients and treat the message as a registration.
        if (!clients.contains(clientName)) {
            Debug.noteln("Sync-state server Treating as old client",
                         clientName);
            registerClient(clientName);
        }
    }
    

    /* *** State change from client *** */

    public void handle(ClientStateChange change) {
        ListOfPatternAssignment delta = change.getDelta();
        // /\/: Assumes only one option.
        ip2.getIp2ModelManager()
            .handleEffects(delta);
        registerIfNew(change.getClientName());
    }

    /* *** State deletion from client *** */

    public void handle(ClientStateDeletion deletion) {
        ListOfPatternAssignment delta = deletion.getDelta();
        deletionUtility(delta);
        registerIfNew(deletion.getClientName());
    }

}
