/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Wed Sep 27 16:43:04 2006 by Jeff Dalton
 * Copyright: (c) 2006, AIAI, University of Edinburgh
 */

package ix.ip2;

import java.util.*;

import ix.icore.*;
import ix.icore.domain.*;
import ix.icore.process.event.ProcessStatusEvent;

import ix.iplan.IPlanOptionManager;

import ix.util.*;

public class SyncStateClient extends SyncState implements IXAgentExtension {

    protected String serverName;
    protected String clientName;

    StateViewer processListener = new ProcessListener();

    protected boolean doingUpdateFromServer = false;

    public SyncStateClient(Ip2 ip2) {
        super(ip2);
    }

    public void installExtension() {
        serverName = Parameters.requireParameter("sync-state-server");
        clientName = ip2.getAgentSymbolName();
        ip2.getController()
            .addActivityHandler(new SyncStateClientHandler(ip2));
        ip2.addStartupHook(new StartupHook());
        ip2.addResetHook(new ResetHook());

    }

    protected class StartupHook implements Runnable {
        public void run() {
            // Register with the sync-state server.
            Map state = ip2.getIp2ModelManager().getWorldStateMap();
            // /\/: Must copy state because we get a context-layered one.
            state = new StableHashMap(state);
            while (true) {
                try {
                    send(serverName, new RegisterClient(clientName, state));
                    break;
                }
                catch (Exception e) {
                    Debug.noteException(e);
                    if (!tryRegisteringAgain())
                        return;
                }
            }
            // Add a listener that will copy any state changes
            // to the server.  We can't do this when installing
            // the extension because IPC isn't set up and we
            /// might load a plan before it is.
            ip2.getModelManager().addProcessStatusListener(processListener);
        }
        boolean tryRegisteringAgain() {
            Object[] message = {
		"Could not register " + clientName
		    + " with sync-state server " + serverName,
		"Do you want to try again?"
	    };
            return Util.dialogConfirms(ip2.getFrame(), message);
        }
    }

    protected class ResetHook implements Runnable {
        public void run() {
            processListener.reset();
        }
    }

    protected class ProcessListener extends NullStateViewer {

        public void stateChange(ProcessStatusEvent event, Map delta) {
            if (!(ip2.isReloadingViewers()
                  || !optMan.canTakeInput()
                  || doingUpdateFromServer))
                send(serverName, new ClientStateChange(clientName, delta));
        }

        public void stateDeletion(ProcessStatusEvent event, Map delta) {
            Debug.expect(!ip2.isReloadingViewers());
            if (optMan.canTakeInput()
                  && !doingUpdateFromServer)
                send(serverName, new ClientStateDeletion(clientName, delta));
        }

    }

    /*
     * Client operations
     */

    public class SyncStateClientHandler extends SyncStateHandler {
        public SyncStateClientHandler(Ip2 ip2) {
            super(ip2, "Synchronize state in client");
        }
        public void handle(AgendaItem item) {
            handleOperation((ClientOperation)getOperation(item));
        }
    }

    protected void handleOperation(ClientOperation op) {
        Debug.noteln("Sync-state client handing", op);
        op.run(this);
    }

    /* *** State change from server *** */

    public void handle(ServerStateChange change) {
        if (optMan.canTakeInput())
            handleServerStateChange(change.getDelta());
        else
            optMan.getOptionForInput()
                .recordDelayedInput(new ChangeMessage(change.getDelta()));
    }

    protected void handleServerStateChange(ListOfPatternAssignment delta) {
        try {
            doingUpdateFromServer = true;
            // Assumes we're in the right option.
            mm.handleEffects(delta);
        }
        finally {
            doingUpdateFromServer = false;
        }
    }

    protected class ChangeMessage extends IPlanOptionManager.PseudoMessage {
        ChangeMessage(ListOfPatternAssignment delta) {
            super(delta);
        }
	public void receivedBy(IPlanOptionManager.Opt option,
                               Ip2 modelHolder) {
	    Debug.noteln("Delayed state change in", option);
            Debug.expectSame(ip2, modelHolder);
            handleServerStateChange((ListOfPatternAssignment)getContents());
	}

    }

    /* *** Full state from server *** */

    public void handle(ServerFullState change) {
        // /\/: For now, we do the same as with a ServerStateChange,
        // but we ought to notice that this is supposed to be our
        // entire state.
        Debug.noteln("Full state from sync-state server.");
        if (optMan.canTakeInput())
            handleServerStateChange(change.getDelta());
        else
            optMan.getOptionForInput()
                .recordDelayedInput(new ChangeMessage(change.getDelta()));
    }

    /* *** State deletion from server *** */

    public void handle(ServerStateDeletion deletion) {
        if (optMan.canTakeInput())
            handleServerStateDeletion(deletion.getDelta());
        else
            optMan.getOptionForInput()
                .recordDelayedInput(new DeleteMessage(deletion.getDelta()));
    }

    protected void handleServerStateDeletion(ListOfPatternAssignment delta) {
        try {
            doingUpdateFromServer = true;
            // Assumes we're in the right option.
            deletionUtility(delta);
        }
        finally {
            doingUpdateFromServer = false;
        }
    }

    protected class DeleteMessage extends IPlanOptionManager.PseudoMessage {
        DeleteMessage(ListOfPatternAssignment delta) {
            super(delta);
        }
	public void receivedBy(IPlanOptionManager.Opt option,
                               Ip2 modelHolder) {
	    Debug.noteln("Delayed state deletion in", option);
            Debug.expectSame(ip2, modelHolder);
            handleServerStateDeletion((ListOfPatternAssignment)getContents());
	}

    }

}
