/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Sat May 10 11:48:47 2008 by Jeff Dalton
 * Copyright: (c) 2006, 2008, AIAI, University of Edinburgh
 */

package ix.ip2;

import java.io.Serializable;
import java.util.*;

import ix.icore.*;
import ix.icore.domain.PatternAssignment;
import ix.icore.domain.ListOfPatternAssignment;

import ix.iplan.IPlanOptionManager;

import ix.util.*;
import ix.util.lisp.*;

/**
 * Framework and operations for synchronised-state servers and clients.
 */
public class SyncState {

    protected static final Symbol SYNCHRONIZE_STATE =
        Symbol.intern("synchronize_state");

    protected Ip2 ip2;
    protected Ip2ModelManager mm;
    protected IPlanOptionManager optMan;

    public SyncState(Ip2 ip2) {
        this.ip2 = ip2;
        this.mm = ip2.getIp2ModelManager();
        this.optMan = ip2.getOptionManager();
    }

    /*
     * Activities that contain operations
     */

    protected Activity makeSyncStateActivity(Operation op) {
        Activity act = new Activity(Lisp.list(SYNCHRONIZE_STATE, op));
        return act;
    }

    protected void send(String destination, Operation op) {
        IPC.sendObject(destination, makeSyncStateActivity(op));
    }

    /*
     * Activity handlers
     */

    public static abstract class SyncStateHandler extends ActivityHandler {
        protected Ip2 ip2;
        public SyncStateHandler (Ip2 ip2, String description) {
            super(description);
            this.ip2 = ip2;
        }
        public List getSyntaxList() {
            return (LList)Lisp.readFromString
                ("((synchronize-state ?sync-state-operation))");
        }
        public boolean appliesTo(AgendaItem item) {
            LList pattern = item.getPattern();
            return pattern.get(0) == SYNCHRONIZE_STATE
                && pattern.get(1) instanceof Operation;
        }
        protected Operation getOperation(AgendaItem item) {
            return (Operation)item.getPattern().get(1);
        }
        public boolean isAutomatic() {
            return true;
        }
        public void addHandlerActions(AgendaItem item) {
            item.addAction
                (new HandlerAction.AutomaticWhenBound(item, this));
        }
        public abstract void handle(AgendaItem item);
    }

    /*
     * Operations
     */

    // /\/: We have to add Serializable somewhere for the
    // communication strategies that use it.  Making
    // Operation extend Serializable is not ideal.

    // /\/: The zero-argument constructors are needed by the XML tools.

    // /\/: We want to use ListOfPatternAssignment in the objects we
    // send, rather than using Map, because we want to make sure the
    // patterns come through as LLists when using an "xml" comm strategy.
    // However, we still create the "Operation" objects using Maps
    // for historical reasons and because that's what we get
    // from the model-manager.

    // The "run" methods in operations always look like
    //    server.handle(this)
    // or
    //    client.handle(this)
    // Even though the text is always the same, "this" has a
    // different class in each case and so causes a different
    // method in the server or client to be selected.
    // The server and client typically have a different
    // "handle" method for each relevant operation class.

    public static interface Operation extends Serializable { }

    public static interface ServerOperation extends Operation {
        public String getClientName();
        public void run(SyncStateServer server);
    }

    public static interface ClientOperation extends Operation {
        public void run(SyncStateClient client);
    }

    /*
     * Server operations - processed in the server
     */

    public static abstract class AbstractServerOperation
                           implements ServerOperation {
        protected String clientName;
        public AbstractServerOperation(String clientName) {
            this.clientName = clientName;
        }
        public String getClientName() {
            return clientName;
        }
        public void setClientName(String name) {
            this.clientName = name;
        }
    }

    public static abstract class ClientStateDelta
                           extends AbstractServerOperation {
        protected ListOfPatternAssignment delta;
        public ClientStateDelta(String clientName,
				Map delta) {
            super(clientName);
	    if (delta != null)
		this.delta = PatternAssignment.mapToAssignments(delta);
        }
        public ListOfPatternAssignment getDelta() {
            return delta;
        }
        public void setDelta(ListOfPatternAssignment delta) {
            this.delta = delta;
        }
    }

    public static class RegisterClient extends ClientStateDelta {
        public RegisterClient() {
            this(null, null);
        }
        public RegisterClient(String clientName,
			      Map delta) {
            super(clientName, delta);
        }
        public void run(SyncStateServer server) {
            server.handle(this);
        }
    }

    public static class ClientStateChange extends ClientStateDelta {
        public ClientStateChange() {
            this(null, null);
        }
        public ClientStateChange(String clientName,
				 Map delta) {
            super(clientName, delta);
        }
        public void run(SyncStateServer server) {
            server.handle(this);
        }
    }

    public static class ClientStateDeletion extends ClientStateDelta {
        public ClientStateDeletion() {
            this(null, null);
        }
        public ClientStateDeletion(String clientName,
				   Map delta) {
            super(clientName, delta);
        }
        public void run(SyncStateServer server) {
            server.handle(this);
        }
    }

    /*
     * Client operations - processed in the client
     */

    public static abstract class AbstractClientOperation
                           implements ClientOperation {
        public AbstractClientOperation() {
        }
    }

    public static abstract class ServerStateDelta
                           extends AbstractClientOperation {
        protected ListOfPatternAssignment delta;
        public ServerStateDelta(Map delta) {
	    if (delta != null)
		this.delta = PatternAssignment.mapToAssignments(delta);
        }
        public ListOfPatternAssignment getDelta() {
            return delta;
        }
        public void setDelta(ListOfPatternAssignment delta) {
            this.delta = delta;
        }
    }

    public static class ServerStateChange extends ServerStateDelta {
        public ServerStateChange() {
            this(null);
        }
        public ServerStateChange(Map delta) {
            super(delta);
        }
        public void run(SyncStateClient client) {
            client.handle(this);
        }
    }

    public static class ServerFullState extends ServerStateDelta {
        public ServerFullState() {
            this(null);
        }
        public ServerFullState(Map delta) {
            super(delta);
        }
        public void run(SyncStateClient client) {
            client.handle(this);
        }
    }

    public static class ServerStateDeletion extends ServerStateDelta {
        public ServerStateDeletion() {
            this(null);
        }
        public ServerStateDeletion(Map delta) {
            super(delta);
        }
        public void run(SyncStateClient client) {
            client.handle(this);
        }
    }

    /*
     * Utilities
     */

    protected void deletionUtility(ListOfPatternAssignment delta) {
        Debug.expect(delta.size() == 1, "delete for != 1 constraint");
        // ListOfPatternAssignment assignments =
	//     PatternAssignment.mapToAssignments(delta);
	ListOfPatternAssignment assignments = delta;
        PatternAssignment pv = assignments.get(0);
        // See if it's currently in the state.
        // Sometimes, it will already have been deleted.
        // That's what's usually happened if this agent
        // initiated the deletion.  If it has a different
        // value, delete it anyway on the grounds that
        // deleting p=v is like asserting p=undef.
        Object v = mm.getWorldStateValue(pv.getPattern());
        if (v != null) {
            if (!pv.getValue().equals(v))
                pv.setValue(v);
            mm.deleteEffect(pv);
        }
    }

}
